% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mies_methods.R
\name{mies_evaluate_offspring}
\alias{mies_evaluate_offspring}
\title{Evaluate Proposed Configurations Generated in a MIES Iteration}
\usage{
mies_evaluate_offspring(
  inst,
  offspring,
  budget_id = NULL,
  fidelity = NULL,
  reevaluate_fidelity = NULL,
  fidelity_monotonic = TRUE
)
}
\arguments{
\item{inst}{(\code{OptimInstance})\cr
Optimization instance to evaluate.}

\item{offspring}{(\code{data.frame})\cr
Proposed configurations to be evaluated, must have columns named after the \code{inst}'s search space, minus \code{budget_id} if not \code{NULL}.}

\item{budget_id}{(\code{character(1)} | \code{NULL})\cr
Budget component when doing multi-fidelity optimization. This component of the search space is added
to \code{individuals} according to \code{fidelity}. Should be \code{NULL} when no multi-fidelity optimization is performed (default).}

\item{fidelity}{(\code{atomic(1)} | \code{NULL})\cr
Atomic scalar indicating the value to be assigned to the \code{budget_id} component of offspring.
This value must be \code{NULL} if no multi-fidelity optimization is performed (the default).}

\item{reevaluate_fidelity}{(\code{atomic(1)})\cr
Fidelity with which to evaluate alive individuals from previous generations that have a budget value below (if \code{fidelity_monotonic} is \code{TRUE}) or
different from the current \code{fidelity} value. Default \code{NULL}: Do not re-evaluate. Must be \code{NULL} when \code{budget_id} and \code{fidelity} are \code{NULL}.
See also \code{\link{mies_step_fidelity}}.}

\item{fidelity_monotonic}{(\code{logical(1)})\cr
When \code{reevaluate_fidelity} is non-\code{NULL}, then this indicates whether individuals should only ever be re-evaluated when fidelity would be increased.
Default \code{TRUE}. Ignored when \code{reevaluate_fidelity} is \code{NULL}}
}
\value{
\link{invisible} \code{\link[data.table:data.table]{data.table}}: the performance values returned when evaluating the \code{offspring} values
through \code{eval_batch}.
}
\description{
Calls \verb{$eval_batch} of a given \code{OptimInstance} on a set
of configurations as part of a MIES operation. The \code{dob} extra-info in the archive
is also set properly to indicate a progressed generation.

This function can be used directly, but it is easier to use it within the \code{\link{OptimizerMies}}
class if standard GA operation is desired.

Multifidelity evaluation is supported as described in \code{vignette("mies-multifid")}. For this,
an extra component named after \code{budget_id} is appended to each individual, chosen from
the \code{fidelity} argument and depending on the value of \code{survivor_budget}. \code{budget_id} should
have the same values as given to the other \verb{mies_*} functions.
}
\examples{
library("bbotk")
lgr::threshold("warn")

# Define the objective to optimize
objective <- ObjectiveRFun$new(
  fun = function(xs) {
    z <- exp(-xs$x^2 - xs$y^2) + 2 * exp(-(2 - xs$x)^2 - (2 - xs$y)^2)
    list(Obj = z)
  },
  domain = ps(x = p_dbl(-2, 4), y = p_dbl(-2, 4)),
  codomain = ps(Obj = p_dbl(tags = "maximize"))
)

# Get a new OptimInstance
oi <- OptimInstanceSingleCrit$new(objective,
  terminator = trm("evals", n_evals = 100)
)

mies_init_population(inst = oi, mu = 3)
# Initial state:
oi$archive

# 'offspring' is just a data.frame of values to evaluate.
# In general it should be created using 'mies_generate_offspring()'.
offspring = data.frame(x = 1:2, y = 2:1)

mies_evaluate_offspring(oi, offspring = offspring)

# This evaluated the given points and assigned them 'dob' 2.
oi$archive

# Note that at this point one would ordinarily call a 'mies_survival_*()'
# function.

###
# Advanced demo, making use of additional components and doing multi-fidelity
##

# declare 'y' the budget parameter. It will not be in the 'offspring'
# table any more.
budget_id = "y"
# but: offspring may contain any other value that is appended to 'oi'. These
# are ignored by the objective.
offspring = data.frame(x = 0:1, z = 3)

mies_evaluate_offspring(oi, offspring = offspring, budget_id = budget_id,
  fidelity = 1)

# This now has the additional column 'z'. Values of y for the new evaluations
# are 1.
oi$archive

offspring = data.frame(x = 2, z = 3)
# Increasing the fidelity will not cause re-evaluation of existing individuals
# when `reevaluate_fidelity` is not given.
mies_evaluate_offspring(oi, offspring = offspring, budget_id = budget_id,
  fidelity = 2)
oi$archive

offspring = data.frame(x = 3, z = 3)
# Depending on the effect of fidelity, this may however have a biasing effect,
# so it may be desirable to re-evaluate surviving individuals from previous
# generations. The 'reevaluate_fidelity' may even be different from 'fidelity'
mies_evaluate_offspring(oi, offspring = offspring, budget_id = budget_id,
  fidelity = 3, reevaluate_fidelity = 2)

# In this example, only individuals with 'y = 1' were re-evaluated, since
# 'fidelity_monotonic' is TRUE.
oi$archive

}
\seealso{
Other mies building blocks: 
\code{\link{mies_generate_offspring}()},
\code{\link{mies_get_fitnesses}()},
\code{\link{mies_init_population}()},
\code{\link{mies_select_from_archive}()},
\code{\link{mies_step_fidelity}()},
\code{\link{mies_survival_comma}()},
\code{\link{mies_survival_plus}()}
}
\concept{mies building blocks}
