% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mice.impute.pmm.R
\name{mice.impute.pmm}
\alias{mice.impute.pmm}
\alias{pmm}
\title{Imputation by predictive mean matching}
\usage{
mice.impute.pmm(
  y,
  ry,
  x,
  wy = NULL,
  donors = 5L,
  matchtype = 1L,
  ridge = 1e-05,
  ...
)
}
\arguments{
\item{y}{Vector to be imputed}

\item{ry}{Logical vector of length \code{length(y)} indicating the 
the subset \code{y[ry]} of elements in \code{y} to which the imputation 
model is fitted. The \code{ry} generally distinguishes the observed 
(\code{TRUE}) and missing values (\code{FALSE}) in \code{y}.}

\item{x}{Numeric design matrix with \code{length(y)} rows with predictors for 
\code{y}. Matrix \code{x} may have no missing values.}

\item{wy}{Logical vector of length \code{length(y)}. A \code{TRUE} value 
indicates locations in \code{y} for which imputations are created.}

\item{donors}{The size of the donor pool among which a draw is made. 
The default is \code{donors = 5L}. Setting \code{donors = 1L} always selects 
the closest match, but is not recommended. Values between 3L and 10L 
provide the best results in most cases (Morris et al, 2015).}

\item{matchtype}{Type of matching distance. The default choice 
(\code{matchtype = 1L}) calculates the distance between 
the \emph{predicted} value of \code{yobs} and 
the \emph{drawn} values of \code{ymis} (called type-1 matching). 
Other choices are \code{matchtype = 0L} 
(distance between predicted values) and \code{matchtype = 2L} 
(distance between drawn values).}

\item{ridge}{The ridge penalty used in \code{.norm.draw()} to prevent 
problems with multicollinearity. The default is \code{ridge = 1e-05}, 
which means that 0.01 percent of the diagonal is added to the cross-product. 
Larger ridges may result in more biased estimates. For highly noisy data 
(e.g. many junk variables), set \code{ridge = 1e-06} or even lower to 
reduce bias. For highly collinear data, set \code{ridge = 1e-04} or higher.}

\item{\dots}{Other named arguments.}
}
\value{
Vector with imputed data, same type as \code{y}, and of length 
\code{sum(wy)}
}
\description{
Calculates imputations for univariate missing data by predictive mean matching.
}
\details{
Imputation of \code{y} by predictive mean matching, based on 
van Buuren (2012, p. 73). The procedure is as follows:

\enumerate{
\item{Calculate the cross-product matrix \eqn{S=X_{obs}'X_{obs}}.}
\item{Calculate \eqn{V = (S+{diag}(S)\kappa)^{-1}}, with some small ridge 
parameter \eqn{\kappa}.}
\item{Calculate regression weights \eqn{\hat\beta = VX_{obs}'y_{obs}.}}
\item{Draw \eqn{q} independent \eqn{N(0,1)} variates in vector \eqn{\dot z_1}.}
\item{Calculate \eqn{V^{1/2}} by Cholesky decomposition.}
\item{Calculate \eqn{\dot\beta = \hat\beta + \dot\sigma\dot z_1 V^{1/2}}.}
\item{Calculate \eqn{\dot\eta(i,j)=|X_{{obs},[i]|}\hat\beta-X_{{mis},[j]}\dot\beta} 
with \eqn{i=1,\dots,n_1} and \eqn{j=1,\dots,n_0}.}
\item{Construct \eqn{n_0} sets \eqn{Z_j}, each containing \eqn{d} candidate donors, from Y_{obs} such that \eqn{\sum_d\dot\eta(i,j)} is minimum for all \eqn{j=1,\dots,n_0}. Break ties randomly.}
\item{Draw one donor \eqn{i_j} from \eqn{Z_j} randomly for \eqn{j=1,\dots,n_0}.}
\item{Calculate imputations \eqn{\dot y_j = y_{i_j}} for \eqn{j=1,\dots,n_0}.}
}

The name \emph{predictive mean matching} was proposed by Little (1988).
}
\examples{
# We normally call mice.impute.pmm() from within mice()
# But we may call it directly as follows (not recommended)

set.seed(53177)
xname <- c('age', 'hgt', 'wgt')
r <- stats::complete.cases(boys[, xname])
x <- boys[r, xname]
y <- boys[r, 'tv']
ry <- !is.na(y)
table(ry)

# percentage of missing data in tv
sum(!ry) / length(ry)

# Impute missing tv data
yimp <- mice.impute.pmm(y, ry, x)
length(yimp)
hist(yimp, xlab = 'Imputed missing tv')

# Impute all tv data
yimp <- mice.impute.pmm(y, ry, x, wy = rep(TRUE, length(y)))
length(yimp)
hist(yimp, xlab = 'Imputed missing and observed tv')
plot(jitter(y), jitter(yimp), 
    main = 'Predictive mean matching on age, height and weight', 
    xlab = 'Observed tv (n = 224)',
    ylab = 'Imputed tv (n = 224)')
abline(0, 1)
cor(y, yimp, use = 'pair')
}
\references{
Little, R.J.A. (1988), Missing data adjustments in large surveys
(with discussion), Journal of Business Economics and Statistics, 6, 287--301.

Morris TP, White IR, Royston P (2015). Tuning multiple imputation by predictive 
mean matching and local residual draws. BMC Med Res Methodol. ;14:75.

Van Buuren, S. (2018). 
\href{https://stefvanbuuren.name/fimd/sec-pmm.html}{\emph{Flexible Imputation of Missing Data. Second Edition.}}
Chapman & Hall/CRC. Boca Raton, FL.

Van Buuren, S., Groothuis-Oudshoorn, K. (2011). \code{mice}: Multivariate
Imputation by Chained Equations in \code{R}. \emph{Journal of Statistical
Software}, \bold{45}(3), 1-67. \url{https://www.jstatsoft.org/v45/i03/}
}
\seealso{
Other univariate imputation functions: 
\code{\link{mice.impute.cart}()},
\code{\link{mice.impute.lda}()},
\code{\link{mice.impute.logreg.boot}()},
\code{\link{mice.impute.logreg}()},
\code{\link{mice.impute.mean}()},
\code{\link{mice.impute.midastouch}()},
\code{\link{mice.impute.norm.boot}()},
\code{\link{mice.impute.norm.nob}()},
\code{\link{mice.impute.norm.predict}()},
\code{\link{mice.impute.norm}()},
\code{\link{mice.impute.polr}()},
\code{\link{mice.impute.polyreg}()},
\code{\link{mice.impute.quadratic}()},
\code{\link{mice.impute.rf}()},
\code{\link{mice.impute.ri}()}
}
\author{
Stef van Buuren, Karin Groothuis-Oudshoorn
}
\concept{univariate imputation functions}
\keyword{datagen}
