% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pca.R
\name{pca}
\alias{pca}
\title{Principal Component Analysis}
\usage{
pca(x, ncomp = 15, center = T, scale = F, cv = NULL, exclrows = NULL,
  exclcols = NULL, x.test = NULL, alpha = 0.05, method = "svd",
  info = "")
}
\arguments{
\item{x}{a numerical matrix with calibration data.}

\item{ncomp}{maximum number of components to calculate.}

\item{center}{logical, do mean centering of data or not.}

\item{scale}{logical, do sdandardization of data or not.}

\item{cv}{number of segments for random cross-validation (1 for full cross-validation).}

\item{exclrows}{rows to be excluded from calculations (numbers, names or vector with logical values)}

\item{exclcols}{columns to be excluded from calculations (numbers, names or vector with logical values)}

\item{x.test}{a numerical matrix with test data.}

\item{alpha}{significance level for calculating limit for Q residuals.}

\item{method}{method to compute principal components ('svd', 'nipals').}

\item{info}{a short text line with model description.}
}
\value{
Returns an object of \code{pca} class with following fields:
\item{ncomp }{number of components included to the model.} 
\item{ncomp.selected }{selected (optimal) number of components.} 
\item{loadings }{matrix with loading values (nvar x ncomp).} 
\item{eigenvals }{vector with eigenvalues for all existent components.} 
\item{expvar }{vector with explained variance for each component (in percent).} 
\item{cumexpvar }{vector with cumulative explained variance for each component (in percent).} 
\item{T2lim }{statistical limit for T2 distance.} 
\item{Qlim }{statistical limit for Q residuals.} 
\item{info }{information about the model, provided by user when build the model.} 
\item{calres }{an object of class \code{\link{pcares}} with PCA results for a calibration data.} 
\item{testres }{an object of class \code{\link{pcares}} with PCA results for a test data, if it 
was provided.} 
\item{cvres }{an object of class \code{\link{pcares}} with PCA results for cross-validation, 
if this option was chosen.}
}
\description{
\code{pca} is used to build and explore a principal component analysis (PCA) model.
}
\details{
By default \code{pca} uses number of components (\code{ncomp}) as a minimum of number of 
objects - 1, number of variables and default or provided value. Besides that, there is also 
a parameter for selecting an optimal number of components (\code{ncomp.selected}). The optimal 
number of components is used to build a residuals plot (with Q residuals vs. Hotelling T2 
values), calculate confidence limits for Q residuals, as well as for SIMCA classification. 
  
You can provde number, names or logical values to exclode rows or columns from calibration and
validation of PCA model. In this case the outcome, e.g. scores and loadings will correspond to 
the original size of the data, but:

\enumerate{
  \item Loadings (and all performance statistics) will be computed without excluded objects and variables
  \item Matrix with loadings will have zero values for the excluded variables and the corresponding columns will be hidden.
  \item Matrix with scores will have score values calculated for the hidden objects but the rows will be hidden.
}

You can see scores and loadings for hidden rows and columns by using parameter 'show.excluded = T'
in plots. If you see other packages to make plots (e.g. ggplot2) you will not be able to distinguish
between hidden and normal objects.
}
\examples{
library(mdatools)
### Examples for PCA class

## 1. Make PCA model for People data with autoscaling
## and full cross-validation

data(people)
model = pca(people, scale = TRUE, cv = 1, info = 'Simple PCA model')
model = selectCompNum(model, 4)
summary(model)
plot(model, show.labels = TRUE)

## 3. Show scores and loadings plots for the model
par(mfrow = c(2, 2))
plotScores(model, comp = c(1, 3), show.labels = TRUE)
plotScores(model, comp = 2, type = 'h', show.labels = TRUE)
plotLoadings(model, comp = c(1, 3), show.labels = TRUE)
plotLoadings(model, comp = c(1, 2), type = 'h', show.labels = TRUE)
par(mfrow = c(1, 1))

## 4. Show residuals and variance plots for the model
par(mfrow = c(2, 2))
plotVariance(model, type = 'h')
plotCumVariance(model, show.labels = TRUE, legend.position = 'bottomright')
plotResiduals(model, show.labels = TRUE)
plotResiduals(model, ncomp = 2, show.labels = TRUE)
par(mfrow = c(1, 1))

}
\author{
Sergey Kucheryavskiy (svkucheryavski@gmail.com)
}
\seealso{
Methods for \code{pca} objects:
\tabular{ll}{
   \code{plot.pca} \tab makes an overview of PCA model with four plots.\cr
   \code{summary.pca} \tab shows some statistics for the model.\cr
   \code{\link{selectCompNum.pca}} \tab set number of optimal components in the model\cr
   \code{\link{predict.pca}} \tab applies PCA model to a new data.\cr
   \code{\link{plotScores.pca}} \tab shows scores plot.\cr
   \code{\link{plotLoadings.pca}} \tab shows loadings plot.\cr
   \code{\link{plotVariance.pca}} \tab shows explained variance plot.\cr
   \code{\link{plotCumVariance.pca}} \tab shows cumulative explained variance plot.\cr
   \code{\link{plotResiduals.pca}} \tab shows Q vs. T2 residuals plot.\cr
}
 Most of the methods for plotting data are also available for PCA results (\code{\link{pcares}})
 objects. Also check \code{\link{pca.mvreplace}}, which replaces missing values in a data matrix 
 with approximated using iterative PCA decomposition.
}

