% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stabit2.R
\name{stabit2}
\alias{stabit2}
\title{Matching model and selection correction for college admissions}
\usage{
stabit2(
  OUT = NULL,
  SEL = NULL,
  colleges = NULL,
  students = NULL,
  outcome = NULL,
  selection,
  binary = FALSE,
  niter,
  gPrior = FALSE,
  censored = 1,
  thin = 1,
  nCores = max(1, detectCores() - 1),
  verbose = FALSE,
  ...
)
}
\arguments{
\item{OUT}{data frame with characteristics of all observed matches, including
market identifier \code{m.id}, college identifier \code{c.id} and student identifier \code{s.id}.}

\item{SEL}{optional: data frame with characteristics of all observed and unobserved matches, including 
market identifier \code{m.id}, college identifier \code{c.id} and student identifier \code{s.id}.}

\item{colleges}{character vector of variable names for college characteristics. These variables carry the same value for any college.}

\item{students}{character vector of variable names for student characteristics. These variables carry the same value for any student.}

\item{outcome}{formula for match outcomes.}

\item{selection}{formula for match valuations.}

\item{binary}{logical: if \code{TRUE} outcome variable is taken to be binary; if \code{FALSE} outcome variable is taken to be continuous.}

\item{niter}{number of iterations to use for the Gibbs sampler.}

\item{gPrior}{logical: if \code{TRUE} the g-prior (Zellner, 1986) is used for the variance-covariance matrix. (Not yet implemented)}

\item{censored}{draws of the \code{kappa} parameter that estimates the covariation between the error terms in selection and outcome equation are 0:not censored, 1:censored from below, 2:censored from above.}

\item{thin}{integer indicating the level of thinning in the MCMC draws. The default \code{thin=1} saves every draw, \code{thin=2} every second, etc.}

\item{nCores}{number of cores to be used in parallel Gibbs sampling.}

\item{verbose}{logical. When set to \code{TRUE}, writes information messages on the console (recommended). Defaults to \code{FALSE}, which suppresses such messages.}

\item{...}{.}
}
\value{
\code{stabit2} returns a list of the estimation results with the following elements.
\item{sigma}{numeric scalar: standard deviation fixed to 1.}
\item{eta}{numeric vector: residuals of the selection equation.}
\item{vcov}{List of variance covariance matrices for coefficients alpha and beta of selection and outcome equations.}
\item{coefficients}{numeric vector: coefficients of selection and outcome equations.}
\item{fitted.values}{numeric vector: fitted values for outcome data.}
\item{residuals}{numeric vector: residuals of the outcome equation.}
\item{df}{integer: degrees of freedom.}
\item{binary}{logical: if \code{TRUE} outcome variable was taken to be binary; if \code{FALSE} outcome variable was taken to be continuous.}
\item{formula}{estimated formula.}
\item{call}{function call.}
\item{method}{One of "Sorensen", "Klein" or "Klein-selection". Method "Sorensen" is used when a single selection equation is passed. It assumes an equal sharing rule for student and college utility. Method "Klein" is used when two selection equations (one for students, one for schools) and one outcome equations are passed. Method "Klein-selection" only models selection and therefore does not require an outcome equations.}
\item{draws}{List of Gibbs sampling draws for alpha and beta coefficients.}
\item{coefs}{Posterior means of the Gibbs sampling draws.}
\item{variables}{List of data used in the estimation.}
}
\description{
The function provides a Gibbs sampler for a structural matching model that 
estimates preferences and corrects for sample selection bias when the selection process 
is a two-sided matching game; i.e., a matching of students to colleges.

The structural model consists of a selection and an outcome equation. The \emph{Selection Equation} 
determines which matches are observed (\eqn{D=1}) and which are not (\eqn{D=0}).
\deqn{ \begin{array}{lcl}
       D &= & 1[V \in \Gamma] \\
       V &= & W\beta + \eta
       \end{array}
     }{ D = 1[V in \Gamma] with V = W\beta + \eta
     }
Here, \eqn{V} is a vector of latent valuations of \emph{all feasible} matches, ie observed and 
unobserved, and \eqn{1[.]} is the Iverson bracket. 
A match is observed if its match valuation is in the set of valuations \eqn{\Gamma}
that satisfy the equilibrium condition (see Sorensen, 2007). 
The match valuation \eqn{V} is a linear function of \eqn{W}, a matrix of characteristics for 
\emph{all feasible} matches, and \eqn{\eta}, a vector of random errors. \eqn{\beta} is a paramter 
vector to be estimated.

The \emph{Outcome Equation} determines the outcome for \emph{observed} matches. The dependent
variable can either be continuous or binary, dependent on the value of the \code{binary}
argument. In the binary case, the dependent variable \eqn{R} is determined by a threshold 
rule for the latent variable \eqn{Y}.
\deqn{ \begin{array}{lcl}
       R &= & 1[Y > c] \\
       Y &= & X\alpha + \epsilon
       \end{array}
     }{ R = 1[Y > c] with Y = X\alpha + \epsilon
     }
Here, \eqn{Y} is a linear function of \eqn{X}, a matrix of characteristics for \emph{observed} 
matches, and \eqn{\epsilon}, a vector of random errors. \eqn{\alpha} is a paramter vector to 
be estimated.

The structural model imposes a linear relationship between the error terms of both equations 
as \eqn{\epsilon = \kappa\eta + \nu}, where \eqn{\nu} is a vector of random errors and \eqn{\kappa}
is the covariance paramter to be estimated. If \eqn{\kappa} were zero, the marginal distributions
of \eqn{\epsilon} and \eqn{\eta} would be independent and the selection problem would vanish.
That is, the observed outcomes would be a random sample from the population of interest.
}
\examples{
\donttest{
## --- SIMULATED EXAMPLE ---

## 1. Simulate two-sided matching data for 20 markets (m=20) with 100 students
##    (nStudents=100) per market and 20 colleges with quotas of 5 students, each
##    (nSlots=rep(5,20)). True parameters in selection and outcome equations are 
##    all equal to 1.

xdata <- stabsim2(m=20, nStudents=100, nSlots=rep(5,20), verbose=FALSE,
  colleges = "c1", students = "s1",
  outcome = ~ c1:s1 + eta + nu,
  selection = ~ -1 + c1:s1 + eta
)
head(xdata$OUT)


## 2. Correction for sorting bias when match valuations V are observed

## 2-a. Bias from sorting
 lm1 <- lm(y ~ c1:s1, data=xdata$OUT)
 summary(lm1)

## 2-b. Cause of the bias
 with(xdata$OUT, cor(c1*s1, eta))

## 2-c. Correction for sorting bias
 lm2a <- lm(V ~ -1 + c1:s1, data=xdata$SEL); summary(lm2a)
 etahat <- lm2a$residuals[xdata$SEL$D==1]
 
 lm2b <- lm(y ~ c1:s1 + etahat, data=xdata$OUT)
 summary(lm2b)


## 3. Correction for sorting bias when match valuations V are unobserved

## 3-a. Run Gibbs sampler (when SEL is given)
 fit2 <- stabit2(OUT = xdata$OUT, 
           SEL = xdata$SEL,
           outcome = y ~ c1:s1, 
           selection = ~ -1 + c1:s1,
           niter=1000
 )

## 3-b. Alternatively: Run Gibbs sampler (when SEL is not given)
 fit2 <- stabit2(OUT = xdata$OUT, 
           colleges = "c1",
           students = "s1",
           outcome = y ~ c1:s1, 
           selection = ~ -1 + c1:s1,
           niter=1000
 )


## 4. Implemented methods

## 4-a. Get coefficients
 fit2
 
## 4-b. Coefficient table
 summary(fit2)
 
## 4-c. Get marginal effects
 summary(fit2, mfx=TRUE)
 
## 4-d. Also try the following functions
 #coef(fit2)
 #fitted(fit2)
 #residuals(fit2)
 #predict(fit2, newdata=NULL)

   
## 5. Plot MCMC draws for coefficients
 plot(fit2)
}
}
\references{
Sorensen, M. (2007). How Smart is Smart Money? A Two-Sided Matching Model of Venture Capital.
\emph{Journal of Finance}, 62 (6): 2725-2762.
}
\author{
Thilo Klein
}
\keyword{regression}
