### Functions to setup and run code using logrx

#' Creation of a log and axecution of a file
#'
#' `axecute()` creates a log, executes a file, and returns 0 if there are no
#' errors or 1 if there are any errors
#'
#' @param file String. Path to file to execute
#' @param log_name String. Name of log file
#' @param log_path String. Path to log file
#' @param remove_log_object Boolean. Should the log object be removed after
#' writing the log file? Defaults to TRUE
#' @param quit_on_error Boolean. Should the session quit with status 1 on error?
#' Defaults to TRUE
#' @param to_report String vector. Objects to optionally report, may include as
#' many as necessary:
#' * messages: any messages generated by program execution
#' * output: any output generated by program execution
#' * result: any result generated by program execution
#'
#' @importFrom purrr map_chr
#'
#' @return 0 if there are no errors or 1 if there are any errors
#' @export
#'
#' @examples
#' dir <- tempdir()
#' text <- 'print("Hello, logrxperson!")'
#' fileConn <- file(file.path(dir, "hello.R"))
#' writeLines(text, fileConn)
#' close(fileConn)
#'
#' axecute(file.path(dir, "hello.R"))
axecute <- function(file, log_name = NA,
                    log_path = NA,
                    remove_log_object = TRUE,
                    quit_on_error = TRUE,
                    to_report = c("messages", "output", "result")){

   # lower everything for consistency and check values
   to_report <- map_chr(to_report, tolower)
   match.arg(to_report, several.ok = TRUE)

   # initialize log
   log_config(file = file, log_name = log_name, log_path = log_path)

   # run the code
   run_safely_loudly(file)

   # check for errors prior to log_write() since this can remove the log
   any_errors <- get_log_element("errors")

   # write log
   log_write(file = file, remove_log_object = remove_log_object, to_report = to_report)

   # if error, quit with status = 1 if not interactive
   if(!interactive() & !is.null(any_errors) & quit_on_error) {
      quit(status = 1)
   }

}
