### Functions to setup and run code using logrx

#' Create a log and run a file
#'
#' @param file File path of file to run
#' @param log_name Name of log file
#' @param log_path Path to output log to
#' @param remove_log_object Should the log object be removed after writing, defaults to TRUE
#' @param quit_on_error Should the session quit with status of 1 with error, defaults to TRUE
#' @param to_report toggle for optional reporting objects.
#'   choices outlined below, may include as many as necessary
#'   messages: any messages generated by program execution
#'   output: any output generated by program execution
#'   result: any result generated by program execution
#'
#' @importFrom purrr map_chr
#'
#' @return 0 if there are no errors and 1 if any error
#' @export
#'
axecute <- function(file, log_name = NA,
                    log_path = NA,
                    remove_log_object = TRUE,
                    quit_on_error = TRUE,
                    to_report = c("messages", "output", "result")){

   # lower everything for consistency and check values
   to_report <- map_chr(to_report, tolower)
   match.arg(to_report, several.ok = TRUE)

   # initialize log
   log_config(file = file, log_name = log_name, log_path = log_path)

   # run the code
   run_safely_loudly(file)

   # check for errors prior to log_write() since this can remove the log
   any_errors <- get_log_element("errors")

   # write log
   log_write(file = file, remove_log_object = remove_log_object, to_report = to_report)

   # if error, quit with status = 1 if not interactive
   if(!interactive() & !is.null(any_errors) & quit_on_error) {
      quit(status = 1)
   }

}
