% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lassnags.r
\name{lassnags}
\alias{lassnags}
\title{Snag classification}
\usage{
lassnags(las, algorithm, attribute = "snagCls")
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{algorithm}{function. An algorithm for snag segmentation. \code{lidR} has \link{wing2015}.}

\item{attribute}{character. The returned LAS object automatically has a new
attribute (a new column). This parameter is the name of this new attribute.}
}
\value{
If the input is a \code{LAS} object, return a \code{LAS} object. If the input is a
\code{LAScatalog}, returns a \code{LAScatalog}.
}
\description{
Snag classification/segmentation using several possible algorithms (see details).
The function attributes a number identifying a snag class (\code{snagCls} attribute) to each point
of the point cloud. The classification/segmentation is done at the point cloud level and currently
only one algorithm implemented, which uses LiDAR intensity thresholds and specified neighborhoods
to differentiate bole and branch from foliage points (see details).
}
\section{Working with a \code{LAScatalog}}{

This section appears in each function that supports a LAScatalog as input.\cr

In \code{lidR} when the input of a function is a \link[lidR:LAScatalog-class]{LAScatalog} the
function uses the LAScatalog processing engine. The user can modify the engine options using
the \link[lidR:catalog_options_tools]{available options}. A careful reading of the
\link[lidR:LAScatalog-class]{engine documentation} is recommended before processing \code{LAScatalogs}. Each
\code{lidR} function should come with a section that documents the supported engine options.\cr

The \code{LAScatalog} engine supports \code{.lax} files that \emph{significantly} improve the computation
speed of spatial queries using a spatial index. Users should really take advantage a \code{.lax} files,
but this is not mandatory.\cr
}

\section{Supported processing options}{

Supported processing options for a \code{LAScatalog} (in bold). For more details see the
\link[lidR:LAScatalog-class]{LAScatalog engine documentation}:
\itemize{
\item \strong{chunk size}: How much data is loaded at once.
\item \strong{chunk buffer*}: Mandatory to get a continuous output without edge effects. The buffer
is always removed once processed and will never be returned either in R or in files.
\item \strong{chunk alignment}: Align the processed chunks.
\item \strong{cores}: How many cores are used. More cores means more data is loaded at once.
\item \strong{progress}: Displays a progression estimation.
\item \strong{output_files*}: Mandatory because the output is likely to be too big to be returned
 in R and needs to be written in las/laz files. Supported templates are \code{\{XLEFT\}}, \code{\{XRIGHT\}},
\code{\{YBOTTOM\}}, \code{\{YTOP\}}, \code{\{XCENTER\}}, \code{\{YCENTER\}} \code{\{ID\}} and, if
chunk size is equal to 0 (processing by file), \code{\{ORIGINALFILENAME\}}.
\item \strong{laz_compression}: write \code{las} or \code{laz} files
\item select: The function will write files equivalent to the original ones. Thus \code{select = "*"}
and cannot be changed.
\item \strong{filter}: Read only points of interest.
}
}

\examples{
\dontrun{
LASfile <- system.file("extdata", "MixedConifer.laz", package="lidR")
las <- readLAS(LASfile, select = "xyzi", filter="-keep_first") # Wing also included -keep_single

# For the Wing2015 method, supply a matrix of snag BranchBolePtRatio conditional
# assessment thresholds (see Wing et al. 2015, Table 2, pg. 172)
bbpr_thresholds <- matrix(c(0.80, 0.80, 0.70,
                          0.85, 0.85, 0.60,
                          0.80, 0.80, 0.60,
                          0.90, 0.90, 0.55),
                          nrow =3, ncol = 4)

# Run snag classification and assign classes to each point
las <- lassnags(las, wing2015(neigh_radii = c(1.5, 1, 2), BBPRthrsh_mat = bbpr_thresholds))

# Plot it all, tree and snag points...
plot(las, color="snagCls", colorPalette = rainbow(5))

# Filter and plot snag points only
snags <- lasfilter(las, snagCls > 0)
plot(snags, color="snagCls", colorPalette = rainbow(5)[-1])

# Wing et al's (2015) methods ended with performing tree segmentation on the
# classified and filtered point cloud using the watershed method
}
}
