% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dataset-utils.R
\name{timeseries_dataset_from_array}
\alias{timeseries_dataset_from_array}
\title{Creates a dataset of sliding windows over a timeseries provided as array.}
\usage{
timeseries_dataset_from_array(
  data,
  targets,
  sequence_length,
  sequence_stride = 1L,
  sampling_rate = 1L,
  batch_size = 128L,
  shuffle = FALSE,
  seed = NULL,
  start_index = NULL,
  end_index = NULL
)
}
\arguments{
\item{data}{array or eager tensor
containing consecutive data points (timesteps).
The first dimension is expected to be the time dimension.}

\item{targets}{Targets corresponding to timesteps in \code{data}.
\code{targets[i]} should be the target
corresponding to the window that starts at index \code{i}
(see example 2 below).
Pass \code{NULL} if you don't have target data (in this case the dataset
will only yield the input data).}

\item{sequence_length}{Length of the output sequences
(in number of timesteps).}

\item{sequence_stride}{Period between successive output sequences.
For stride \code{s}, output samples would
start at index \code{data[i]}, \code{data[i + s]}, \code{data[i + 2 * s]}, etc.}

\item{sampling_rate}{Period between successive individual timesteps
within sequences. For rate \code{r}, timesteps
\verb{data[i], data[i + r], ... data[i + sequence_length]}
are used for creating a sample sequence.}

\item{batch_size}{Number of timeseries samples in each batch
(except maybe the last one). If \code{NULL}, the data will not be batched
(the dataset will yield individual samples).}

\item{shuffle}{Whether to shuffle output samples,
or instead draw them in chronological order.}

\item{seed}{Optional int; random seed for shuffling.}

\item{start_index}{Optional int; data points earlier (exclusive)
than \code{start_index} will not be used
in the output sequences. This is useful to reserve part of the
data for test or validation.}

\item{end_index}{Optional int; data points later (exclusive) than \code{end_index}
will not be used in the output sequences.
This is useful to reserve part of the data for test or validation.}
}
\value{
A \code{tf$data$Dataset} instance. If \code{targets} was passed, the dataset yields
list \verb{(batch_of_sequences, batch_of_targets)}. If not, the dataset yields
only \code{batch_of_sequences}.

Example 1:

Consider indices \verb{[0, 1, ... 98]}.
With \verb{sequence_length=10,  sampling_rate=2, sequence_stride=3},
\code{shuffle=FALSE}, the dataset will yield batches of sequences
composed of the following indices:

\if{html}{\out{<div class="sourceCode">}}\preformatted{First sequence:  [0  2  4  6  8 10 12 14 16 18]
Second sequence: [3  5  7  9 11 13 15 17 19 21]
Third sequence:  [6  8 10 12 14 16 18 20 22 24]
...
Last sequence:   [78 80 82 84 86 88 90 92 94 96]
}\if{html}{\out{</div>}}

In this case the last 2 data points are discarded since no full sequence
can be generated to include them (the next sequence would have started
at index 81, and thus its last step would have gone over 98).

Example 2: Temporal regression.

Consider an array \code{data} of scalar values, of shape \verb{(steps,)}.
To generate a dataset that uses the past 10
timesteps to predict the next timestep, you would use:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data <- op_array(1:20)
input_data <- data[1:10]
targets <- data[11:20]
dataset <- timeseries_dataset_from_array(
  input_data, targets, sequence_length=10)
iter <- reticulate::as_iterator(dataset)
reticulate::iter_next(iter)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [[1]]
## tf.Tensor([[ 1  2  3  4  5  6  7  8  9 10]], shape=(1, 10), dtype=int32)
##
## [[2]]
## tf.Tensor([11], shape=(1), dtype=int32)

}\if{html}{\out{</div>}}

Example 3: Temporal regression for many-to-many architectures.

Consider two arrays of scalar values \code{X} and \code{Y},
both of shape \verb{(100,)}. The resulting dataset should consist samples with
20 timestamps each. The samples should not overlap.
To generate a dataset that uses the current timestamp
to predict the corresponding target timestep, you would use:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{X <- op_array(1:100)
Y <- X*2

sample_length <- 20
input_dataset <- timeseries_dataset_from_array(
    X, NULL, sequence_length=sample_length, sequence_stride=sample_length)
target_dataset <- timeseries_dataset_from_array(
    Y, NULL, sequence_length=sample_length, sequence_stride=sample_length)


inputs <- reticulate::as_iterator(input_dataset) \%>\% reticulate::iter_next()
targets <- reticulate::as_iterator(target_dataset) \%>\% reticulate::iter_next()
}\if{html}{\out{</div>}}
}
\description{
This function takes in a sequence of data-points gathered at
equal intervals, along with time series parameters such as
length of the sequences/windows, spacing between two sequence/windows, etc.,
to produce batches of timeseries inputs and targets.
}
\seealso{
\itemize{
\item \url{https://keras.io/api/data_loading/timeseries#timeseriesdatasetfromarray-function}
}

Other dataset utils: \cr
\code{\link{audio_dataset_from_directory}()} \cr
\code{\link{image_dataset_from_directory}()} \cr
\code{\link{split_dataset}()} \cr
\code{\link{text_dataset_from_directory}()} \cr

Other utils: \cr
\code{\link{audio_dataset_from_directory}()} \cr
\code{\link{clear_session}()} \cr
\code{\link{config_disable_interactive_logging}()} \cr
\code{\link{config_disable_traceback_filtering}()} \cr
\code{\link{config_enable_interactive_logging}()} \cr
\code{\link{config_enable_traceback_filtering}()} \cr
\code{\link{config_is_interactive_logging_enabled}()} \cr
\code{\link{config_is_traceback_filtering_enabled}()} \cr
\code{\link{get_file}()} \cr
\code{\link{get_source_inputs}()} \cr
\code{\link{image_array_save}()} \cr
\code{\link{image_dataset_from_directory}()} \cr
\code{\link{image_from_array}()} \cr
\code{\link{image_load}()} \cr
\code{\link{image_smart_resize}()} \cr
\code{\link{image_to_array}()} \cr
\code{\link{layer_feature_space}()} \cr
\code{\link{normalize}()} \cr
\code{\link{pad_sequences}()} \cr
\code{\link{set_random_seed}()} \cr
\code{\link{split_dataset}()} \cr
\code{\link{text_dataset_from_directory}()} \cr
\code{\link{to_categorical}()} \cr
\code{\link{zip_lists}()} \cr

Other preprocessing: \cr
\code{\link{image_dataset_from_directory}()} \cr
\code{\link{image_smart_resize}()} \cr
\code{\link{text_dataset_from_directory}()} \cr
}
\concept{dataset utils}
\concept{preprocessing}
\concept{timesery dataset utils}
\concept{utils}
