% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/callbacks.R
\name{callback_tensorboard}
\alias{callback_tensorboard}
\title{Enable visualizations for TensorBoard.}
\usage{
callback_tensorboard(
  log_dir = "logs",
  histogram_freq = 0L,
  write_graph = TRUE,
  write_images = FALSE,
  write_steps_per_second = FALSE,
  update_freq = "epoch",
  profile_batch = 0L,
  embeddings_freq = 0L,
  embeddings_metadata = NULL
)
}
\arguments{
\item{log_dir}{the path of the directory where to save the log files to be
parsed by TensorBoard. e.g.,
\code{log_dir = file.path(working_dir, 'logs')}.
This directory should not be reused by any other callbacks.}

\item{histogram_freq}{frequency (in epochs) at which to compute
weight histograms for the layers of the model. If set to 0,
histograms won't be computed. Validation data (or split) must be
specified for histogram visualizations.}

\item{write_graph}{(Not supported at this time)
Whether to visualize the graph in TensorBoard.
Note that the log file can become quite large
when \code{write_graph} is set to \code{TRUE}.}

\item{write_images}{whether to write model weights to visualize as image in
TensorBoard.}

\item{write_steps_per_second}{whether to log the training steps per second
into TensorBoard. This supports both epoch and batch frequency
logging.}

\item{update_freq}{\code{"batch"} or \code{"epoch"} or integer. When using \code{"epoch"},
writes the losses and metrics to TensorBoard after every epoch.
If using an integer, let's say \code{1000}, all metrics and losses
(including custom ones added by \code{Model.compile}) will be logged to
TensorBoard every 1000 batches. \code{"batch"} is a synonym for 1,
meaning that they will be written every batch.
Note however that writing too frequently to TensorBoard can slow
down your training, especially when used with distribution
strategies as it will incur additional synchronization overhead.
Batch-level summary writing is also available via \code{train_step}
override. Please see
\href{https://www.tensorflow.org/tensorboard/scalars_and_keras#batch-level_logging}{TensorBoard Scalars tutorial}  # noqa: E501
for more details.}

\item{profile_batch}{(Not supported at this time)
Profile the batch(es) to sample compute characteristics.
profile_batch must be a non-negative integer or a tuple of integers.
A pair of positive integers signify a range of batches to profile.
By default, profiling is disabled.}

\item{embeddings_freq}{frequency (in epochs) at which embedding layers will be
visualized. If set to 0, embeddings won't be visualized.}

\item{embeddings_metadata}{Named list which maps embedding layer names to the
filename of a file in which to save metadata for the embedding layer.
In case the same metadata file is to be
used for all embedding layers, a single filename can be passed.}
}
\value{
A \code{Callback} instance that can be passed to \code{\link[=fit.keras.src.models.model.Model]{fit.keras.src.models.model.Model()}}.
}
\description{
TensorBoard is a visualization tool provided with TensorFlow. A TensorFlow
installation is required to use this callback.

This callback logs events for TensorBoard, including:
\itemize{
\item Metrics summary plots
\item Training graph visualization
\item Weight histograms
\item Sampled profiling
}

When used in \code{model |> evaluate()} or regular validation
in addition to epoch summaries, there will be a summary that records
evaluation metrics vs \code{model$optimizer$iterations} written. The metric names
will be prepended with \code{evaluation}, with \code{model$optimizer$iterations} being
the step in the visualized TensorBoard.

If you have installed TensorFlow with \code{pip} or \code{reticulate::py_install()}, you should be able
to launch TensorBoard from the command line:

\if{html}{\out{<div class="sourceCode">}}\preformatted{tensorboard --logdir=path_to_your_logs
}\if{html}{\out{</div>}}

or from R with \code{tensorflow::tensorboard()}.

You can find more information about TensorBoard
\href{https://www.tensorflow.org/get_started/summaries_and_tensorboard}{here}.
}
\section{Examples}{
Basic usage:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tensorboard_callback <- callback_tensorboard(log_dir = "./logs")
model \%>\% fit(x_train, y_train, epochs = 2, callbacks = list(tensorboard_callback))
# Then run the tensorboard command to view the visualizations.
}\if{html}{\out{</div>}}

Custom batch-level summaries in a subclassed Model:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{MyModel <- new_model_class("MyModel",
  initialize = function() \{
    self$dense <- layer_dense(units = 10)
  \},
  call = function(x) \{
    outputs <- x |> self$dense()
    tf$summary$histogram('outputs', outputs)
    outputs
  \}
)

model <- MyModel()
model |> compile(optimizer = 'sgd', loss = 'mse')

# Make sure to set `update_freq = N` to log a batch-level summary every N
# batches. In addition to any `tf$summary` contained in `model$call()`,
# metrics added in `model |>compile` will be logged every N batches.
tb_callback <- callback_tensorboard(log_dir = './logs', update_freq = 1)
model |> fit(x_train, y_train, callbacks = list(tb_callback))
}\if{html}{\out{</div>}}

Custom batch-level summaries in a Functional API Model:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_summary <- function(x) \{
  tf$summary$histogram('x', x)
  x
\}

inputs <- layer_input(10)
outputs <- inputs |>
  layer_dense(10) |>
  layer_lambda(my_summary)

model <- keras_model(inputs, outputs)
model |> compile(optimizer = 'sgd', loss = 'mse')

# Make sure to set `update_freq = N` to log a batch-level summary every N
# batches. In addition to any `tf.summary` contained in `Model.call`,
# metrics added in `Model.compile` will be logged every N batches.
tb_callback <- callback_tensorboard(log_dir = './logs', update_freq = 1)
model |> fit(x_train, y_train, callbacks = list(tb_callback))
}\if{html}{\out{</div>}}

Profiling:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Profile a single batch, e.g. the 5th batch.
tensorboard_callback <- callback_tensorboard(
  log_dir = './logs', profile_batch = 5)
model |> fit(x_train, y_train, epochs = 2,
             callbacks = list(tensorboard_callback))

# Profile a range of batches, e.g. from 10 to 20.
tensorboard_callback <- callback_tensorboard(
  log_dir = './logs', profile_batch = c(10, 20))
model |> fit(x_train, y_train, epochs = 2,
             callbacks = list(tensorboard_callback))
}\if{html}{\out{</div>}}
}

\seealso{
\itemize{
\item \url{https://keras.io/api/callbacks/tensorboard#tensorboard-class}
}

Other callbacks: \cr
\code{\link{Callback}()} \cr
\code{\link{callback_backup_and_restore}()} \cr
\code{\link{callback_csv_logger}()} \cr
\code{\link{callback_early_stopping}()} \cr
\code{\link{callback_lambda}()} \cr
\code{\link{callback_learning_rate_scheduler}()} \cr
\code{\link{callback_model_checkpoint}()} \cr
\code{\link{callback_reduce_lr_on_plateau}()} \cr
\code{\link{callback_remote_monitor}()} \cr
\code{\link{callback_swap_ema_weights}()} \cr
\code{\link{callback_terminate_on_nan}()} \cr
}
\concept{callbacks}
