#' Test whether sampling weights are needed
#'
#' Use the DuMouchel-Duncan (1983) test to assess the need for sampling weights
#' in your linear regression analysis.
#'
#' @param model The unweighted linear model (must be \code{lm}) you want to check
#' @param weights The name of the weights column in \code{model}'s data frame
#'   or a vector of weights equal in length to the number of observations
#'   included in \code{model}.
#' @param model_output Should a summary of the model with weights as predictor
#'   be printed? Default is TRUE, but you may not want it if you are trying to
#'   declutter a document.
#'
#' @details
#'
#' This is designed to be similar to the \code{wgttest} macro for Stata
#' (\url{http://fmwww.bc.edu/repec/bocode/w/wgttest.html}). This method,
#' advocated for by DuMouchel and Duncan (1983), is fairly straightforward. To
#' decide whether weights are needed, the weights are added to the linear model
#' as a predictor and interaction with each other predictor. Then, an omnibus
#' test of significance is performed to compare the weights-added model to the
#' original; if insignificant, weights are not significantly related to the
#' result and you can use the more efficient estimation from unweighted OLS.
#'
#' It can be helpful to look at the created model using \code{model_output = TRUE}
#' to see which variables might be the ones affected by inclusion of weights.
#'
#' @references
#'
#' DuMouchel, W. H. & Duncan, D.J. (1983). Using sample survey weights in
#'   multiple regression analyses of stratified samples. \emph{Journal of the
#'   American Statistical Association}, \emph{78}. 535-543.
#'
#' Winship, C. & Radbill, L. (1994). Sampling weights and regression
#'   analysis. \emph{Sociological Methods and Research}, \emph{23}, 230-257.
#'
#'
#' @examples
#' # First, let's create some fake sampling weights
#' wts <- runif(50, 0, 5)
#' # Create model
#' fit <- lm(Income ~ Frost + Illiteracy + Murder,
#'           data = as.data.frame(state.x77))
#' # See if the weights change the model
#' wgttest(fit, wts)
#'
#'
#' @importFrom stats anova reformulate
#' @export

wgttest <- function(model, weights, model_output = TRUE) {

  # Need to parse the arguments
  wname <- as.character(substitute(weights))

  call <- getCall(model)
  if (!is.null(call$data)) {
    d <- eval(call$data)
    mframe <- FALSE # telling myself whether I use model.frame
  } else {
    d <- model.frame(model)
    mframe <- TRUE
  }

  # Save response variable
  resp <- as.character(formula(model)[2])

  terms <- attributes(terms(model))$term.labels
  terms <- unname(as.character(terms))
  nterms <- sapply(terms, c, paste(" *", wname), USE.NAMES = F, simplify = F)
  nterms <- sapply(nterms, paste, sep = "", collapse = "")

  newf <- reformulate(nterms, response = resp)

  newmod <- update(model, formula. = newf, data = d)

  aout <- anova(model, newmod)

  out <- list(aout = aout, newmod = newmod, model_output = model_output)
  class(out) <- "wgttest"
  return(out)

}

#### PRINT FUNCTION

#' @export

print.wgttest <- function(x, ...) {

  cat("Omnibus test of model change with weights\n", " F(", x$aout$Df[2], ",",
      x$aout$Res.Df[2], ") = ", round(x$aout$F[2],3),
      "\n p = ", round(x$aout$`Pr(>F)`[2],3), "\n", sep = "")

  if (x$model_output == TRUE) {
    j_summ(x$newmod, model.info = F, model.fit = F)
  }

}
