% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluation.R
\name{evaluation}
\alias{evaluation}
\title{Evaluation of Audit Samples using Confidence / Credible Bounds}
\usage{
evaluation(confidence = 0.95, method = "binomial", N = NULL,
            sample = NULL, bookValues = NULL, auditValues = NULL, counts = NULL, 
            nSumstats = NULL, kSumstats = NULL, 
            materiality = NULL, minPrecision = NULL,
            prior = FALSE, nPrior = 0, kPrior = 0, 
            rohrbachDelta = 2.7, momentPoptype = "accounts", populationBookValue = NULL,
            csA = 1, csB = 3, csMu = 0.5)
}
\arguments{
\item{confidence}{the required confidence level for the bound. Default is 0.95 for 95\% confidence.}

\item{method}{the method that is used to evaluate the sample. This can be either one of \code{poisson}, \code{binomial}, \code{hypergeometric}, \code{stringer}, \code{stringer-meikle}, \code{stringer-lta}, \code{stringer-pvz}, \code{rohrbach}, \code{moment}, \code{direct}, \code{difference}, \code{quotient}, or \code{regression}.}

\item{N}{an integer specifying the total number of units (transactions or monetary units) in the population.}

\item{sample}{a data frame containing at least a column of Ist values and a column of Soll (true) values.}

\item{bookValues}{a character specifying the column name for the Ist values in the sample.}

\item{auditValues}{a character specifying the column name for the Soll values in the sample.}

\item{counts}{a integer vector of the number of times each transaction in the sample is to be evaluated (due to it being selected multiple times for the sample).}

\item{nSumstats}{an integer specifying the number of transactions in the sample. If specified, overrides the \code{sample}, \code{bookValues} and \code{auditValues} arguments and assumes that the data come from summary statistics specified by both \code{nSumstats} and \code{kSumstats}.}

\item{kSumstats}{a value specifying the sum of taints (proportional errors) found in the sample. If specified, overrides the \code{sample}, \code{bookValues} and \code{auditValues} arguments and assumes that the data come from summary statistics specified by both \code{kSumstats} and \code{nSumstats}.}

\item{materiality}{a value specifying the performance materiality as a fraction of the total value (or size) of the population (a value between 0 and 1). If specified, the function also returns the conclusion of the analysis with respect to the performance materiality. The value is discarded when \code{direct}, \code{difference}, \code{quotient}, or \code{regression} method is chosen.}

\item{minPrecision}{a value specifying the required minimum precision. If specified, the function also returns the conclusion of the analysis with respect to the required minimum precision. This value must be specified as a fraction of the total value of the population (a value between 0 and 1).}

\item{prior}{a logical indicating whether to use a prior distribution when evaluating. Defaults to \code{FALSE} for frequentist evaluation. If \code{TRUE}, the prior distribution is updated by the corresponding likelihood. Chooses a conjugate gamma distribution for the Poisson likelihood, a conjugate beta distribution for the binomial likelihood, and a conjugate beta-binomial distribution for the hypergeometric likelihood.}

\item{nPrior}{a value for the prior parameter \eqn{\beta} (number of transactions in the assumed prior sample).}

\item{kPrior}{a value for the prior parameter \eqn{\alpha} (total tainting in the assumed prior sample).}

\item{rohrbachDelta}{a value specifying \eqn{\Delta} in Rohrbach's augmented variance bound (Rohrbach, 1993).}

\item{momentPoptype}{a character specifying the type of population for the modified moment method (Dworin and Grimlund, 1986). Can be either one of \code{accounts} or \code{inventory}. Options result in different methods for calculating the central moments.}

\item{populationBookValue}{a value specifying the total value of the transactions in the population. Required when \code{method} is one of \code{direct}, \code{difference}, \code{quotient}, or \code{regression}, but optional otherwise.}

\item{csA}{if \code{method = "coxsnell"}, the \eqn{\alpha} parameter of the prior distribution on the mean taint. Default is set to 1, as recommended by Cox and Snell (1979).}

\item{csB}{if \code{method = "coxsnell"}, the \eqn{\beta} parameter of the prior distribution on the mean taint. Default is set to 3, as recommended by Cox and Snell (1979).}

\item{csMu}{if \code{method = "coxsnell"}, the mean of the prior distribution on the mean taint. Default is set to 0.5, as recommended by Cox and Snell (1979).}
}
\value{
An object of class \code{jfaEvaluation} containing:

\item{confidence}{a value specifying the confidence level of the result.}
\item{method}{the evaluation method that was used.}
\item{N}{if \code{N} is specified, the population size that is used.}
\item{n}{an integer specifying the sample size used in the evaluation.}
\item{k}{an integer specifying the number of transactions that contained an error.}
\item{t}{a value specifying the sum of observed taints.}
\item{materiality}{if \code{materiality} is specified, the performance materiality used.}
\item{minPrecision}{if \code{minPrecision} is specified, the minimum required precision used.}
\item{mle}{a value specifying the most likely error in the population as a proportion.}
\item{precision}{a value specifying the difference between the mle and the upper confidence bound as a proportion.}
\item{popBookvalue}{if specified as input, the total Ist value of the population.}
\item{pointEstimate}{if method is one of \code{direct}, \code{difference}, \code{quotient}, or \code{regression}, the value of the point estimate.}
\item{lowerBound}{if method is one of \code{direct}, \code{difference}, \code{quotient}, or \code{regression}, the value of the lower bound of the interval.}
\item{upperBound}{if method is one of \code{direct}, \code{difference}, \code{quotient}, or \code{regression}, the value of the upper bound of the interval.}
\item{confBound}{the upper confidence bound on the error percentage. }
\item{conclusion}{if \code{materiality} is specified, the conclusion about whether to approve or not approve the population.}
\item{populationK}{the assumed total errors in the population. Used in inferences with \code{hypergeometric} method.}
\item{prior}{an object of class 'jfaPrior' to represents the prior distribution.}
\item{posterior}{an object of class 'jfaPosterior' to represents the posterior distribution.}
}
\description{
This function takes a data frame (using \code{sample}, \code{bookValue}, and \code{auditValues}) or summary statistics (using \code{nSumstats} and \code{kSumstats}) and evaluates the audit sample according to the specified method. The returned object is of class \code{jfaEvaluation} and can be used with associated \code{print()} and \code{plot()} methods.
}
\details{
This section lists the available options for the \code{methods} argument.

\itemize{
 \item{\code{poisson}:          The confidence bound taken from the Poisson distribution. If combined with \code{prior = TRUE}, performs Bayesian evaluation using a \emph{gamma} prior and posterior.}
 \item{\code{binomial}:         The confidence bound taken from the binomial distribution. If combined with \code{prior = TRUE}, performs Bayesian evaluation using a \emph{beta} prior and posterior.}
 \item{\code{hypergeometric}:   The confidence bound taken from the hypergeometric distribution. If combined with \code{prior = TRUE}, performs Bayesian evaluation using a \emph{beta-binomial} prior and posterior.}
 \item{\code{stringer}:         The Stringer bound (Stringer, 1963).}
 \item{\code{stringer-meikle}:  Stringer bound with Meikle's correction for understatements (Meikle, 1972).}
 \item{\code{stringer-lta}:     Stringer bound with LTA correction for understatements (Leslie, Teitlebaum, and Anderson, 1979).}
 \item{\code{stringer-pvz}:     Stringer bound with Pap and van Zuijlen's correction for understatements (Pap and van Zuijlen, 1996).}
 \item{\code{rohrbach}:         Rohrbach's augmented variance bound (Rohrbach, 1993).}
 \item{\code{moment}:           Modified moment bound (Dworin and Grimlund, 1986).}
 \item{\code{coxsnell}:         Cox and Snell bound (Cox and Snell, 1979).}
 \item{\code{direct}:           Confidence interval using the direct method (Touw and Hoogduin, 2011).}
 \item{\code{difference}:       Confidence interval using the difference method (Touw and Hoogduin, 2011).}
 \item{\code{quotient}:         Confidence interval using the quotient method (Touw and Hoogduin, 2011).}
 \item{\code{regression}:       Confidence interval using the regression method (Touw and Hoogduin, 2011).}
}
}
\examples{
library(jfa)
set.seed(1)

# Generate some audit data (N = 1000):
data <- data.frame(ID = sample(1000:100000, size = 1000, replace = FALSE), 
                   bookValue = runif(n = 1000, min = 700, max = 1000))

# Using monetary unit sampling, draw a random sample from the population.
s1 <- selection(population = data, sampleSize = 100, units = "mus", 
                 bookValues = "bookValue", algorithm = "random")
s1_sample <- s1$sample
s1_sample$trueValue <- s1_sample$bookValue
s1_sample$trueValue[2] <- s1_sample$trueValue[2] - 500 # One overstatement is found

# Using summary statistics, calculate the upper confidence bound according
# to the binomial distribution:

e1 <- evaluation(nSumstats = 100, kSumstats = 1, method = "binomial", 
                 materiality = 0.05)
print(e1)

# ------------------------------------------------------------ 
#             jfa Evaluation Summary (Frequentist)
# ------------------------------------------------------------ 
# Input: 
#
# Confidence:               95\%   
# Materiality:              5\% 
# Minium precision:         Not specified 
# Sample size:              100 
# Sample errors:            1 
# Sum of taints:            1 
# Method:                   binomial 
# ------------------------------------------------------------
# Output:
#
# Most likely error:        1\% 
# Upper bound:              4.66\% 
# Precision:                3.66\% 
# Conclusion:               Approve population
# ------------------------------------------------------------

# Evaluate the raw sample using the stringer bound and the sample counts:

e2 <- evaluation(sample = s1_sample, bookValues = "bookValue", auditValues = "trueValue", 
                 method = "stringer", materiality = 0.05, counts = s1_sample$counts)
print(e2)

# ------------------------------------------------------------ 
#             jfa Evaluation Summary (Frequentist)
# ------------------------------------------------------------ 
# Input: 
#
# Confidence:               95\%   
# Materiality:              5\% 
# Minium precision:         Not specified 
# Sample size:              100 
# Sample errors:            1 
# Sum of taints:            1 
# Method:                   stringer 
# ------------------------------------------------------------
# Output:
#
# Most likely error:        0.69\% 
# Upper bound:              4.12\% 
# Precision:                3.44\% 
# Conclusion:               Approve population 
# ------------------------------------------------------------  

}
\references{
Cox, D. and Snell, E. (1979). On sampling and the estimation of rare errors. \emph{Biometrika}, 66(1), 125-132.

Dworin, L., and Grimlund, R. A. (1986). Dollar-unit sampling: A comparison of the quasi-Bayesian and moment bounds. \emph{Accounting Review}, 36-57.

Leslie, D. A., Teitlebaum, A. D., & Anderson, R. J. (1979). \emph{Dollar-unit sampling: a practical guide for auditors}. Copp Clark Pitman; Belmont, Calif.: distributed by Fearon-Pitman.

Meikle, G. R. (1972). \emph{Statistical Sampling in an Audit Context: An Audit Technique}. Canadian Institute of Chartered Accountants.

Pap, G., and van Zuijlen, M. C. (1996). On the asymptotic behavior of the Stringer bound 1. \emph{Statistica Neerlandica}, 50(3), 367-389.

Rohrbach, K. J. (1993). Variance augmentation to achieve nominal coverage probability in sampling from audit populations. \emph{Auditing}, 12(2), 79.

Stringer, K. W. (1963). Practical aspects of statistical sampling in auditing. \emph{In Proceedings of the Business and Economic Statistics Section} (pp. 405-411). American Statistical Association.

Touw, P., and Hoogduin, L. (2011). \emph{Statistiek voor Audit en Controlling}. Boom uitgevers Amsterdam.
}
\seealso{
\code{\link{auditPrior}} \code{\link{planning}} \code{\link{selection}}
}
\author{
Koen Derks, \email{k.derks@nyenrode.nl}
}
\keyword{audit}
\keyword{bound}
\keyword{confidence}
\keyword{evaluation}
