% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iccNA.R
\name{iccNA}
\alias{iccNA}
\title{Intraclass correlation coefficients (ICCs) -- generalized for randomly incomplete datasets}
\usage{
iccNA(ratings, rho0 = 0, conf = 0.95, detail = FALSE, oneG = TRUE, Cs = 10000)
}
\arguments{
\item{ratings}{n*m matrix or data frame; n objects (rows), m raters (columns)}

\item{rho0}{numeric value; correlation in population (\eqn{\rho}) according to the null 
hypothesis (0 is default)}

\item{conf}{numeric value; confidence level (95\% is default)}

\item{detail}{logical; if TRUE, returns additional information (sums of squares, degrees of 
freedom, the means per object, data corrected for the raters' biases)}

\item{oneG}{logical; if TRUE, the ipsation (correction for the raters' effects) is done the 
simple way, using the difference of each raters mean to the one grand mean (\eqn{G}) of all 
values to estimate the raters' biases. If FALSE the weighted sub-means (\eqn{G_j} of those
objects that an individual rater \eqn{j} rated are used instead (cp. Brueckl, 2011, 
Equation 4.30).}

\item{Cs}{numeric value; denominator (10000 is default) of the effect-size-criterion to stop 
iteration of the correction for the raters' biases; the enumerator denotes a small effect 
(\eqn{\eta}-squared = 1\%)}
}
\value{
\item{ICCs}{data frame containing the intraclass correlation coefficients, the corresponding 
   p-values, and confidence intervals}
   \item{n}{number of rated objects}
   \item{k}{maximum number of raters per object}
   \item{amk}{mean number of ratings per object}
   \item{k_0}{approximate harmonic mean (cp. Ebel, 1951) of the number of ratings per object}
   \item{n_iter}{number of iterations for correcting for the raters' biases}
   \item{corr_ratings}{ratings, corrected for the individual raters' biases}
   \item{amO}{means of ratings for each object, based on (1) the original data and on (2) the 
   data that 
   are corrected for the raters' biases}
   \item{oneway}{statistics for the oneway ICCs}
   \item{twoway}{statistics for the twoway ICCs}
}
\description{
This function computes intraclass correlation coefficients (ICCs) as indices of 
interrater reliability or agreement based on cardinally scaled data. This function also works on 
(unbalanced) incomplete datasets without any imputation of missing values (\emph{NA}s) or (row- or 
cloumn-wise) omissions of data! p-values and confidence intervals are provided. In case of 
extreme input data (e.g. zero variances) output \emph{NaN}s are avoided by approximation.
}
\details{
This function is able to compute ICCs on randomly incomplete (i.e. unbalanced) data sets. 
Thus, both an imputation of missing values (\emph{NA}s) and row-wise or column-wise omissions of 
data are obsolete. Working on complete datasets, it yields the same results as the common 
functions, e.g. \link[irrNA]{icc_corr}.\cr
The method of Ebel (1951) is used to calculate the oneway ICCs. The solution for the twoway 
ICCs is derived from the oneway solution (cp. Brueckl, 2011, p. 96 ff.): The raters' individual 
effects (biases) are estimated, reducing this problem again to the oneway problem (cp. Greer & 
Dunlap, 1997).\cr
This estimation can be done using the difference of a certain (\eqn{j}) rater's mean to the 
grand mean (\eqn{G}) or to the sub-mean (\eqn{G_j}) representing only those objects 
that were rated by this rater. The first method is fail-safe. The second method is thought to 
provide the more precise estimates (of the raters' biases), the more the mean of the true values 
of the objects that each rater rated differ from the grand mean, e.g. if there are raters that 
only rate objects with low true values (and therefore also other raters that only rate objects 
with high true values).\cr
If the second method is chosen and if the ratings are unbalanced, which happens most of the time 
if not all raters rated all objects, the raters' biases cannot be determined exactly -- but as 
approximately as desired. This approximation needs an iteration, thus a stop criterion 
(\code{Cs}): 
The iteration is stopped, when the difference in the raters' effect size (\eqn{\eta}-squared) 
between subsequent iterations would be equal to or smaller than the \code{Cs}th part of a small 
effect (i.e. \eqn{\eta}-squared = 1\%).\cr
\cr
Just as in \link[irrNA]{icc_corr} and \link[irr]{icc}, the designation established by McGraw 
& Wong (1996) -- \emph{A} for \emph{absolute agreement} and \emph{C} for \emph{consistency} 
-- is used to differ between the (twoway) ICCs that rely on different cases and thus must be 
interpreted differently.\cr
\cr
The generalization of the procedure entails a generalization of the three cases that 
differentiate the ICCs (cp. Shrout & Fleiss, 1979):\cr
 - Case 1 (oneway case, treated by ICC(1) and ICC(k)): \cr
 Each object -- of a sample that was randomly drawn from the population of objects; also 
 holds true for case 2 and case 3 -- is rated by (a different number of) different raters 
 that were randomly drawn from the population of raters.\cr
 - Case 2 (twoway case, treated by ICC(A,1) and ICC(A,k)): \cr
 Each object is rated by a random subset of the group of raters that is drawn randomly from 
 the population of raters.\cr
 - Case 3 (twoway case, treated by ICC(C,1) and ICC(C,k)): \cr
 Each object is rated by a random subset of the group of all relevant (i.e. fixed) raters.\cr
\cr
 Output NaNs, that usually occur (see e.g. \link[irr]{icc} or \link[irrNA]{icc_corr}) in case 
 of extreme input data (e.g. in case of zero variance(s), within or between objects) are 
 avoided by approximation from little less extreme input data. Warning messages are given in 
 these cases.\cr
 \cr
}
\examples{
# Example 1:
data(ConsistNA)
# ConsistNA exhibits missing values, a perfect consistency, and 
# a moderate agreement between raters:
ConsistNA
# Common ICC-algorithms fail, since each row as well as each 
# column of ConsistNA exhibits unfilled cells and these missing 
# data are omitted column-wise or row-wise:
library(irr)
icc(ConsistNA, r0=0.3)
# Ebel's (1951) method for computing ICC(1) and ICC(k) that is 
# implemented in iccNA can cope with such data without an 
# omission or an imputation of missing values, but still can 
# not depict the raters' interdependency...
iccNA(ConsistNA, rho0=0.3)
# ...but generalizations of Ebel's method for the twoway ICCs 
# are able to assess moderate agreement (ICC(A,1) and ICC(A,k)) 
# and perfect consistency (ICC(C,1) and ICC(C,k)), assuming that 
# the data were acquired under case 2 or case 3, see Details in 
# the Help file.
#
# Example 2:
data(IndepNA)
# IndepNA exhibits missing values and zero variance between 
# the raters just as well as between the objects:
IndepNA
# Again, common ICC-algorithms fail:
icc(IndepNA)
# But iccNA is able to include all available data in its 
# calculation and thereby to show the perfect independence of 
# the ratings:
iccNA(IndepNA)
#
# Example 3:
# The example provided by Ebel (1951, Tables 2 and 3):
# data(Ebel51)
Ebel51
# iCCNA achieves to include all available ratings and to assess 
# twoway ICCs, assuming that the data were acquired under 
# case 2 or case 3:
iccNA(Ebel51, detail=TRUE)
}
\references{
Brueckl, M. (2011). Statistische Verfahren zur Ermittlung der 
Urteileruebereinstimmung. in: Altersbedingte Veraenderungen der Stimme und Sprechweise von 
Frauen, Berlin: Logos, 88--103.

Ebel, R.L. (1951). Estimation of the reliability of ratings. Psychometrika, 16(4), 
407--424.

Greer, T., & Dunlap, W.P. (1997). Analysis of variance with ipsative measures. 
Psychological Methods, 2, 200--207.

McGraw, K.O., & Wong, S.P. (1996). Forming inferences about some intraclass 
correlation coefficients. Psychological Methods, 1, 30--46.

Shrout, P.E., & Fleiss, J.L. (1979). Intraclass correlations: uses in assessing rater
reliability. Psychological Bulletin, 86(2), 420--428.
}
\seealso{
\code{\link[irrNA]{kendallNA}, \link[irrNA]{icc_corr}, \link[irr]{icc}}
}
\author{
Markus Brueckl
}
