% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shiny_support.R
\name{SurveyApp}
\alias{SurveyApp}
\title{Shiny application to generate a discrete choice survey.}
\usage{
SurveyApp(des = NULL, n.total, alts, atts, lvl.names, coding,
  alt.cte = NULL, no.choice = NULL, buttons.text, intro.text, end.text,
  data.dir = NULL, c.lvls = NULL, prior.mean = NULL,
  prior.covar = NULL, cand.set = NULL, n.draws = NULL,
  lower = NULL, upper = NULL, parallel = TRUE, reduce = TRUE)
}
\arguments{
\item{des}{A numeric matrix which represents the design matrix. Each row is a
profile.}

\item{n.total}{A numeric value indicating the total number of choice sets.}

\item{alts}{A character vector containing the names of the alternatives.}

\item{atts}{A character vector containing the names of the attributes.}

\item{lvl.names}{A list containing character vectors with the values of each
level of each attribute.}

\item{coding}{A character vector denoting the type of coding used for each
attribute. See also \code{\link[idefix]{Profiles}}.}

\item{alt.cte}{A binary vector indicating for each alternative if an
alternative specific constant is present. The default is \code{NULL}.}

\item{no.choice}{An integer indicating which alternative should be a no choice
alternative. The default is \code{NULL}.}

\item{buttons.text}{A string containing the text presented together with the
option buttons.}

\item{intro.text}{A string containing the text presented before the choice
survey.}

\item{end.text}{A string containing the text presented after the choice
survey.}

\item{data.dir}{A character string with the directory denoting where the data
needs to be written. The default is NULL}

\item{c.lvls}{A list containing numeric vectors with the attribute levels for
each continuous attribute. The default is \code{NULL}.}

\item{prior.mean}{Numeric vector indicating the mean of the multivariate
normal distribution (prior).}

\item{prior.covar}{Covariance matrix of the prior distribution.}

\item{cand.set}{A numeric matrix in which each row is a possible profile. The
\code{\link{Profiles}} function can be used to generate this matrix.}

\item{n.draws}{Numeric value indicating the number of draws.}

\item{lower}{Numeric vector of lower truncation points, the default
is \code{NULL}.}

\item{upper}{Numeric vector of upper truncation points, the default
is \code{NULL}.}

\item{parallel}{Logical value indicating whether computations should be done
over multiple cores. The default is \code{TRUE}.}

\item{reduce}{Logical value indicating whether the candidate set should be
reduced or not.}
}
\value{
After completing the survey, two text files can be found in
\code{data.dir}. The file with "num" in the filename is a matrix with the
numeric choice data. The coded design matrix ("par"), presented during the
survey, together with the observed responses ("resp") can be found here.
Rownames indicate the setnumbers. The file with "char" in the filename is a
matrix with character choice data. The labeled design matrix ("par"),
presented during the survey, together with the observed responses ("resp")
can be found here. See \code{\link{LoadData}} to load the data.
}
\description{
This function starts a shiny application which puts choice sets on screen and
saves the responses. The complete choice design can be provided in advance, or
can be generated sequentially adaptively, or can be a combination of both.
}
\details{
A pregenerated design can be specified in \code{des}. This should be a matrix
in which each row is a profile. This can be generated with \code{\link{Modfed}}
or  \code{\link{CEA}}, but it is not necessary.

If \code{n.total} = \code{nrow(des)} / \code{length(alts)}, the specified
design will be put on screen, one set after the other, and the responses will
be saved. If \code{n.total} > (\code{nrow(des)} / \code{length(alts)}), first
the specified design will be shown and afterwards the remaining sets will be
generated adaptively. If \code{des} = \code{NULL}, \code{n.total} sets will be
generated adaptively. See \code{\link{SeqMOD}} for more information on adaptive
choice sets.

Whenever adaptive sets will be generated, \code{prior.mean},
\code{prior.covar}, \code{cand.set} and \code{n.draws}, should be specified.
These arguments are necessary for the underlying importance sampling algorithm
to update the prior preference distribution. \code{lower} and \code{upper} can
be used to specify lower and upper truncation points. See
\code{\link{ImpsampMNL}} for more details.

The names specified in \code{alts} will be used to label the choice
alternatives. The names specified in \code{atts} will be used to name the
attributes in the choice sets. The values of \code{lvl.names} will be used to
create the values in the choice sets. See \code{\link{Decode}} for more
details.

The text specified in \code{buttons.text} will be displayed above the buttons
to indicate the preferred choice (for example: "indicate your preferred
choice"). The text specified in \code{intro.text} will be displayed before the
choice sets. This will generally be a description of the survey and some
instructions. The text specified in \code{end.text} will be displayed after
the survey. This will generally be a thanking note and some further
instructions.

A no choice alternative is coded as an alternative with 1 alternative specific
constant and zero's for all other attribute levels. If a no choice alternative
is present in \code{des}, or is desired when generating adaptive choice sets,
\code{no.choice} should be specified. This should be done with an integer,
indicating which alternative is the no choice option. This alternative will
not be presented on screen, but the option to select "no choice" will be. The
\code{alt.cte} argument should be specified accordingly, namely with a
\code{1} on the location of the \code{no.choice} option. See examples for
illustration.

When \code{parallel} is \code{TRUE}, \code{\link[parallel]{detectCores}} will
be used to decide upon the number of available cores. That number minus 1
cores will be used to search for the optimal adaptive choice set. For small problems
(6 parameters), \code{parallel = TRUE} can be slower. For larger problems the
computation time will decrease significantly.

When \code{reduce = TRUE}, the set of all potential choice sets will be
reduced to choice sets that have a unique information matrix. If no
alternative specific constants are used, \code{reduce} should always be
\code{TRUE}. When alternative specific constants are used \code{reduce} can be
\code{TRUE} so that the algorithm will be faster, but the combinations of
constants and profiles will not be evaluated exhaustively.
}
\examples{
\dontrun{
#### Present choice design without adaptive sets (n.total = sets in des)
# example design 
data("example_design") # pregenerated design
xdes <- example_design
### settings of the design 
code <- c("D", "D", "D")
n.sets <- 8
# settings of the survey
alternatives <- c("Alternative A", "Alternative B")
attributes <- c("Price", "Time", "Comfort")
labels <- vector(mode="list", length(attributes))
labels[[1]] <- c("$10", "$5", "$1")
labels[[2]] <- c("20 min", "12 min", "3 min")
labels[[3]] <- c("bad", "average", "good")
i.text <- "Welcome, here are some instructions ... good luck!"
b.text <- "Please choose the alternative you prefer"
e.text <- "Thanks for taking the survey"
dataDir <- getwd()
# Display the survey 
SurveyApp (des = xdes, n.total = n.sets, alts = alternatives, 
          atts = attributes, lvl.names = labels, coding = code, 
          buttons.text = b.text, intro.text = i.text, end.text = e.text)

#### Present choice design with partly adaptive sets (n.total > sets in des)
# example design 
data("example_design") # pregenerated design
xdes <- example_design
### settings of the design 
code <- c("D", "D", "D")
n.sets <- 12
# settings of the survey
alternatives <- c("Alternative A", "Alternative B")
attributes <- c("Price", "Time", "Comfort")
labels <- vector(mode="list", length(attributes))
labels[[1]] <- c("$10", "$5", "$1")
labels[[2]] <- c("20 min", "12 min", "3 min")
labels[[3]] <- c("bad", "average", "good")
i.text <- "Welcome, here are some instructions ... good luck!"
b.text <- "Please choose the alternative you prefer"
e.text <- "Thanks for taking the survey"
# setting for adaptive sets 
levels <- c(3, 3, 3)
cand <- Profiles(lvls = levels, coding = code)
p.mean <- c(0.3, 0.7, 0.3, 0.7, 0.3, 0.7)
p.var <- diag(length(p.mean))
dataDir <- getwd()
# Display the survey 
SurveyApp(des = xdes, n.total = n.sets, alts = alternatives, 
          atts = attributes, lvl.names = labels, coding = code, 
          buttons.text = b.text, intro.text = i.text, end.text = e.text, 
          prior.mean = p.mean, prior.covar = p.var, cand.set = cand, 
          n.draws = 50)
          
#### Choice design with only adaptive sets (des=NULL)
# setting for adaptive sets 
levels <- c(3, 3, 3)
p.mean <- c(0.3, 0.7, 0.3, 0.7, 0.3, 0.7)
low = c(-Inf, -Inf, -Inf, 0, 0, -Inf)
up = rep(Inf, length(p.mean))
p.var <- diag(length(p.mean)) 
code <- c("D", "D", "D")
cand <- Profiles(lvls = levels, coding = code)
n.sets <- 12
# settings of the survey
alternatives <- c("Alternative A", "Alternative B")
attributes <- c("Price", "Time", "Comfort")
labels <- vector(mode="list", length(attributes))
labels[[1]] <- c("$10", "$5", "$1")
labels[[2]] <- c("20 min", "12 min", "3 min")
labels[[3]] <- c("bad", "average", "good")
i.text <- "Welcome, here are some instructions ... good luck!"
b.text <- "Please choose the alternative you prefer"
e.text <- "Thanks for taking the survey"
dataDir <- getwd()
# Display the survey 
SurveyApp(des = NULL, n.total = n.sets, alts = alternatives,
          atts = attributes, lvl.names = labels, coding = code, 
          buttons.text = b.text, intro.text = i.text, end.text = e.text, 
          prior.mean = p.mean, prior.covar = p.var, cand.set = cand, 
          lower = low, upper = up, n.draws = 50)
# If CEA algorithm is desired, cand.set argument is not needed
SurveyApp(des = NULL, n.total = n.sets, alts = alternatives,
         atts = attributes, lvl.names = labels, coding = code, 
         buttons.text = b.text, intro.text = i.text, end.text = e.text, 
         prior.mean = p.mean, prior.covar = p.var, 
         lower = low, upper = up, n.draws = 50)
         
#### Present choice design with a no choice alternative.
# example design 
data("nochoice_design") # pregenerated design
xdes <- nochoice_design
### settings of the design 
code <- c("D", "D", "D")
n.sets <- 8
# settings of the survey
alternatives <- c("Alternative A", "Alternative B", "None")
attributes <- c("Price", "Time", "Comfort")
labels <- vector(mode = "list", length(attributes))
labels[[1]] <- c("$10", "$5", "$1")
labels[[2]] <- c("20 min", "12 min", "3 min")
labels[[3]] <- c("bad", "average", "good")
i.text <- "Welcome, here are some instructions ... good luck!"
b.text <- "Please choose the alternative you prefer"
e.text <- "Thanks for taking the survey"

# Display the survey 
SurveyApp(des = xdes, n.total = n.sets, alts = alternatives, 
          atts = attributes, lvl.names = labels, coding = code, 
          buttons.text = b.text, intro.text = i.text, end.text = e.text,
          no.choice = 3, alt.cte = c(0, 0, 1))
}
}
\references{
\insertRef{ju}{idefix}
}
