% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IMSE.R
\name{crit_IMSE}
\alias{crit_IMSE}
\title{Sequential IMSPE criterion}
\usage{
crit_IMSE(x, model, id = NULL, Wijs = NULL)
}
\arguments{
\item{x}{matrix for the new design (size 1 x d)}

\item{model}{\code{homGP} or \code{hetGP} model, including inverse matrices}

\item{id}{instead of providing \code{x}, one can provide the index of a considered existing design}

\item{Wijs}{optional previously computed matrix of Wijs, to avoid recomputing it; see \code{\link[hetGP]{Wij}}}
}
\description{
Compute the integrated mean square prediction error after adding a new design
}
\details{
The computations are scale free, i.e., values are not multiplied by \code{nu2_hat} from \code{homGP} or \code{hetGP}.
}
\examples{
## One-d toy example

set.seed(42)
ftest <- function(x, coef = 0.1) return(sin(2*pi*x) + rnorm(1, sd = coef))

n <- 9
designs <- matrix(seq(0.1, 0.9, length.out = n), ncol = 1)
X <- matrix(designs[rep(1:n, sample(1:10, n, replace = TRUE)),])
Z <- apply(X, 1, ftest)

prdata <- find_reps(X, Z, inputBounds = matrix(c(0,1), nrow = 2, ncol = 1))
Z <- prdata$Z
plot(prdata$X0[rep(1:n, times = prdata$mult),], prdata$Z, xlab = "x", ylab = "Y")

model <- mleHetGP(X = list(X0 = prdata$X0, Z0 = prdata$Z0, mult = prdata$mult),
                  Z = Z, lower = 0.1, upper = 5)

ngrid <- 501
xgrid <- matrix(seq(0,1, length.out = ngrid), ncol = 1)

## Precalculations
Wijs <- Wij(mu1 = model$X0, theta = model$theta, type = model$covtype)


t0 <- Sys.time()

IMSE_grid <- apply(xgrid, 1, crit_IMSE, Wijs = Wijs, model = model)

t1 <- Sys.time()
print(t1 - t0)

plot(xgrid, IMSE_grid/ngrid, xlab = "x", ylab = "crit_IMSE values")
abline(v = designs)

###############################################################################
## Bi-variate case

nvar <- 2 

set.seed(42)
ftest <- function(x, coef = 0.1) return(sin(2*pi*sum(x)) + rnorm(1, sd = coef))

n <- 16 # must be a square
xgrid0 <- seq(0.1, 0.9, length.out = sqrt(n))
designs <- as.matrix(expand.grid(xgrid0, xgrid0))
X <- designs[rep(1:n, sample(1:10, n, replace = TRUE)),]
Z <- apply(X, 1, ftest)

prdata <- find_reps(X, Z, inputBounds = matrix(c(0,1), nrow = 2, ncol = 1))
Z <- prdata$Z

model <- mleHetGP(X = list(X0 = prdata$X0, Z0 = prdata$Z0, mult = prdata$mult), Z = Z, 
 lower = rep(0.1, nvar), upper = rep(1, nvar))
ngrid <- 51
xgrid <- seq(0,1, length.out = ngrid)
Xgrid <- as.matrix(expand.grid(xgrid, xgrid))
## Precalculations
Wijs <- Wij(mu1 = model$X0, theta = model$theta, type = model$covtype)
t0 <- Sys.time()

IMSE_grid <- apply(Xgrid, 1, crit_IMSE, Wijs = Wijs, model = model)
filled.contour(x = xgrid, y = xgrid, matrix(IMSE_grid, ngrid),
               nlevels = 20, color.palette = terrain.colors,
               main = "Sequential IMSPE values")
}
\seealso{
\code{\link[hetGP]{deriv_crit_IMSE}} for the derivative
}
