/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND 

#include "H5FSmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5FSpkg.h"     
#include "H5MFprivate.h" 
#include "H5SLprivate.h" 
#include "H5VMprivate.h" 

typedef struct {
    H5FS_t         *fspace;  
    H5FS_operator_t op;      
    void           *op_data; 
} H5FS_iter_ud_t;

static herr_t H5FS__sect_increase(H5FS_t *fspace, const H5FS_section_class_t *cls, unsigned flags);
static herr_t H5FS__sect_decrease(H5FS_t *fspace, const H5FS_section_class_t *cls);
static herr_t H5FS__size_node_decr(H5FS_sinfo_t *sinfo, unsigned bin, H5FS_node_t *fspace_node,
                                   const H5FS_section_class_t *cls);
static herr_t H5FS__sect_unlink_size(H5FS_sinfo_t *sinfo, const H5FS_section_class_t *cls,
                                     H5FS_section_info_t *sect);
static herr_t H5FS__sect_unlink_rest(H5FS_t *fspace, const H5FS_section_class_t *cls,
                                     H5FS_section_info_t *sect);
static herr_t H5FS__sect_remove_real(H5FS_t *fspace, H5FS_section_info_t *sect);
static herr_t H5FS__sect_link_size(H5FS_sinfo_t *sinfo, const H5FS_section_class_t *cls,
                                   H5FS_section_info_t *sect);
static herr_t H5FS__sect_link_rest(H5FS_t *fspace, const H5FS_section_class_t *cls, H5FS_section_info_t *sect,
                                   unsigned flags);
static herr_t H5FS__sect_link(H5FS_t *fspace, H5FS_section_info_t *sect, unsigned flags);
static herr_t H5FS__sect_merge(H5FS_t *fspace, H5FS_section_info_t **sect, void *op_data);
static htri_t H5FS__sect_find_node(H5FS_t *fspace, hsize_t request, H5FS_section_info_t **node);
static herr_t H5FS__sect_serialize_size(H5FS_t *fspace);

H5FL_DEFINE(H5FS_node_t);

H5FL_SEQ_DEFINE(H5FS_bin_t);

H5FL_DEFINE(H5FS_sinfo_t);

H5FS_sinfo_t *
H5FS__sinfo_new(H5F_t *f, H5FS_t *fspace)
{
    H5FS_sinfo_t *sinfo     = NULL; 
    H5FS_sinfo_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(fspace);
#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr, "%s: fspace->addr = %" PRIuHADDR "\n", __func__, fspace->addr);
#endif 

    
    if (NULL == (sinfo = H5FL_CALLOC(H5FS_sinfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    sinfo->nbins            = H5VM_log2_gen(fspace->max_sect_size);
    sinfo->sect_prefix_size = H5FS_SINFO_PREFIX_SIZE(f);
    sinfo->sect_off_size    = (fspace->max_sect_addr + 7) / 8;
    sinfo->sect_len_size    = H5VM_limit_enc_size((uint64_t)fspace->max_sect_size);
#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr, "%s: fspace->max_sect_size = %" PRIuHSIZE "\n", __func__, fspace->max_sect_size);
    Rfprintf(Rstderr, "%s: fspace->max_sect_addr = %u\n", __func__, fspace->max_sect_addr);
    Rfprintf(Rstderr, "%s: sinfo->nbins = %u\n", __func__, sinfo->nbins);
    Rfprintf(Rstderr, "%s: sinfo->sect_off_size = %u, sinfo->sect_len_size = %u\n", __func__,
            sinfo->sect_off_size, sinfo->sect_len_size);
#endif 

    
    if (NULL == (sinfo->bins = H5FL_SEQ_CALLOC(H5FS_bin_t, (size_t)sinfo->nbins)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "memory allocation failed for free space section bin array");

    
    if (H5FS__incr(fspace) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTINC, NULL, "unable to increment ref. count on free space header");
    sinfo->fspace = fspace;

    
    
    assert(fspace->sinfo == NULL);
    fspace->sinfo = sinfo;

    
    ret_value = sinfo;

done:
    if (ret_value == NULL && sinfo) {
        
        if (sinfo->bins)
            sinfo->bins = H5FL_SEQ_FREE(H5FS_bin_t, sinfo->bins);

        
        sinfo = H5FL_FREE(H5FS_sinfo_t, sinfo);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sinfo_lock(H5F_t *f, H5FS_t *fspace, unsigned accmode)
{
    H5FS_sinfo_cache_ud_t cache_udata;         
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr,
            "%s: Called, fspace->addr = %" PRIuHADDR ", fspace->sinfo = %p, fspace->sect_addr = %" PRIuHADDR
            "\n",
            __func__, fspace->addr, (void *)fspace->sinfo, fspace->sect_addr);
    Rfprintf(Rstderr, "%s: fspace->alloc_sect_size = %" PRIuHSIZE ", fspace->sect_size = %" PRIuHSIZE "\n",
            __func__, fspace->alloc_sect_size, fspace->sect_size);
#endif 

    
    assert(f);
    assert(fspace);

    
    assert((accmode & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    if (fspace->sinfo) {
        

        
        assert(((fspace->sinfo_accmode) & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

        if (fspace->sinfo_protected && accmode != fspace->sinfo_accmode) {
            
            if (0 == (accmode & (unsigned)(~H5AC__READ_ONLY_FLAG))) {
                
                if (H5AC_unprotect(f, H5AC_FSPACE_SINFO, fspace->sect_addr, fspace->sinfo,
                                   H5AC__NO_FLAGS_SET) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTUNPROTECT, FAIL,
                                "unable to release free space section info");

                
                cache_udata.f      = f;
                cache_udata.fspace = fspace;
                if (NULL == (fspace->sinfo = (H5FS_sinfo_t *)H5AC_protect(
                                 f, H5AC_FSPACE_SINFO, fspace->sect_addr, &cache_udata, H5AC__NO_FLAGS_SET)))
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTPROTECT, FAIL, "unable to load free space sections");

                
                fspace->sinfo_accmode = H5AC__NO_FLAGS_SET;
            } 
        }     
    }         
    else {
        
        if (H5_addr_defined(fspace->sect_addr)) {
            
            assert(fspace->sinfo_protected == false);
            assert(H5_addr_defined(fspace->addr));

#ifdef H5FS_SINFO_DEBUG
            Rfprintf(Rstderr, "%s: Reading in existing sections, fspace->sect_addr = %" PRIuHADDR "\n",
                    __func__, fspace->sect_addr);
#endif 
            
            cache_udata.f      = f;
            cache_udata.fspace = fspace;
            if (NULL == (fspace->sinfo = (H5FS_sinfo_t *)H5AC_protect(f, H5AC_FSPACE_SINFO, fspace->sect_addr,
                                                                      &cache_udata, accmode)))
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTPROTECT, FAIL, "unable to load free space sections");

            
            fspace->sinfo_protected = true;
            fspace->sinfo_accmode   = accmode;
        } 
        else {
#ifdef H5FS_SINFO_DEBUG
            Rfprintf(Rstderr, "%s: Creating new section info\n", __func__);
#endif 
            
            assert(fspace->tot_sect_count == 0);
            assert(fspace->serial_sect_count == 0);
            assert(fspace->ghost_sect_count == 0);

            
            if (NULL == (fspace->sinfo = H5FS__sinfo_new(f, fspace)))
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTCREATE, FAIL, "can't create section info");

            
            fspace->sect_size = fspace->alloc_sect_size = 0;
        } 
    }     
    assert(fspace->rc == 2);

    
    fspace->sinfo_lock_count++;

done:
#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr,
            "%s: Leaving, fspace->addr = %" PRIuHADDR ", fspace->sinfo = %p, fspace->sect_addr = %" PRIuHADDR
            "\n",
            __func__, fspace->addr, (void *)fspace->sinfo, fspace->sect_addr);
    Rfprintf(Rstderr, "%s: fspace->alloc_sect_size = %" PRIuHSIZE ", fspace->sect_size = %" PRIuHSIZE "\n",
            __func__, fspace->alloc_sect_size, fspace->sect_size);
#endif 
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sinfo_unlock(H5F_t *f, H5FS_t *fspace, bool modified)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE
#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr,
            "%s: Called, modified = %d, fspace->addr = %" PRIuHADDR ", fspace->sect_addr = %" PRIuHADDR "\n",
            __func__, modified, fspace->addr, fspace->sect_addr);
    Rfprintf(Rstderr,
            "%s: fspace->sinfo_lock_count = %u, fspace->sinfo_modified = %d, fspace->sinfo_protected = %d\n",
            __func__, fspace->sinfo_lock_count, fspace->sinfo_modified, fspace->sinfo_protected);
    Rfprintf(Rstderr, "%s: fspace->alloc_sect_size = %" PRIuHSIZE ", fspace->sect_size = %" PRIuHSIZE "\n",
            __func__, fspace->alloc_sect_size, fspace->sect_size);
#endif 

    
    assert(f);
    assert(fspace);
    assert(fspace->rc == 2);
    assert(fspace->sinfo);

    
    if (modified) {
        
        if (fspace->sinfo_protected && (0 != ((fspace->sinfo_accmode) & H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTDIRTY, FAIL, "attempt to modify read-only section info");

        
        fspace->sinfo->dirty = true;

        
        fspace->sinfo_modified = true;

        
        if (H5FS__dirty(fspace) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL, "unable to mark free space header as dirty");
    } 

    
    fspace->sinfo_lock_count--;

    
    if (fspace->sinfo_lock_count == 0) {
        bool release_sinfo_space = false; 
        bool closing_or_flushing = f->shared->closing; 

        
        if (!closing_or_flushing &&
            H5AC_get_cache_flush_in_progress(f->shared->cache, &closing_or_flushing) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Can't get flush_in_progress");

        
        if (fspace->sinfo_protected) {
            unsigned cache_flags = H5AC__NO_FLAGS_SET; 

            
            assert(H5_addr_defined(fspace->addr));

            
            if (fspace->sinfo_modified) {
                
                cache_flags |= H5AC__DIRTIED_FLAG;

                
                if (closing_or_flushing) {
                    if (fspace->sect_size > fspace->alloc_sect_size)
                        cache_flags |= H5AC__DELETED_FLAG | H5AC__TAKE_OWNERSHIP_FLAG;
                    else
                        fspace->sect_size = fspace->alloc_sect_size;

                    
                }
                else if (fspace->sect_size != fspace->alloc_sect_size)
                    cache_flags |= H5AC__DELETED_FLAG | H5AC__TAKE_OWNERSHIP_FLAG;

            } 

            
            assert(H5_addr_defined(fspace->sect_addr));

            
            
            
#ifdef H5FS_SINFO_DEBUG
            Rfprintf(Rstderr, "%s: Unprotecting section info, cache_flags = %u\n", __func__, cache_flags);
#endif 
            if (H5AC_unprotect(f, H5AC_FSPACE_SINFO, fspace->sect_addr, fspace->sinfo, cache_flags) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTUNPROTECT, FAIL, "unable to release free space section info");

            
            fspace->sinfo_protected = false;

            
            if ((cache_flags & H5AC__TAKE_OWNERSHIP_FLAG)) {
#ifdef H5FS_SINFO_DEBUG
                Rfprintf(Rstderr, "%s: Taking ownership of section info\n", __func__);
#endif 
                
                release_sinfo_space = true;
            } 
            else {
#ifdef H5FS_SINFO_DEBUG
                Rfprintf(Rstderr, "%s: Relinquishing section info ownership\n", __func__);
#endif 
                
                fspace->sinfo = NULL;
            } 
        }     
        else {
            
            if (fspace->sinfo_modified) {
                
                if (H5_addr_defined(fspace->sect_addr)) {
                    
                    
                    if (closing_or_flushing) {
                        if (fspace->sect_size > fspace->alloc_sect_size)
                            release_sinfo_space = true;
                        else
                            fspace->sect_size = fspace->alloc_sect_size;
                    }
                    else
                        release_sinfo_space = true;
                }
                else
                    assert(fspace->alloc_sect_size == 0);

            } 
            else {
                
                if (H5_addr_defined(fspace->sect_addr))
                    assert(fspace->alloc_sect_size == fspace->sect_size);
                else
                    assert(fspace->alloc_sect_size == 0);
            } 
        }     

        
        fspace->sinfo_modified = false;

        
        if (release_sinfo_space) {
            haddr_t old_sect_addr       = fspace->sect_addr; 
            hsize_t old_alloc_sect_size = fspace->alloc_sect_size; 

            
            assert(H5_addr_defined(fspace->addr));

            
            fspace->sect_addr       = HADDR_UNDEF;
            fspace->alloc_sect_size = 0;

            
            if (!modified)
                if (H5FS__dirty(fspace) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL,
                                "unable to mark free space header as dirty");

#ifdef H5FS_SINFO_DEBUG
            Rfprintf(Rstderr,
                    "%s: Freeing section info on disk, old_sect_addr = %" PRIuHADDR
                    ", old_alloc_sect_size = %" PRIuHSIZE "\n",
                    __func__, old_sect_addr, old_alloc_sect_size);
#endif 
            
            if (!H5F_IS_TMP_ADDR(f, old_sect_addr))
                if (H5MF_xfree(f, H5FD_MEM_FSPACE_SINFO, old_sect_addr, old_alloc_sect_size) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "unable to free free space sections");
        } 
    }     

done:
#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr, "%s: Leaving, ret_value = %d\n", __func__, ret_value);
#endif 
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_serialize_size(H5FS_t *fspace)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(fspace);

    
    if (fspace->serial_sect_count > 0) {
        size_t sect_buf_size; 

        
        sect_buf_size = fspace->sinfo->sect_prefix_size;

        
        sect_buf_size +=
            fspace->sinfo->serial_size_count * H5VM_limit_enc_size((uint64_t)fspace->serial_sect_count);

        
        sect_buf_size += fspace->sinfo->serial_size_count * fspace->sinfo->sect_len_size;

        
        sect_buf_size += fspace->serial_sect_count * fspace->sinfo->sect_off_size;

        
        sect_buf_size += fspace->serial_sect_count * 1 ;

        
        sect_buf_size += fspace->sinfo->serial_size;

        
        fspace->sect_size = sect_buf_size;
    } 
    else
        
        fspace->sect_size = fspace->sinfo->sect_prefix_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FS__sect_increase(H5FS_t *fspace, const H5FS_section_class_t *cls, unsigned flags)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fspace);
    assert(fspace->sinfo);
    assert(cls);

    
    fspace->tot_sect_count++;

    
    if (cls->flags & H5FS_CLS_GHOST_OBJ) {
        
        assert(cls->serial_size == 0);

        
        fspace->ghost_sect_count++;
    } 
    else {
        
        fspace->serial_sect_count++;

        
        fspace->sinfo->serial_size += cls->serial_size;

        
        
        if (!(flags & H5FS_ADD_DESERIALIZING)) {
            if (H5FS__sect_serialize_size(fspace) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTCOMPUTE, FAIL,
                            "can't adjust free space section size on disk");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_decrease(H5FS_t *fspace, const H5FS_section_class_t *cls)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fspace);
    assert(fspace->sinfo);
    assert(cls);

    
    fspace->tot_sect_count--;

    
    if (cls->flags & H5FS_CLS_GHOST_OBJ) {
        
        assert(cls->serial_size == 0);

        
        fspace->ghost_sect_count--;
    } 
    else {
        
        fspace->serial_sect_count--;

        
        fspace->sinfo->serial_size -= cls->serial_size;

        
        if (H5FS__sect_serialize_size(fspace) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTCOMPUTE, FAIL, "can't adjust free space section size on disk");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__size_node_decr(H5FS_sinfo_t *sinfo, unsigned bin, H5FS_node_t *fspace_node,
                     const H5FS_section_class_t *cls)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(sinfo);
    assert(fspace_node);
    assert(cls);

    
    
    sinfo->bins[bin].tot_sect_count--;

    
    if (cls->flags & H5FS_CLS_GHOST_OBJ) {
        
        fspace_node->ghost_count--;

        
        sinfo->bins[bin].ghost_sect_count--;

        
        if (fspace_node->ghost_count == 0)
            sinfo->ghost_size_count--;
    } 
    else {
        
        fspace_node->serial_count--;

        
        sinfo->bins[bin].serial_sect_count--;

        
        if (fspace_node->serial_count == 0)
            sinfo->serial_size_count--;
    } 

    
    if (H5SL_count(fspace_node->sect_list) == 0) {
        H5FS_node_t *tmp_fspace_node; 

        
        assert(fspace_node->ghost_count == 0);
        assert(fspace_node->serial_count == 0);

        
        tmp_fspace_node = (H5FS_node_t *)H5SL_remove(sinfo->bins[bin].bin_list, &fspace_node->sect_size);
        if (tmp_fspace_node == NULL || tmp_fspace_node != fspace_node)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTREMOVE, FAIL, "can't remove free space node from skip list");

        
        if (H5SL_close(fspace_node->sect_list) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTCLOSEOBJ, FAIL, "can't destroy size tracking node's skip list");

        
        fspace_node = H5FL_FREE(H5FS_node_t, fspace_node);

        
        sinfo->tot_size_count--;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_unlink_size(H5FS_sinfo_t *sinfo, const H5FS_section_class_t *cls, H5FS_section_info_t *sect)
{
    H5FS_node_t         *fspace_node;         
    H5FS_section_info_t *tmp_sect_node;       
    unsigned             bin;                 
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(sinfo);
    assert(sinfo->bins);
    assert(sect);
    assert(cls);

    
    bin = H5VM_log2_gen(sect->size);
    assert(bin < sinfo->nbins);
    if (sinfo->bins[bin].bin_list == NULL)
        HGOTO_ERROR(H5E_FSPACE, H5E_NOTFOUND, FAIL, "node's bin is empty?");

    
    if ((fspace_node = (H5FS_node_t *)H5SL_search(sinfo->bins[bin].bin_list, &sect->size)) == NULL)
        HGOTO_ERROR(H5E_FSPACE, H5E_NOTFOUND, FAIL, "can't find section size node");

    
    tmp_sect_node = (H5FS_section_info_t *)H5SL_remove(fspace_node->sect_list, &sect->addr);
    if (tmp_sect_node == NULL || tmp_sect_node != sect)
        HGOTO_ERROR(H5E_FSPACE, H5E_NOTFOUND, FAIL, "can't find section node on size list");

    
    if (H5FS__size_node_decr(sinfo, bin, fspace_node, cls) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTREMOVE, FAIL, "can't remove free space size node from skip list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_unlink_rest(H5FS_t *fspace, const H5FS_section_class_t *cls, H5FS_section_info_t *sect)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fspace);
    assert(fspace->sinfo);
    assert(cls);
    assert(sect);

    
    if (!(cls->flags & H5FS_CLS_SEPAR_OBJ)) {
        H5FS_section_info_t *tmp_sect_node; 

        tmp_sect_node = (H5FS_section_info_t *)H5SL_remove(fspace->sinfo->merge_list, &sect->addr);
        if (tmp_sect_node == NULL || tmp_sect_node != sect)
            HGOTO_ERROR(H5E_FSPACE, H5E_NOTFOUND, FAIL, "can't find section node on size list");
    } 

    
    if (H5FS__sect_decrease(fspace, cls) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't increase free space section size on disk");

    
    fspace->tot_space -= sect->size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_remove_real(H5FS_t *fspace, H5FS_section_info_t *sect)
{
    const H5FS_section_class_t *cls;                 
    herr_t                      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fspace);
    assert(fspace->sinfo);
    assert(sect);

    
    cls = &fspace->sect_cls[sect->type];

    
    if (H5FS__sect_unlink_size(fspace->sinfo, cls, sect) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL,
                    "can't remove section from size tracking data structures");

    
    if (H5FS__sect_unlink_rest(fspace, cls, sect) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL,
                    "can't remove section from non-size tracking data structures");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_sect_remove(H5F_t *f, H5FS_t *fspace, H5FS_section_info_t *sect)
{
    bool   sinfo_valid = false;   
    herr_t ret_value   = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(f);
    assert(fspace);
    assert(sect);

    
    if (H5FS__sinfo_lock(f, fspace, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get section info");
    sinfo_valid = true;

    
    if (H5FS__sect_remove_real(fspace, sect) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTREMOVE, FAIL, "can't remove section");

done:
    
    if (sinfo_valid && H5FS__sinfo_unlock(f, fspace, true) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL, "can't release section info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_link_size(H5FS_sinfo_t *sinfo, const H5FS_section_class_t *cls, H5FS_section_info_t *sect)
{
    H5FS_node_t *fspace_node       = NULL;  
    bool         fspace_node_alloc = false; 
    unsigned     bin;                       
    herr_t       ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(sinfo);
    assert(sect);
    assert(H5_addr_defined(sect->addr));
    assert(sect->size);

    
    bin = H5VM_log2_gen(sect->size);
    assert(bin < sinfo->nbins);
    if (sinfo->bins[bin].bin_list == NULL) {
        if (NULL == (sinfo->bins[bin].bin_list = H5SL_create(H5SL_TYPE_HSIZE, NULL)))
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTCREATE, FAIL, "can't create skip list for free space nodes");
    } 
    else
        
        fspace_node = (H5FS_node_t *)H5SL_search(sinfo->bins[bin].bin_list, &sect->size);

    
    if (fspace_node == NULL) {
        
        if (NULL == (fspace_node = H5FL_MALLOC(H5FS_node_t)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for free space node");
        fspace_node_alloc = true;

        
        fspace_node->sect_size    = sect->size;
        fspace_node->serial_count = fspace_node->ghost_count = 0;
        if (NULL == (fspace_node->sect_list = H5SL_create(H5SL_TYPE_HADDR, NULL)))
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTCREATE, FAIL, "can't create skip list for free space nodes");

        
        if (H5SL_insert(sinfo->bins[bin].bin_list, fspace_node, &fspace_node->sect_size) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't insert free space node into skip list");
        fspace_node_alloc = false; 

        
        sinfo->tot_size_count++;
    } 

    
    
    sinfo->bins[bin].tot_sect_count++;
    if (cls->flags & H5FS_CLS_GHOST_OBJ) {
        sinfo->bins[bin].ghost_sect_count++;
        fspace_node->ghost_count++;

        
        if (fspace_node->ghost_count == 1)
            sinfo->ghost_size_count++;
    } 
    else {
        sinfo->bins[bin].serial_sect_count++;
        fspace_node->serial_count++;

        
        if (fspace_node->serial_count == 1)
            sinfo->serial_size_count++;
    } 

    
    if (H5SL_insert(fspace_node->sect_list, sect, &sect->addr) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't insert free space node into skip list");

done:
    if (ret_value < 0)
        if (fspace_node && fspace_node_alloc) {
            if (fspace_node->sect_list && H5SL_close(fspace_node->sect_list) < 0)
                HDONE_ERROR(H5E_FSPACE, H5E_CANTCLOSEOBJ, FAIL,
                            "can't destroy size free space node's skip list");
            fspace_node = H5FL_FREE(H5FS_node_t, fspace_node);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_link_rest(H5FS_t *fspace, const H5FS_section_class_t *cls, H5FS_section_info_t *sect,
                     unsigned flags)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fspace);
    assert(fspace->sinfo);
    assert(sect);

    
    if (!(cls->flags & H5FS_CLS_SEPAR_OBJ)) {
        if (fspace->sinfo->merge_list == NULL)
            if (NULL == (fspace->sinfo->merge_list = H5SL_create(H5SL_TYPE_HADDR, NULL)))
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTCREATE, FAIL,
                            "can't create skip list for merging free space sections");
        if (H5SL_insert(fspace->sinfo->merge_list, sect, &sect->addr) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL,
                        "can't insert free space node into merging skip list");
    } 

    
    if (H5FS__sect_increase(fspace, cls, flags) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't increase free space section size on disk");

    
    fspace->tot_space += sect->size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_link(H5FS_t *fspace, H5FS_section_info_t *sect, unsigned flags)
{
    const H5FS_section_class_t *cls;                   
    bool                        linked_sect = false;   
    herr_t                      ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fspace);
    assert(fspace->sinfo);
    assert(sect);

    
    cls = &fspace->sect_cls[sect->type];

    
    if (H5FS__sect_link_size(fspace->sinfo, cls, sect) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't add section to size tracking data structures");
    linked_sect = true;

    
    if (H5FS__sect_link_rest(fspace, cls, sect, flags) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL,
                    "can't add section to non-size tracking data structures");

done:
    if (ret_value < 0) {
        if (linked_sect && H5FS__sect_unlink_size(fspace->sinfo, cls, sect) < 0)
            HDONE_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL,
                        "can't remove section from size tracking data structures");
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sect_merge(H5FS_t *fspace, H5FS_section_info_t **sect, void *op_data)
{
    H5FS_section_class_t *sect_cls;            
    bool                  modified;            
    bool                  remove_sect = false; 
    htri_t                status;              
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fspace);
    assert(*sect);
    assert(H5_addr_defined((*sect)->addr));
    assert((*sect)->size);

    
    if (fspace->sinfo->merge_list) {
        do {
            H5SL_node_t *less_sect_node;           
            H5SL_node_t *greater_sect_node = NULL; 
            H5FS_section_info_t  *tmp_sect;        
            H5FS_section_class_t *tmp_sect_cls;    
            bool greater_sect_node_valid = false;  

            
            modified = false;

            
            less_sect_node = H5SL_below(fspace->sinfo->merge_list, &(*sect)->addr);

            
            if (less_sect_node) {
                
                greater_sect_node       = H5SL_next(less_sect_node);
                greater_sect_node_valid = true;

                
                tmp_sect = (H5FS_section_info_t *)H5SL_item(less_sect_node);

                
                tmp_sect_cls = &fspace->sect_cls[tmp_sect->type];
                sect_cls     = &fspace->sect_cls[(*sect)->type];

                
                if ((!(tmp_sect_cls->flags & H5FS_CLS_MERGE_SYM) || (tmp_sect->type == (*sect)->type)) &&
                    tmp_sect_cls->can_merge) {
                    
                    if ((status = (*tmp_sect_cls->can_merge)(tmp_sect, *sect, op_data)) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTMERGE, FAIL, "can't check for merging sections");
                    if (status > 0) {
                        
                        assert(tmp_sect_cls->merge);

                        
                        if (H5FS__sect_remove_real(fspace, tmp_sect) < 0)
                            HGOTO_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL,
                                        "can't remove section from internal data structures");

                        
                        if ((*tmp_sect_cls->merge)(&tmp_sect, *sect, op_data) < 0)
                            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't merge two sections");

                        
                        *sect = tmp_sect;
                        if (*sect == NULL)
                            HGOTO_DONE(ret_value);

                        
                        modified = true;
                    } 
                }     
            }         

            
            if (!greater_sect_node_valid)
                greater_sect_node = H5SL_above(fspace->sinfo->merge_list, &(*sect)->addr);

            
            if (greater_sect_node) {
                
                tmp_sect = (H5FS_section_info_t *)H5SL_item(greater_sect_node);

                
                sect_cls     = &fspace->sect_cls[(*sect)->type];
                tmp_sect_cls = &fspace->sect_cls[tmp_sect->type];

                
                if ((!(sect_cls->flags & H5FS_CLS_MERGE_SYM) || ((*sect)->type == tmp_sect->type)) &&
                    sect_cls->can_merge) {

                    
                    if ((status = (*sect_cls->can_merge)(*sect, tmp_sect, op_data)) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTMERGE, FAIL, "can't check for merging sections");
                    if (status > 0) {
                        
                        assert(sect_cls->merge);

                        
                        if (H5FS__sect_remove_real(fspace, tmp_sect) < 0)
                            HGOTO_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL,
                                        "can't remove section from internal data structures");

                        
                        if ((*sect_cls->merge)(sect, tmp_sect, op_data) < 0)
                            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't merge two sections");

                        
                        if (*sect == NULL)
                            HGOTO_DONE(ret_value);

                        
                        modified = true;
                    } 
                }     
            }         
        } while (modified);
    } 
    assert(*sect);

    
    do {
        
        modified = false;

        
        sect_cls = &fspace->sect_cls[(*sect)->type];
        if (sect_cls->can_shrink) {
            if ((status = (*sect_cls->can_shrink)(*sect, op_data)) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTSHRINK, FAIL, "can't check for shrinking container");
            if (status > 0) {
                
                
                if (remove_sect) {
                    if (H5FS__sect_remove_real(fspace, *sect) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL,
                                    "can't remove section from internal data structures");
                    remove_sect = false;
                } 

                
                
                assert(sect_cls->shrink);
                if ((*sect_cls->shrink)(sect, op_data) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't shrink free space container");

                
                if (*sect == NULL) {
                    
                    if (fspace->sinfo->merge_list) {
                        H5SL_node_t *last_node; 

                        
                        if (NULL != (last_node = H5SL_last(fspace->sinfo->merge_list))) {
                            
                            *sect = (H5FS_section_info_t *)H5SL_item(last_node);
                            assert(*sect);

                            
                            remove_sect = true;
                        } 
                    }     
                }         

                
                modified = true;
            } 
        }     
    } while (modified && *sect);

    
    if (remove_sect && (*sect != NULL))
        *sect = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_sect_add(H5F_t *f, H5FS_t *fspace, H5FS_section_info_t *sect, unsigned flags, void *op_data)
{
    H5FS_section_class_t *cls;                      
    bool                  sinfo_valid    = false;   
    bool                  sinfo_modified = false;   
    herr_t                ret_value      = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr, "%s: *sect = {%" PRIuHADDR ", %" PRIuHSIZE ", %u, %s}\n", __func__, sect->addr,
            sect->size, sect->type,
            (sect->state == H5FS_SECT_LIVE ? "H5FS_SECT_LIVE" : "H5FS_SECT_SERIALIZED"));
#endif 

    
    assert(fspace);
    assert(sect);
    assert(H5_addr_defined(sect->addr));
    assert(sect->size);

    
    if (H5FS__sinfo_lock(f, fspace, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get section info");
    sinfo_valid = true;

    
    cls = &fspace->sect_cls[sect->type];
    if (cls->add)
        if ((*cls->add)(&sect, &flags, op_data) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "'add' section class callback failed");

    
    if (flags & H5FS_ADD_RETURNED_SPACE) {
#ifdef H5FS_SINFO_DEBUG
        Rfprintf(Rstderr, "%s: Returning space\n", __func__);
#endif 

        
        if (H5FS__sect_merge(fspace, &sect, op_data) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTMERGE, FAIL, "can't merge sections");
    } 

    
    
    if (sect)
        if (H5FS__sect_link(fspace, sect, flags) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't insert free space section into skip list");

#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr, "%s: fspace->tot_space = %" PRIuHSIZE "\n", __func__, fspace->tot_space);
#endif 
    
    
    if (!(flags & (H5FS_ADD_DESERIALIZING | H5FS_PAGE_END_NO_ADD)))
        sinfo_modified = true;

done:
    
    if (sinfo_valid && H5FS__sinfo_unlock(f, fspace, sinfo_modified) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL, "can't release section info");

#ifdef H5FS_DEBUG_ASSERT
    if (!(flags & (H5FS_ADD_DESERIALIZING | H5FS_ADD_SKIP_VALID)))
        H5FS__assert(fspace);
#endif 
#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr, "%s: Leaving, ret_value = %d\n", __func__, ret_value);
#endif 
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5FS_sect_try_extend(H5F_t *f, H5FS_t *fspace, haddr_t addr, hsize_t size, hsize_t extra_requested,
                     unsigned flags, void *op_data)
{
    bool   sinfo_valid    = false; 
    bool   sinfo_modified = false; 
    htri_t ret_value      = false; 

    FUNC_ENTER_NOAPI(FAIL)

#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr, "%s: addr = %" PRIuHADDR ", size = %" PRIuHSIZE ", extra_requested = %" PRIuHSIZE "\n",
            __func__, addr, size, extra_requested);
#endif 

    
    assert(f);
    assert(fspace);
    assert(H5_addr_defined(addr));
    assert(size > 0);
    assert(extra_requested > 0);

    
#ifdef H5FS_SINFO_DEBUG
    Rfprintf(Rstderr, "%s: fspace->tot_sect_count = %" PRIuHSIZE "\n", __func__, fspace->tot_sect_count);
    Rfprintf(Rstderr, "%s: fspace->serial_sect_count = %" PRIuHSIZE "\n", __func__, fspace->serial_sect_count);
    Rfprintf(Rstderr, "%s: fspace->ghost_sect_count = %" PRIuHSIZE "\n", __func__, fspace->ghost_sect_count);
#endif 
    if (fspace->tot_sect_count > 0) {
        H5FS_section_info_t *sect; 

        
        if (H5FS__sinfo_lock(f, fspace, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get section info");
        sinfo_valid = true;

        
        
        if ((sect = (H5FS_section_info_t *)H5SL_greater(fspace->sinfo->merge_list, &addr))) {
            
            if (sect->size >= extra_requested && (addr + size) == sect->addr) {
                H5FS_section_class_t *cls; 

                
                if (H5FS__sect_remove_real(fspace, sect) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL,
                                "can't remove section from internal data structures");

                
                cls = &fspace->sect_cls[sect->type];

                
                
                if (sect->size > extra_requested) {
                    
                    assert(cls->flags & H5FS_CLS_ADJUST_OK);

                    
                    sect->addr += extra_requested;
                    sect->size -= extra_requested;
                    if (cls->add)
                        if ((*cls->add)(&sect, &flags, op_data) < 0)
                            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL,
                                        "'add' section class callback failed");

                    
                    if (sect) {
                        
                        if (H5FS__sect_link(fspace, sect, 0) < 0)
                            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL,
                                        "can't insert free space section into skip list");
                    } 
                }     
                else {
                    
                    assert(sect->size == extra_requested);

                    
                    if ((*cls->free)(sect) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "can't free section");
                } 

                
                sinfo_modified = true;

                
                HGOTO_DONE(true);
            } 
        }     
    }         

done:
    
    if (sinfo_valid && H5FS__sinfo_unlock(f, fspace, sinfo_modified) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL, "can't release section info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5FS_sect_try_merge(H5F_t *f, H5FS_t *fspace, H5FS_section_info_t *sect, unsigned flags, void *op_data)
{
    bool    sinfo_valid    = false; 
    bool    sinfo_modified = false; 
    hsize_t saved_fs_size;          
    htri_t  ret_value = false;      

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(fspace);
    assert(sect);
    assert(H5_addr_defined(sect->addr));
    assert(sect->size);

    
    if (H5FS__sinfo_lock(f, fspace, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get section info");
    sinfo_valid   = true;
    saved_fs_size = sect->size;

    
    if (H5FS__sect_merge(fspace, &sect, op_data) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTMERGE, FAIL, "can't merge sections");

    
    if (!sect) {
        sinfo_modified = true;
        HGOTO_DONE(true);
    } 
    else {
        
        if (sect->size != saved_fs_size) {
            if (H5FS__sect_link(fspace, sect, flags) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL,
                            "can't insert free space section into skip list");
            sinfo_modified = true;
            HGOTO_DONE(true);
        } 
    }     

done:
    
    if (sinfo_valid && H5FS__sinfo_unlock(f, fspace, sinfo_modified) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL, "can't release section info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5FS__sect_find_node(H5FS_t *fspace, hsize_t request, H5FS_section_info_t **node)
{
    H5FS_node_t *fspace_node;       
    unsigned     bin;               
    htri_t       ret_value = false; 

    H5SL_node_t                *curr_size_node = NULL;
    const H5FS_section_class_t *cls; 
    hsize_t                     alignment;

    FUNC_ENTER_PACKAGE

    
    assert(fspace);
    assert(fspace->sinfo);
    assert(fspace->sinfo->bins);
    assert(request > 0);
    assert(node);

    
    bin = H5VM_log2_gen(request);
    assert(bin < fspace->sinfo->nbins);
    alignment = fspace->alignment;
    if (!((alignment > 1) && (request >= fspace->align_thres)))
        alignment = 0; 

    do {
        
        if (fspace->sinfo->bins[bin].bin_list) {

            if (!alignment) { 
                
                
                
                if ((fspace_node =
                         (H5FS_node_t *)H5SL_greater(fspace->sinfo->bins[bin].bin_list, &request))) {
                    
                    if (NULL == (*node = (H5FS_section_info_t *)H5SL_remove_first(fspace_node->sect_list)))
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTREMOVE, FAIL,
                                    "can't remove free space node from skip list");

                    
                    cls = &fspace->sect_cls[(*node)->type];
                    
                    if (H5FS__size_node_decr(fspace->sinfo, bin, fspace_node, cls) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTREMOVE, FAIL,
                                    "can't remove free space size node from skip list");
                    if (H5FS__sect_unlink_rest(fspace, cls, *node) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL,
                                    "can't remove section from non-size tracking data structures");
                    
                    HGOTO_DONE(true);
                }  
            }      
            else { 
                
                curr_size_node = H5SL_first(fspace->sinfo->bins[bin].bin_list);
                while (curr_size_node != NULL) {
                    H5FS_node_t *curr_fspace_node = NULL;
                    H5SL_node_t *curr_sect_node   = NULL;

                    
                    curr_fspace_node = (H5FS_node_t *)H5SL_item(curr_size_node);

                    
                    curr_sect_node = (H5SL_node_t *)H5SL_first(curr_fspace_node->sect_list);

                    while (curr_sect_node != NULL) {
                        H5FS_section_info_t *curr_sect = NULL;
                        hsize_t              mis_align = 0, frag_size = 0;
                        H5FS_section_info_t *split_sect = NULL;

                        
                        curr_sect = (H5FS_section_info_t *)H5SL_item(curr_sect_node);

                        assert(H5_addr_defined(curr_sect->addr));
                        assert(curr_fspace_node->sect_size == curr_sect->size);

                        cls = &fspace->sect_cls[curr_sect->type];

                        assert(alignment);
                        assert(cls);

                        if ((mis_align = curr_sect->addr % alignment))
                            frag_size = alignment - mis_align;

                        if ((curr_sect->size >= (request + frag_size)) && (cls->split)) {
                            
                            if (NULL == (*node = (H5FS_section_info_t *)H5SL_remove(
                                             curr_fspace_node->sect_list, &curr_sect->addr)))
                                HGOTO_ERROR(H5E_FSPACE, H5E_CANTREMOVE, FAIL,
                                            "can't remove free space node from skip list");
                            
                            if (H5FS__size_node_decr(fspace->sinfo, bin, curr_fspace_node, cls) < 0)
                                HGOTO_ERROR(H5E_FSPACE, H5E_CANTREMOVE, FAIL,
                                            "can't remove free space size node from skip list");

                            if (H5FS__sect_unlink_rest(fspace, cls, *node) < 0)
                                HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL,
                                            "can't remove section from non-size tracking data structures");

                            
                            if (mis_align) {
                                split_sect = cls->split(*node, frag_size);
                                if ((H5FS__sect_link(fspace, split_sect, 0) < 0))
                                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL,
                                                "can't insert free space section into skip list");
                                
                                assert(split_sect->addr < (*node)->addr);
                                assert(request <= (*node)->size);
                            } 
                            
                            HGOTO_DONE(true);
                        } 

                        
                        curr_sect_node = H5SL_next(curr_sect_node);
                    } 

                    
                    curr_size_node = H5SL_next(curr_size_node);
                } 
            }     
        }         
        
        bin++;
    } while (bin < fspace->sinfo->nbins);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5FS_sect_find(H5F_t *f, H5FS_t *fspace, hsize_t request, H5FS_section_info_t **node)
{
    bool   sinfo_valid    = false; 
    bool   sinfo_modified = false; 
    htri_t ret_value      = false; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fspace);
    assert(fspace->nclasses);
    assert(request);
    assert(node);

    
    if (fspace->tot_sect_count > 0) {
        
        if (H5FS__sinfo_lock(f, fspace, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get section info");
        sinfo_valid = true;

        
        if ((ret_value = H5FS__sect_find_node(fspace, request, node)) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "can't remove section from bins");

        
        if (ret_value > 0) {
            
            sinfo_modified = true;
        } 
    }     

done:
    
    if (sinfo_valid && H5FS__sinfo_unlock(f, fspace, sinfo_modified) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL, "can't release section info");

#ifdef H5FS_DEBUG_ASSERT
    H5FS__assert(fspace);
#endif 
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__iterate_sect_cb(void *_item, void H5_ATTR_UNUSED *key, void *_udata)
{
    H5FS_section_info_t *sect_info = (H5FS_section_info_t *)_item; 
    H5FS_iter_ud_t      *udata     = (H5FS_iter_ud_t *)_udata;     
    herr_t               ret_value = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    
    assert(sect_info);
    assert(udata->fspace);
    assert(udata->op);

    
    if ((*udata->op)(sect_info, udata->op_data) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_BADITER, FAIL, "iteration callback failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__iterate_node_cb(void *_item, void H5_ATTR_UNUSED *key, void *_udata)
{
    H5FS_node_t    *fspace_node = (H5FS_node_t *)_item;     
    H5FS_iter_ud_t *udata       = (H5FS_iter_ud_t *)_udata; 
    herr_t          ret_value   = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    assert(fspace_node);
    assert(udata->fspace);
    assert(udata->op);

    
    assert(fspace_node->sect_list);
    if (H5SL_iterate(fspace_node->sect_list, H5FS__iterate_sect_cb, udata) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_BADITER, FAIL, "can't iterate over section nodes");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_sect_iterate(H5F_t *f, H5FS_t *fspace, H5FS_operator_t op, void *op_data)
{
    H5FS_iter_ud_t udata;                 
    bool           sinfo_valid = false;   
    herr_t         ret_value   = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(fspace);
    assert(op);

    
    udata.fspace  = fspace;
    udata.op      = op;
    udata.op_data = op_data;

    
    if (fspace->tot_sect_count) {
        unsigned bin; 

        
        if (H5FS__sinfo_lock(f, fspace, H5AC__READ_ONLY_FLAG) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get section info");
        sinfo_valid = true;

        
        for (bin = 0; bin < fspace->sinfo->nbins; bin++) {
            
            if (fspace->sinfo->bins[bin].bin_list) {
                
                if (H5SL_iterate(fspace->sinfo->bins[bin].bin_list, H5FS__iterate_node_cb, &udata) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_BADITER, FAIL, "can't iterate over section size nodes");
            } 
        }     
    }         

done:
    
    if (sinfo_valid && H5FS__sinfo_unlock(f, fspace, false) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL, "can't release section info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_sect_stats(const H5FS_t *fspace, hsize_t *tot_space, hsize_t *nsects)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fspace);

    
    if (tot_space)
        *tot_space = fspace->tot_space;
    if (nsects)
        *nsects = fspace->tot_sect_count;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5FS_sect_change_class(H5F_t *f, H5FS_t *fspace, H5FS_section_info_t *sect, uint16_t new_class)
{
    const H5FS_section_class_t *old_cls;               
    const H5FS_section_class_t *new_cls;               
    unsigned                    old_class;             
    bool                        sinfo_valid = false;   
    herr_t                      ret_value   = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(fspace);
    assert(sect);
    assert(sect->type < fspace->nclasses);
    assert(new_class < fspace->nclasses);

    
    if (H5FS__sinfo_lock(f, fspace, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get section info");
    sinfo_valid = true;

    
    old_class = sect->type;
    old_cls   = &fspace->sect_cls[sect->type];
    new_cls   = &fspace->sect_cls[new_class];

    
    if ((old_cls->flags & H5FS_CLS_GHOST_OBJ) != (new_cls->flags & H5FS_CLS_GHOST_OBJ)) {
        H5FS_node_t *fspace_node; 
        unsigned     bin;         
        bool         to_ghost;    

        
        if (old_cls->flags & H5FS_CLS_GHOST_OBJ)
            to_ghost = false;
        else
            to_ghost = true;

        
        assert(fspace->sinfo->bins);

        
        bin = H5VM_log2_gen(sect->size);
        assert(bin < fspace->sinfo->nbins);
        assert(fspace->sinfo->bins[bin].bin_list);

        
        fspace_node = (H5FS_node_t *)H5SL_search(fspace->sinfo->bins[bin].bin_list, &sect->size);
        assert(fspace_node);

        
        if (to_ghost) {
            
            fspace->serial_sect_count--;
            fspace->ghost_sect_count++;

            
            fspace->sinfo->bins[bin].serial_sect_count--;
            fspace->sinfo->bins[bin].ghost_sect_count++;

            
            fspace_node->serial_count--;
            fspace_node->ghost_count++;

            
            if (fspace_node->serial_count == 0)
                fspace->sinfo->serial_size_count--;
            if (fspace_node->ghost_count == 1)
                fspace->sinfo->ghost_size_count++;
        } 
        else {
            
            fspace->serial_sect_count++;
            fspace->ghost_sect_count--;

            
            fspace->sinfo->bins[bin].serial_sect_count++;
            fspace->sinfo->bins[bin].ghost_sect_count--;

            
            fspace_node->serial_count++;
            fspace_node->ghost_count--;

            
            if (fspace_node->serial_count == 1)
                fspace->sinfo->serial_size_count++;
            if (fspace_node->ghost_count == 0)
                fspace->sinfo->ghost_size_count--;
        } 
    }     

    
    if ((old_cls->flags & H5FS_CLS_SEPAR_OBJ) != (new_cls->flags & H5FS_CLS_SEPAR_OBJ)) {
        bool to_mergable; 

        
        if (old_cls->flags & H5FS_CLS_SEPAR_OBJ)
            to_mergable = true;
        else
            to_mergable = false;

        
        if (to_mergable) {
            if (fspace->sinfo->merge_list == NULL)
                if (NULL == (fspace->sinfo->merge_list = H5SL_create(H5SL_TYPE_HADDR, NULL)))
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTCREATE, FAIL,
                                "can't create skip list for merging free space sections");
            if (H5SL_insert(fspace->sinfo->merge_list, sect, &sect->addr) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL,
                            "can't insert free space node into merging skip list");
        } 
        else {
            H5FS_section_info_t *tmp_sect_node; 

            tmp_sect_node = (H5FS_section_info_t *)H5SL_remove(fspace->sinfo->merge_list, &sect->addr);
            if (tmp_sect_node == NULL || tmp_sect_node != sect)
                HGOTO_ERROR(H5E_FSPACE, H5E_NOTFOUND, FAIL, "can't find section node on size list");
        } 
    }     

    
    sect->type = new_class;

    
    fspace->sinfo->serial_size -= fspace->sect_cls[old_class].serial_size;
    fspace->sinfo->serial_size += fspace->sect_cls[new_class].serial_size;

    
    if (H5FS__sect_serialize_size(fspace) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTCOMPUTE, FAIL, "can't adjust free space section size on disk");

done:
    
    if (sinfo_valid && H5FS__sinfo_unlock(f, fspace, true) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL, "can't release section info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5FS_DEBUG_ASSERT

void
H5FS__sect_assert(const H5FS_t *fspace)
{
    hsize_t separate_obj; 

    FUNC_ENTER_PACKAGE_NOERR

    
    separate_obj = 0;

    
    if (fspace->sinfo->bins) {
        hsize_t  acc_tot_sect_count;    
        hsize_t  acc_serial_sect_count; 
        hsize_t  acc_ghost_sect_count;  
        size_t   acc_tot_size_count;    
        size_t   acc_serial_size_count; 
        size_t   acc_ghost_size_count;  
        unsigned u;                     

        
        acc_tot_sect_count    = 0;
        acc_serial_sect_count = 0;
        acc_ghost_sect_count  = 0;
        acc_tot_size_count    = 0;
        acc_serial_size_count = 0;
        acc_ghost_size_count  = 0;
        for (u = 0; u < fspace->sinfo->nbins; u++) {
            acc_tot_sect_count += fspace->sinfo->bins[u].tot_sect_count;
            acc_serial_sect_count += fspace->sinfo->bins[u].serial_sect_count;
            acc_ghost_sect_count += fspace->sinfo->bins[u].ghost_sect_count;
            if (fspace->sinfo->bins[u].bin_list) {
                H5SL_node_t *curr_size_node;   
                size_t       bin_serial_count; 
                size_t       bin_ghost_count;  

                acc_tot_size_count += H5SL_count(fspace->sinfo->bins[u].bin_list);

                
                curr_size_node   = H5SL_first(fspace->sinfo->bins[u].bin_list);
                bin_serial_count = 0;
                bin_ghost_count  = 0;
                while (curr_size_node != NULL) {
                    H5FS_node_t *fspace_node;       
                    H5SL_node_t *curr_sect_node;    
                    size_t       size_serial_count; 
                    size_t       size_ghost_count;  

                    
                    fspace_node = (H5FS_node_t *)H5SL_item(curr_size_node);

                    
                    curr_sect_node    = H5SL_first(fspace_node->sect_list);
                    size_serial_count = 0;
                    size_ghost_count  = 0;
                    while (curr_sect_node != NULL) {
                        H5FS_section_class_t *cls;  
                        H5FS_section_info_t  *sect; 

                        
                        sect = (H5FS_section_info_t *)H5SL_item(curr_sect_node);
                        cls  = &fspace->sect_cls[sect->type];

                        
                        assert(H5_addr_defined(sect->addr));
                        assert(fspace_node->sect_size == sect->size);
                        if (cls->valid)
                            (*cls->valid)(cls, sect);

                        
                        if (cls->flags & H5FS_CLS_GHOST_OBJ)
                            size_ghost_count++;
                        else
                            size_serial_count++;

                        
                        if (cls->flags & H5FS_CLS_SEPAR_OBJ)
                            separate_obj++;

                        
                        curr_sect_node = H5SL_next(curr_sect_node);
                    } 

                    
                    assert(fspace_node->serial_count == size_serial_count);
                    assert(fspace_node->ghost_count == size_ghost_count);

                    
                    if (fspace_node->serial_count > 0)
                        acc_serial_size_count++;
                    if (fspace_node->ghost_count > 0)
                        acc_ghost_size_count++;

                    
                    bin_serial_count += size_serial_count;
                    bin_ghost_count += size_ghost_count;

                    
                    curr_size_node = H5SL_next(curr_size_node);
                } 

                
                assert(fspace->sinfo->bins[u].tot_sect_count == (bin_serial_count + bin_ghost_count));
                assert(fspace->sinfo->bins[u].serial_sect_count == bin_serial_count);
                assert(fspace->sinfo->bins[u].ghost_sect_count == bin_ghost_count);
            } 
        }     

        
        assert(fspace->sinfo->tot_size_count == acc_tot_size_count);
        assert(fspace->sinfo->serial_size_count == acc_serial_size_count);
        assert(fspace->sinfo->ghost_size_count == acc_ghost_size_count);
        assert(fspace->tot_sect_count == acc_tot_sect_count);
        assert(fspace->serial_sect_count == acc_serial_sect_count);
        assert(fspace->ghost_sect_count == acc_ghost_sect_count);
    } 
    else {
        
        assert(fspace->tot_sect_count == 0);
        assert(fspace->serial_sect_count == 0);
        assert(fspace->ghost_sect_count == 0);
    } 

    
    if (fspace->sinfo->merge_list)
        assert(fspace->tot_sect_count == (separate_obj + H5SL_count(fspace->sinfo->merge_list)));

    FUNC_LEAVE_NOAPI_VOID
} 
#endif 

htri_t
H5FS_sect_try_shrink_eoa(H5F_t *f, H5FS_t *fspace, void *op_data)
{
    bool   sinfo_valid     = false; 
    bool   section_removed = false; 
    htri_t ret_value       = false; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fspace);

    if (H5FS__sinfo_lock(f, fspace, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get section info");
    sinfo_valid = true;

    if (fspace->sinfo && fspace->sinfo->merge_list) {
        H5SL_node_t *last_node; 

        
        if (NULL != (last_node = H5SL_last(fspace->sinfo->merge_list))) {
            H5FS_section_info_t  *tmp_sect;     
            H5FS_section_class_t *tmp_sect_cls; 

            
            tmp_sect = (H5FS_section_info_t *)H5SL_item(last_node);
            assert(tmp_sect);
            tmp_sect_cls = &fspace->sect_cls[tmp_sect->type];
            if (tmp_sect_cls->can_shrink) {
                
                if ((ret_value = (*tmp_sect_cls->can_shrink)(tmp_sect, op_data)) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTSHRINK, FAIL, "can't check for shrinking container");
                if (ret_value > 0) {
                    assert(tmp_sect_cls->shrink);

                    
                    if (H5FS__sect_remove_real(fspace, tmp_sect) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL,
                                    "can't remove section from internal data structures");
                    section_removed = true;

                    
                    if ((*tmp_sect_cls->shrink)(&tmp_sect, op_data) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, FAIL, "can't shrink free space container");
                } 
            }     
        }         
    }             

done:
    
    if (sinfo_valid && H5FS__sinfo_unlock(f, fspace, section_removed) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_CANTRELEASE, FAIL, "can't release section info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_vfd_alloc_hdr_and_section_info_if_needed(H5F_t *f, H5FS_t *fspace, haddr_t *fs_addr_ptr)
{
    hsize_t hdr_alloc_size;
    hsize_t sinfo_alloc_size;
    haddr_t sect_addr = HADDR_UNDEF; 
    haddr_t eoa       = HADDR_UNDEF; 
    herr_t  ret_value = SUCCEED;     

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(f);
    assert(f->shared);
    assert(f->shared->lf);
    assert(fspace);
    assert(fs_addr_ptr);

    
    assert(fspace->sinfo_lock_count == 0);

    
    assert(f->shared->fs_persist);

    
    
    assert((f->shared->fs_strategy == H5F_FSPACE_STRATEGY_FSM_AGGR) ||
           (f->shared->fs_strategy == H5F_FSPACE_STRATEGY_PAGE));

    if (fspace->serial_sect_count > 0 && fspace->sinfo) {
        

        if (!H5_addr_defined(fspace->addr)) {

            

            
            if (HADDR_UNDEF == (eoa = H5F_get_eoa(f, H5FD_MEM_FSPACE_HDR)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "Unable to get eoa");

            
            if (H5F_IS_TMP_ADDR(f, (eoa + fspace->sect_size)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_BADRANGE, FAIL,
                            "hdr file space alloc will overlap into 'temporary' file space");

            hdr_alloc_size = H5FS_HEADER_SIZE(f);

            if (H5F_PAGED_AGGR(f))
                assert(0 == (eoa % f->shared->fs_page_size));

            
            if (HADDR_UNDEF == (fspace->addr = H5MF_alloc(f, H5FD_MEM_FSPACE_HDR, hdr_alloc_size)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "file allocation failed for free space header");

            
            if (H5AC_insert_entry(f, H5AC_FSPACE_HDR, fspace->addr, fspace, H5AC__PIN_ENTRY_FLAG) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTINIT, FAIL, "can't add free space header to cache");

            *fs_addr_ptr = fspace->addr;
        }

        if (!H5_addr_defined(fspace->sect_addr)) {

            

            
            if (HADDR_UNDEF == (eoa = H5F_get_eoa(f, H5FD_MEM_FSPACE_SINFO)))
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "Unable to get eoa");

            
            if (H5F_IS_TMP_ADDR(f, (eoa + fspace->sect_size)))
                HGOTO_ERROR(H5E_FSPACE, H5E_BADRANGE, FAIL,
                            "sinfo file space alloc will overlap into 'temporary' file space");

            sinfo_alloc_size = fspace->sect_size;

            if (H5F_PAGED_AGGR(f))
                assert(0 == (eoa % f->shared->fs_page_size));

            
            if (HADDR_UNDEF == (sect_addr = H5MF_alloc(f, H5FD_MEM_FSPACE_SINFO, sinfo_alloc_size)))
                HGOTO_ERROR(H5E_FSPACE, H5E_NOSPACE, FAIL, "file allocation failed for section info");

            
            if (fspace->sect_size > sinfo_alloc_size) {
                hsize_t saved_sect_size = fspace->sect_size;

                if (H5MF_xfree(f, H5FD_MEM_FSPACE_SINFO, sect_addr, sinfo_alloc_size) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "unable to free free space sections");
                fspace->sect_size = saved_sect_size;
            }
            else {
                fspace->alloc_sect_size = sinfo_alloc_size;
                fspace->sect_size       = sinfo_alloc_size;
                fspace->sect_addr       = sect_addr;

                

                
                if (H5AC_insert_entry(f, H5AC_FSPACE_SINFO, sect_addr, fspace->sinfo, H5AC__NO_FLAGS_SET) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTINIT, FAIL, "can't add free space sinfo to cache");

                
                if (H5AC_mark_entry_dirty(fspace) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL,
                                "unable to mark free space header as dirty");

                
                fspace->sinfo = NULL;
            }
        }
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
