% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GIRF.R, R/plotMethods.R, R/printMethods.R
\name{GIRF}
\alias{GIRF}
\alias{plot.girf}
\alias{print.girf}
\title{Estimate generalized impulse response function for a structural GMVAR model.}
\usage{
GIRF(
  gmvar,
  variables,
  shock_size,
  N = 30,
  R1 = 250,
  R2 = 250,
  init_regimes = 1:M,
  init_values = NULL,
  ci = c(0.95, 0.8),
  include_mixweights = TRUE,
  ncores = min(2, parallel::detectCores()),
  seeds = NULL
)

\method{plot}{girf}(x, add_grid = FALSE, ...)

\method{print}{girf}(x, ..., digits = 2, N_to_print)
}
\arguments{
\item{gmvar}{an object of class \code{'gmvar'} created with \code{fitGMVAR} or \code{GMVAR}.}

\item{variables}{a numeric vector of length at most \eqn{d} (\code{=ncol(data)})
and elements in \eqn{1,...,d} specifying the variables for which the GIRF
should be estimated.}

\item{shock_size}{a vector with the same length as \code{variables} specifying
the size of the structural shock for each variable. By default, the size of one standard
deviation is used, calculated as the weighted average of the component process error term
standard deviations with weights given by the mixing weight parameters.}

\item{N}{a positive integer specifying the horizon how far ahead should the generalized
impulse responses be calculated?}

\item{R1}{the number of repetitions used to estimate GIRF for each initial value?}

\item{R2}{the number of initial values to be drawn from a stationary distribution of the process
or of a specific regime? The confidence bounds will be sample quantiles of the GIRFs based on
different initial values. Ignored if the argument \code{init_value} is specified.}

\item{init_regimes}{a numeric vector of length at most \eqn{M} and elements in \eqn{1,...,M}
specifying the regimes from which the initial values should be generated from. The initial
values will be generated from a mixture distribution with the mixture components being the
stationary distirbutions of the specific regimes and the (proportiional) mixing weights given
by the mixing weight parameters of those regimes. Note that if \code{init_regimes=1:M}, the
initial values are generated from the stationary distribution of the process and if
\code{init_regimes=m}, the initial value are generated from the stationary distribution
of the \eqn{m}th regime. Ignored if \code{init_value} is specified.}

\item{init_values}{a matrix or a multivariate class \code{'ts'} object with \eqn{d} columns
and at least \eqn{p} rows specifying an initial value for the GIRF. The last \eqn{p} rows
are taken to be the initial value assuming that the \strong{last} row is the most recent observation.}

\item{ci}{a numeric vector with elements in \eqn{(0, 1)} specifying the confidence levels of the
confidence intervals.}

\item{include_mixweights}{should the generalized impulse response be calculated for the mixing weights
as well? \code{TRUE} or \code{FALSE}.}

\item{ncores}{the number CPU cores to be used in parallel computing. Only single core computing is
supported if an initial value is specified (and the GIRF won't thus be estimated multiple times).}

\item{seeds}{a length \code{R2} vector containing the random number generator seed for estimation
 of each GIRF. A single number of an initial value is specified.
or \code{NULL} for not initializing the seed. Exists for creating reproducible results.}

\item{x}{object of class \code{'girf'} generated by the function \code{GIRF}.}

\item{add_grid}{should grid be added to the plots?}

\item{...}{arguments passed to \code{grid} which plots grid to the figure.}

\item{digits}{the number of decimals to print}

\item{N_to_print}{an integer specifying the horizon how far to print the estimates and
confidence intervals. The default is that all the values are printed (\code{N_to_print=N}).}
}
\value{
Returns a class \code{'girf'} list with the GIRFs in the first element (\code{$girf_res}) and the used
  arguments the rest. The first element containing the GIRFs is a list with the \eqn{m}th element containing
  the point estimates for the GIRF in \code{$point_est} (the first element) and confidence intervals in
  \code{$conf_ints} (the second element). The first row is for the GIRF at impact \eqn{(n=0)}, the second for
  \eqn{n=1}, the third for \eqn{n=2}, and so on.
}
\description{
\code{GIRF} estimate generalized impulse response function for
 a structural GMVAR model.
}
\details{
The model needs to be structural in order for this function to be applicable. A structural
  GMVAR model can be estimated by specifying the argument \code{structural_pars} in the function \code{fitGMVAR}.

  The confidence bounds reflect uncertainty about the initial state (but currently not about the parameter
  estimates) if initial values are not specified. If initial values are specified, there won't currently
  be confidence intervals. See the cited paper by Virolainen (2020) for details about the algorithm.
}
\section{Methods (by generic)}{
\itemize{
\item \code{plot}: plot method

\item \code{print}: print method
}}

\examples{
 \donttest{
 # These are long-running examples that use parallel computing.
 data(eurusd, package="gmvarkit")
 data <- cbind(10*eurusd[,1], 100*eurusd[,2])
 colnames(data) <- colnames(eurusd)

 # Structural GMVAR(2, 2), d=2 model identified with sign-constraints:
 params222s <- c(1.428, -0.808, 1.029, 5.84, 1.314, 0.145, 0.094, 1.292,
   -0.389, -0.07, -0.109, -0.281, 1.248, 0.077, -0.04, 1.266, -0.272,
   -0.074, 0.034, -0.313, 0.903, 0.718, -0.324, 2.079, 7, 1.44, 0.742)
 W_222 <- matrix(c(1, NA, -1, 1), nrow=2, byrow=FALSE)
 mod222s <- GMVAR(data, p=2, M=2, params=params222s, structural_pars=list(W=W_222))
 mod222s
 # Alternatively, use:
 # fit222s <- fitGMVAR(data, p=2, M=2, structural_pars=list(W=W_222),
 #                     ncalls=20, seeds=1:20)
 # To obtain an estimated version of the same model.

 # Estimating the GIRFs of both variables with default arguments
 # (initial values are drawn from the stationary distribution of the process,
 # 30 periods ahead, confidence levels 0.95 and 0.8):
 girf1 <- GIRF(mod222s)
 plot(girf1)
 girf1

 # Estimating the GIRF of the second variable only, 36 periods ahead
 # and shock size 1, initial values drawn from the stationary distribution
 # of the first regime, confidence level 0.9:
 girf2 <- GIRF(mod222s, variables=2, shock_size=1, N=36, init_regimes=1, ci=0.9)
 plot(girf2)

 # Estimating the GIRFs of both variables, shock sizes 1 and 3, N=50 periods ahead,
 # estimation based on 1000 Monte Carlo simulations, and fixed initial values given
 # by the last p observations of the data:
 girf3 <- GIRF(mod222s, shock_size=c(1, 3), N=50, R1=1000, init_values=mod222s$data)
 }
}
\references{
\itemize{
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2016. Gaussian mixture vector autoregression.
         \emph{Journal of Econometrics}, \strong{192}, 485-498.
   \item Virolainen S. 2020. Structural Gaussian mixture vector autoregressive model. Unpublished working
     paper, available as arXiv:2007.04713.
 }
}
\seealso{
\code{\link{fitGMVAR}}, \code{\link{GMVAR}}, \code{\link{gmvar_to_sgmvar}}, \code{\link{reorder_W_columns}},
 \code{\link{swap_W_signs}}, \code{\link{simulateGMVAR}}, \code{\link{predict.gmvar}},
 \code{\link{profile_logliks}}, \code{\link{quantile_residual_tests}}, \code{\link{LR_test}}, \code{\link{Wald_test}}
}
