## Aims
#
# 1. Reproduce upsetplot
# 2. Expand the function of upsetplot
#   2.1 Allow non-unique intersections
#   2.2 More applicable subplot types (boxplot, points), useful to show subsets properties


## Structures
#
# 1. plot and subplots
# 2. themes
# 3. subsets processing [see regions.R]



## (PART) Plot

#' Plot a upset plot
#'
#' This function generate a upset plot by creating a composite plot which contains subplots generated by ggplot2.
#'
#' @param venn a class Venn object
#' @param nintersects number of intersects. If NULL, all intersections will show.
#' @param order.intersect.by one of 'size' or 'name'
#' @param order.set.by one of 'size' or 'name'
#' @param relative_height the relative height of top panel in upset plot
#' @param relative_width the relative width of left panel in upset plot
#' @return an upset plot
#'
#' @export
#'
#' @examples
#'  list = list(A = sample(LETTERS, 20),
#'              B = sample(LETTERS, 22),
#'              C = sample(LETTERS, 14),
#'              D = sample(LETTERS, 30, replace = TRUE))
#'  venn = Venn(list)
#'  plot_upset(venn)
#'  plot_upset(venn, order.intersect.by = "name")
#'  plot_upset(venn, nintersects = 6)
plot_upset = function(venn,
                      nintersects = NULL,
                      order.intersect.by = c("size", "name"),
                      order.set.by = c("size", "name"),
                      relative_height = 3,
                      relative_width = 0.2){
  # process arguments
  order.intersect.by = match.arg(order.intersect.by)
  order.set.by = match.arg(order.set.by)

  # subplot main
  data = process_upset_data(venn,
                           nintersects = nintersects,
                           order.intersect.by = order.intersect.by,
                           order.set.by = order.set.by)
  p_main = upsetplot_main(data$main_data)

  # subplot top
  p_top = upsetplot_top(data$top_data)

  # subplot left
  p_left = upsetplot_left(data$left_data)

  # combine into a plot
  pp = aplot::insert_top(p_main, p_top, height = relative_height) |>
    aplot::insert_left(p_left, width = relative_width)
  class(pp) = c("upset_plot", class(pp))

  return(pp)
}

upsetplot_main = function(data){
  ggplot2::ggplot(data, aes(.data$id, .data$set)) +
    ggplot2::geom_point(size = 4, color = "grey30", na.rm = FALSE) +
    ggplot2::geom_path(aes(group = .data$id), size = 1.5, color = "grey30", na.rm = FALSE) +
    ggplot2::labs(x = "Set Intersection", y = NULL) +
    theme_upset_main()
}

upsetplot_top = function(data){
  ggplot2::ggplot(data, aes(.data$id, .data$size)) +
    ggplot2::geom_col() +
    ggplot2::labs(x = NULL, y = "Intersection Size") +
    theme_upset_top()
}

upsetplot_left = function(data){
  ggplot2::ggplot(data, aes(x = .data$size, y = .data$set)) +
    ggplot2::geom_col(orientation = "y") +
    ggplot2::scale_y_discrete(position = "right") +
    ggplot2::scale_x_reverse() +
    ggplot2::labs(x = "Set Size", y = NULL) +
    theme_upset_left()
}

## (PART) Theme

theme_upset_main = function(){
  ggplot2::theme_bw() +
    ggplot2::theme(
      axis.title.y = element_blank(),
      axis.ticks.x.bottom = element_blank(),
      axis.text.x.bottom = element_blank(),
      # panel.border = element_blank(),
      plot.margin = margin(t = -20)
    )
}

theme_upset_top = function(){
  ggplot2::theme_bw() +
    ggplot2::theme(
      axis.ticks.x.bottom = element_blank(),
      axis.text.x.bottom = element_blank(),
      # panel.border = element_blank(),
      plot.margin = margin(b = -20, unit = "pt")
    )
}

theme_upset_left = function(){
  ggplot2::theme_bw() +
    ggplot2::theme(
      axis.ticks.y = ggplot2::element_blank(),
      axis.title.y = ggplot2::element_blank(),
      axis.text.y = ggplot2::element_blank(),
      panel.border = ggplot2::element_blank(),
      panel.grid.major = ggplot2::element_blank(),
      plot.margin = margin(r = -20)
    )
}


## (PART) retrieve tidy data from primary subset datasets

##' @importFrom forcats as_factor
process_upset_data = function(venn,
                             nintersects = NULL,
                             order.intersect.by = "size",
                             order.set.by = "name",
                             name_separator = "/"){
  data = process_region_data(venn, sep = name_separator)
  data$size = data$count
  set_name = venn@names

  # top data
  top_data = data |>
    dplyr::select(c('id','name', 'item','size')) |>
    dplyr::mutate(id = forcats::fct_reorder(.data$id, .data[[order.intersect.by]], .desc = TRUE))

  # left data
  left_data = dplyr::tibble(set = set_name,
                            name = set_name,
                            size = sapply(venn@sets, length)) |>
    dplyr::mutate(set = forcats::fct_reorder(.data$set, .data[[order.set.by]], .desc = TRUE))

  # main data
  main_data = data |>
    dplyr::select(c("id", "name", "size")) |>
    dplyr::mutate(set = .data$id,
                  id = forcats::fct_reorder(.data$id, .data[[order.intersect.by]], .desc = TRUE)) |>
    tidyr::separate_longer_delim(.data$set, delim = name_separator)
  main_data$set = factor(set_name[as.integer(main_data$set)],
                         levels = levels(left_data$set))

  # filter intersections
  if (!is.null(nintersects)){
    keep_id = utils::head(levels(top_data$id), nintersects)
    main_data = main_data |> dplyr::filter(.data$id %in% keep_id)
    top_data = top_data |> dplyr::filter(.data$id %in% keep_id)
  }

  # return result as a list
  ret = list(top_data = top_data,
             left_data = left_data,
             main_data = main_data)
  class(ret) = 'upsetPlotData'
  return(ret)
}


#' S3 method for `upsetPlotData`
#'
#' @param x a upsetPlotData object
#' @param ... useless
#'
#' @method print upsetPlotData
#' @docType methods
#' @name print
#' @rdname print
#' @md
#' @export
print.upsetPlotData = function(x, ...){
  cat(sprintf("Class upsetPlotData.\n"))
  cat(sprintf("  Type: list; No. slots: %d; Slot names: %s.\n", length(x), paste(names(x), collapse = ", ")))
  cat(sprintf("  To view the data interactively, use 'obj[[\"slotName\"]]'.\n"))
}

