% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict-geolm_cmodMan.R
\name{predict.geolm_cmodMan}
\alias{predict.geolm_cmodMan}
\title{Predict method for geostatistical models}
\usage{
\method{predict}{geolm_cmodMan}(
  object,
  newdata,
  nsim = 0,
  vop,
  vp,
  return_type = "SpatialPointsDataFrame",
  dmethod = "chol",
  compute_mspe = TRUE,
  sp = NULL,
  ...
)
}
\arguments{
\item{object}{An object produced by the \code{geolm}
function.}

\item{newdata}{An optional data frame in which to look
for the coordinates at which to predict. If omitted,
the observed data locations are used.}

\item{nsim}{A non-negative integer indicating the number
of realizations to sample at the specified coordinates
using conditional simulation.}

\item{vop}{The cross-covariance matrix between the
observed responses and the responses to predict.}

\item{vp}{The covariance matrix of the responses to
predict.}

\item{return_type}{A character string indicating the type
of object that should be returned. The default is
\code{"\link[sp]{SpatialPointsDataFrame}"} for easy
plotting of results (see Examples). Other options
include \code{"\link[base]{data.frame}"},
\code{"gearPredict"}, and \code{"\link[sf]{sf}"}.}

\item{dmethod}{The method used to decompose the
covariance matrix for conditional simulation.  Valid
options are \code{"chol"}, \code{"eigen"}, and
\code{"svd"}.  The default is \code{"chol"}.}

\item{compute_mspe}{A logical value indicating whether
the mean square prediction error should be calculated.
Default is \code{TRUE}.}

\item{sp}{This argument will be deprecated in the future.
Please use the \code{return_type} argument. A logical
value indicating whether to object returned should be
of class \code{\link[sp]{SpatialPointsDataFrame}} for
easier plotting with the \code{sp} package.  Default is
\code{TRUE}.}

\item{...}{Currently unimplemented.}
}
\value{
A \code{data.frame},
  \code{SpatialPointsDataFrame}, \code{"gearPredict"}, or
  \code{sf} object with the kriging predictions
  \code{pred}, kriging variance/mean-square prediction
  error (\code{mspe}), the root mean-square prediction
  error \code{mspe} (\code{rmspe}), and the conditional
  simulation \code{sim.1}, \code{sim.2}, etc.
  \code{sim.1}, \code{sim.2}, etc.
}
\description{
\code{predict} calculates the predicted values at
specified locations.  The method can additionally provide
the mean square prediction error (mspe) and perform
conditional simulation.
}
\details{
The \code{newdata} data frame must include the relevant
covariates for the prediction locations, where the
covariates are specified on the right side of the
\code{~} in \code{object$formula}.  \code{newdata} must
also include the coordinates of the prediction locations,
with these columns having the names provided in
\code{object$coordnames}.
}
\examples{
# generate response
y = rnorm(10)
# generate coordinates
x1 = runif(10); x2 = runif(10)

# data frame for observed data
data = data.frame(y, x1, x2)
coords = cbind(x1, x2)
d = as.matrix(dist(coords))
psill = 2 # partial sill
r = 4 # range parameter
evar = .1 # error variance
fvar = .1 # add finescale variance
# one can't generally distinguish between evar and fvar, but
# this is done for illustration purposes

# manually specify an exponential covariance model
v = psill * exp(-d/r) + (evar + fvar) * diag(10)
mod_man = cmod_man(v = v, evar = evar)

# coordinate names
cnames = c("x1", "x2")

# geolm for universal kriging
gearmod_uk = geolm(y ~ x1 + x2, data = data, mod = mod_man,
                 coordnames = cnames)

# newdata must have columns with prediction coordinates
# add 5 unsampled sites to sampled sites
newdata = data.frame(x1 = c(x1, runif(5)), x2 = c(x2, runif(5)))
newcoords = newdata[, cnames]
# create vop and vp using distances
dop = sp::spDists(as.matrix(coords), as.matrix(newcoords))
dp = as.matrix(dist(newcoords))

# manually create cross-covariance and covariance for
# prediction locations
vop = psill * exp(-dop/r) + fvar * (dop == 0)
vp = psill * exp(-dp/r) + fvar * diag(nrow(newcoords))

# prediction for universal kriging, with conditional simulation,
# using manual covariance matrices
pred_uk_man = predict(gearmod_uk, newdata, nsim = 2,
                      vop = vop, vp = vp, dmethod = "svd")
# plot prediction values using sp package
sp::spplot(pred_uk_man, "pred")

# do the same thing, but using cmod_std

# prediction for universal kriging, with conditional simulation
mod_std = cmod_std("exponential", psill = psill, r = r,
                   evar = evar, fvar = fvar)
gearmod_uk2 = geolm(y ~ x1 + x2, data = data, mod = mod_std,
                    coordnames = c("x1", "x2"))
pred_uk_std = predict(gearmod_uk2, newdata, nsim = 2, dmethod = "svd")

# compare results
all.equal(pred_uk_man$pred, pred_uk_std$pred)
all.equal(pred_uk_man$mspe, pred_uk_std$mspe)
}
\author{
Joshua French
}
