% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eim-class.R
\name{eim}
\alias{eim}
\alias{eim()}
\title{S3 Object for the Expectation-Maximization Algorithm}
\usage{
eim(X = NULL, W = NULL, json_path = NULL)
}
\arguments{
\item{X}{A \verb{(b x c)} matrix representing candidate votes per ballot box.}

\item{W}{A \verb{(b x g)} matrix representing group votes per ballot box.}

\item{json_path}{A path to a JSON file containing \code{X} and \code{W} fields, stored as nested arrays. It may contain additional fields with other attributes, which will be added to the returned object.}
}
\value{
A list of class \code{eim} containing:
\describe{
\item{\code{X}}{The candidate votes matrix \code{(b x c)}.}
\item{\code{W}}{The group votes matrix \code{(b x g)}.}
}
}
\description{
This constructor creates an \code{eim} S3 object, either by using matrices
\code{X} and \code{W} directly or by reading them from a JSON file. Each
\code{eim} object encapsulates the data (votes for candidates and demographic groups) required by the underlying Expectation-Maximization algorithm.
}
\details{
If \code{X} and \code{W} are directly supplied, they must match the
dimensions of ballot boxes \code{(b)}. Alternatively, if \code{json_path} is provided, the function expects
the JSON file to contain elements named \code{"X"} and \code{"W"} under the
top-level object. This two approaches are \strong{mutually exclusable}, yielding an error otherwise.

Internally, this function also initializes the corresponding instance within
the low-level (C-based) API, ensuring the data is correctly registered for
further processing by the EM algorithm.
}
\note{
A way to generate synthetic data for \code{X} and \code{W} is by using the \link{simulate_election} function. See Example 2 below.
}
\section{Methods}{

In addition to this constructor, the "eim" class provides several
S3 methods for common operations. Some of these methods are fully documented,
while others are ommited due to its straightfoward implementantion. The available methods are:

\itemize{
\item \code{\link{run_em}} - Runs the EM algorithm.
\item \code{\link{bootstrap}} - Estimates the standard deviation.
\item \code{\link{save_eim}} - Saves the object to a file.
\item \code{\link{get_agg_proxy}} - Estimates an ideal group aggregation given their standard deviations.
\item \code{\link{get_agg_opt}} - Estimates an ideal group aggregation among all combinations, given the log-likelihood.
\item \code{print.eim} - Print info about the object.
\item \code{summary.eim} - Summarize the object.
\item \code{as.matrix.eim} - Returns the probability matrix.
\item \code{logLik.eim} - Returns the final log-likelihood.
}
}

\examples{

# Example 1: Create an eim object from a JSON file
\dontrun{
model1 <- eim(json_path = "path/to/file.json")
}

# Example 2: Use simulate_election with optional parameters, then create an eim object
# from matrices

# Simulate data for 500 ballot boxes, 4 candidates and 5 groups
sim_result <- simulate_election(
    num_ballots = 500,
    num_candidates = 3,
    num_groups = 5,
    group_proportions = c(0.2, 0.2, 0.4, 0.1, 0.1)
)

model2 <- eim(X = sim_result$X, W = sim_result$W)

# Example 3: Create an object from a user defined matrix with 8 ballot boxes,
# 2 candidates and 7 groups.

x_mat <- matrix(c(
    57, 90,
    60, 84,
    43, 102,
    72, 71,
    63, 94,
    52, 80,
    60, 72,
    54, 77
), nrow = 8, ncol = 2, byrow = TRUE)

w_mat <- matrix(c(
    10, 15, 25, 21, 10, 40, 26,
    11, 21, 37, 32, 8, 23, 12,
    17, 12, 43, 27, 12, 19, 15,
    20, 18, 25, 15, 22, 17, 26,
    21, 19, 27, 16, 23, 22, 29,
    18, 16, 20, 14, 19, 22, 23,
    10, 15, 21, 18, 20, 16, 32,
    12, 17, 19, 22, 15, 18, 28
), nrow = 8, ncol = 7, byrow = TRUE)

model3 <- eim(X = x_mat, W = w_mat)

}
