#' splitDateTime
#
#' @description splitDateTime splits a date-time object into its components date and time.
#
#' @param refOrEsDf a data.frame. Either the reference dataset or the event sampling raw dataset (already merged to a single dataset).
#
#' @param refOrEs a character string. Enter "REF" if the argument refOrEs is the reference dataset, enter "ES" if it is the event sampling dataset.
#
#' @param dateTimeFormat a character string. Choose the current date-time format, "ymd_HMS" (default), "mdy_HMS", or "dmy_HMS".
#
#' @param RELEVANTVN_ES a list. This list is generated by function \code{\link{setES}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @param RELEVANTINFO_ES a list. This list is generated by function \code{\link{setES}}.
#
#' @param RELEVANTVN_REF a list. This list is generated by function \code{\link{setREF}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @details Splitting up a date-time object means to separate it into a data-object, e.g. 2007-10-03 and a time-object, e.g. 12:00:00.
#
#' @return The dataset that was passed as first argument with four additional columns, i.e. the separate date and time objects of the combined date-time objects of both ESM start and ESM end. See \strong{Details} for more information.
#
#' @examples
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#' # Prerequisites in order to execute splitDateTime. Start ------------
#' # keyLsNew is delivered with the package. Remove the separate date
#' # and time for both start and end in each of the ESM datasets.
#' keyLsNewDT <- sapply(keyLsNew, function(x) {
#'      x <- x[,-match(c("start_date", "start_time",
#' "end_date", "end_time"), names(x))]
#'      return(x) } )
#' relEs <- relevantESVN(svyName="survey_name", IMEI="IMEI",
#' START_DATETIME="ES_START_DATETIME", END_DATETIME="ES_END_DATETIME")
#' imeiNumbers <- as.character(referenceDf$imei)
#' surveyNames <- c("morningTestGroup", "dayTestGroup", "eveningTestGroup",
#' "morningControlGroup", "dayControlGroup", "eveningControlGroup")
#' RELEVANT_ES <- setES(4, imeiNumbers, surveyNames, relEs)
#' RELEVANTINFO_ES <- RELEVANT_ES[["RELEVANTINFO_ES"]]
#' RELEVANTVN_ES <- RELEVANT_ES[["RELEVANTVN_ES"]]
#' # referenceDfNew is delivered with the package. Remove the separate
#' # date and time for both start and end.
#' referenceDfNewDT <- referenceDfNew[,-match(c("start_date", "start_time",
#' "end_date", "end_time"), names(referenceDfNew))]
#' relRef <- relevantREFVN(ID="id", IMEI="imei", ST="st",
#' START_DATETIME="REF_START_DATETIME", END_DATETIME="REF_END_DATETIME")
#' RELEVANTVN_REF <- setREF(4, relRef)
#' # Prerequisites in order to execute splitDateTime. End --------------
#' # ------------------------------------------------------
#' # Run function 7 of 28; see esmprep functions' hierarchy.
#' # ------------------------------------------------------
#' # Applying function to reference dataset (7a of 28)
#' referenceDfList <- splitDateTime(referenceDfNewDT, "REF", RELEVANTINFO_ES, RELEVANTVN_ES,
#' RELEVANTVN_REF)
#'
#' # Extract reference dataset from output
#' referenceDfNew <- referenceDfList[["refOrEsDf"]]
#' names(referenceDfNew)
#'
#' # Extract extended list of relevant variables names of reference dataset
#' RELEVANTVN_REF <- referenceDfList[["extendedVNList"]]
#'
#' # Applying function to raw ESM dataset(s) (7b of 28)
#' # keyLs is the result of function 'genKey'.
#' keyList <- splitDateTime(keyLsNewDT, "ES", RELEVANTINFO_ES, RELEVANTVN_ES,
#' RELEVANTVN_REF)
#'
#' # Extract list of raw ESM datasets from output
#' keyLsNew <- keyList[["refOrEsDf"]]
#'
#' # Extract extended list of relevant variables names of raw ESM datasets
#' RELEVANTVN_ES <- keyList[["extendedVNList"]]
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#
#' @seealso Exemplary code (fully executable) in the documentation of \code{\link{esmprep}} (function 26 of 28).\cr \code{splitDateTime} is the reverse function of \code{\link{genDateTime}}.
#
#' @export
#
splitDateTime <- function(refOrEsDf=NULL, refOrEs = NULL, RELEVANTINFO_ES = NULL, RELEVANTVN_ES = NULL, RELEVANTVN_REF = NULL, dateTimeFormat = "ymd_HMS") {

    if ( (refOrEs != "REF" & refOrEs != "ES") | !is.character(refOrEs)) {
        stop("Argument 'refOrEs' must read 'REF' (for reference data.frame) or 'ES' (for event sampling data.frame). Please check.")
    }


    # Error handling function for all set-up lists generated by setES and setREF.
    # Both lists RELEVANTVN_ES and RELEVANTVN_REF get extended either by function
    # genDateTime or by function splitDateTime!
    SETUPLISTCheck(RELEVANTINFO_ES=RELEVANTINFO_ES,
    			   RELEVANTVN_ES=RELEVANTVN_ES,
    			   RELEVANTVN_REF=RELEVANTVN_REF)


    if( (dateTimeFormat != "ymd_HMS" & dateTimeFormat != "mdy_HMS" & dateTimeFormat != "dmy_HMS") | !is.character(dateTimeFormat)) {
        stop("The date-format is invalid. Please choose from the 3 valid formats 'ymd', 'mdy', or 'dmy'. Type 'dateTimeFormats()' for help.")
    }

    if(refOrEs == "REF") {

        if(!is.data.frame(refOrEsDf)) {
            stop("The argument 'refOrEsDf' must be of type data.frame.")
        }

        dateTimeStartDf <- splitDateTimeSingle(refOrEsDf=refOrEsDf, refOrEs=refOrEs, dateTimeFormat=dateTimeFormat, RELEVANTVN_ES = RELEVANTVN_ES, RELEVANTVN_REF = RELEVANTVN_REF)

        RELEVANTVN_REF_Temp <- dateTimeStartDf[["extendedVariableNameList"]]

        dateTimeEndDf <- splitDateTimeSingle(refOrEsDf=refOrEsDf, refOrEs=refOrEs, dateTimeFormat=dateTimeFormat, startOrEnd="END", RELEVANTVN_ES = RELEVANTVN_ES, RELEVANTVN_REF = RELEVANTVN_REF_Temp)

        df_dateTime <- merge(dateTimeStartDf[["refOrEsDfSingle"]], dateTimeEndDf[["refOrEsDfSingle"]], all=TRUE)

        # Return the dataset
        list(refOrEsDf = df_dateTime, extendedVNList=dateTimeEndDf[["extendedVariableNameList"]])

    } else {

        # If event sampling is only one data frame, go on also checking whether it
        # is of tpye data.frame
        if(length(RELEVANTINFO_ES[["SVYNAMES"]]) == 1) {
            if(!is.data.frame(refOrEsDf)) {
                stop("The argument 'refOrEsDf' must be of type data.frame.")
            }
        } else {
            if(!is.list(refOrEsDf) & all(sapply(refOrEsDf, is.data.frame))) {
                stop("The argument 'refOrEsDf' must be of type list, each element of the list must be of type data.frame.")
            }
        }

        esList <- list()
        esListNames <- RELEVANTINFO_ES[["SVYNAMES"]]
        for(i in 1:length(refOrEsDf)) {
            dateTimeStartDf <- splitDateTimeSingle(refOrEsDf[[i]], refOrEs=refOrEs, dateTimeFormat=dateTimeFormat, RELEVANTVN_ES = RELEVANTVN_ES, RELEVANTVN_REF = RELEVANTVN_REF)

            RELEVANTVN_ES_Temp <- dateTimeStartDf[["extendedVariableNameList"]]

            dateTimeEndDf <- splitDateTimeSingle(refOrEsDf[[i]], refOrEs=refOrEs, dateTimeFormat=dateTimeFormat, startOrEnd="END", RELEVANTVN_ES = RELEVANTVN_ES_Temp, RELEVANTVN_REF = RELEVANTVN_REF)

            df_dateTime <- merge(dateTimeStartDf[["refOrEsDfSingle"]], dateTimeEndDf[["refOrEsDfSingle"]], all=TRUE)
            esList[[esListNames[i]]] <- df_dateTime
        }
        # esList
        list(refOrEsDf = esList, extendedVNList=dateTimeEndDf[["extendedVariableNameList"]])
    }

}
