#' computeTimeBetween
#
#' @description computeTimeBetween computes the duration between when an ESM questionnaire was started and when the subsequent one was finished, across all ESM questionnaires per person.
#
#' @param esDf a data.frame. A single ESM dataset. It must contain the 2 columns that hold the date-time object for when an ESM questionnaire was started and finished, respectively.
#
#' @param refDf a data.frame. The reference dataset.
#
#' @param RELEVANTVN_ES a list. This list is generated by function \code{\link{setES}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @param RELEVANTVN_REF a list. This list is generated by function \code{\link{setREF}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @return \code{esDf} with the additional column TBESQ, i.e. the T_ime B_etween E_xperience S_ampling Q_uestionnaires.
#
#' @importFrom lubridate as.interval ymd_hms as.period parse_date_time
#
#' @examples
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#' # Prerequisites in order to execute computeTimeBetween. Start -------
#' # Use example list delivered with the package
#' RELEVANTVN_ES <- RELEVANTVN_ESext
#' # Use example list delivered with the package
#' RELEVANTVN_REF <- RELEVANTVN_REFext
#' intoleranceDf <- data.frame(prompt = c(2, 3, 4, 1, 1),
#' expect = c(1, 1, 1, 2, 3))
#' # expectedDf is a raw ESM dataset, delivered with the package.
#' intolLs <- intolerable(expectedDf, intoleranceDf, RELEVANTINFO_ES)
#' randSelLs <- randomMultSelection(intolLs[["cleanedDf"]])
#' # Prerequisites in order to execute computeTimeBetween. End ---------
#' # -------------------------------------------------------
#' # Run function 27 of 28; see esmprep functions' hierarchy.
#' # -------------------------------------------------------
#' # randSelLs[["esRandSelIn"]] is the result of function 'randomMultSelection'.
#' tbsqDf <- computeTimeBetween(randSelLs[["esRandSelIn"]], referenceDfNew, RELEVANTVN_ES,
#' RELEVANTVN_REF)
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#
#' @seealso Exemplary code (fully executable) in the documentation of \code{\link{esmprep}} (function 27 of 28).
#
#' @export
#
computeTimeBetween <- function(esDf, refDf, RELEVANTVN_ES=NULL, RELEVANTVN_REF=NULL) {
	
	
	# Possible errors when passing arguments to the function -----------------------------
    if(!is.data.frame(esDf)) {
        stop("Argument 'esDf' must be of type data.frame.")
    }
	
	
	# Error handling function for all set-up lists generated by setES and setREF.
    # Both lists RELEVANTVN_ES and RELEVANTVN_REF get extended either by function
    # genDateTime or by function splitDateTime!
    SETUPLISTCheck(RELEVANTINFO_ES=NULL,
    			   RELEVANTVN_ES=RELEVANTVN_ES,
    			   RELEVANTVN_REF=RELEVANTVN_REF)
	
	
    if(any(is.na(match(c("ID",
                         RELEVANTVN_ES[["ES_START_DATETIME"]],
                         RELEVANTVN_ES[["ES_END_DATETIME"]]),
                       names(esDf))))) {
        stop(paste0("In order to compute the time lag the variables ",
                    "ID, ",
                    RELEVANTVN_ES[["ES_START_DATETIME"]]," and ",
                    RELEVANTVN_ES[["ES_END_DATETIME"]],
                    " must be part of the data.frame that is passed to this function."))
    }

    timeBetween <- c()
    for(i in 1:nrow(refDf)) {
        idx_i <- which(esDf[,"ID"] == refDf[i,RELEVANTVN_REF[["REF_ID"]]])

        if(length(idx_i) == 1) {

            dateTime_i1 <- "1970-01-01 00:00:01"
            timeBetween_i <- lubridate::as.period(lubridate::as.interval(lubridate::ymd_hms(dateTime_i1),
                                                                         lubridate::ymd_hms(dateTime_i1)))
            timeBetweenParse <-
                suppressWarnings(lubridate::parse_date_time(paste0(timeBetween_i@hour, ":",
                                                                   timeBetween_i@minute, ":",
                                                                   timeBetween_i@.Data), "%H%M%S"))

            timeBetween <- c(timeBetween, format(timeBetweenParse, "%H:%M:%S"))

        } else {

            timeBetweenDf_temp <- data.frame(timeEnd_temp=
                                             # 1st column consists of date-time the prior questionnaire has been ended
                                             c(as.character(esDf[idx_i[1],RELEVANTVN_ES[["ES_START_DATETIME"]]]),
                                               as.character(esDf[idx_i[-length(idx_i)], RELEVANTVN_ES[["ES_END_DATETIME"]]])),
                                             # 2nd column consists of date-time the subsequent questionnaire has been started
                                             timeStart_temp=as.character(esDf[idx_i,RELEVANTVN_ES[["ES_START_DATETIME"]]]))

            timeBetween_i <- lubridate::as.period(lubridate::as.interval(lubridate::ymd_hms(timeBetweenDf_temp[,1]),
                                                                         lubridate::ymd_hms(timeBetweenDf_temp[,2])))

            timeBetweenParse <-
                suppressWarnings(lubridate::parse_date_time(paste0(timeBetween_i@hour, ":",
                                                                   timeBetween_i@minute, ":",
                                                                   timeBetween_i@.Data), "%H%M%S"))

            timeBetween <- c(timeBetween, format(timeBetweenParse, "%H:%M:%S"))
        }
    }
    # T_ime that has passed b_etween s_ubsequent q_uestionnaires = tbsq
    esDf[,"TBSQ"] <- timeBetween

    return(esDf)
}
