#' @title Deconvoluting Wastewater Data to Incidence
#' @description Function estimates incidence from smoothed data
#'
#' @param d Data frame. Wastewater dataframe. Must include at least `date`, time `t` and
#'  `obs` columns.
#' @param fec List. Parameters for a single fecal shedding distribution, as generated by [sample_a_dist()].
#' @inheritParams estimate_R_ww
#' @template param-silent
#' @param RL.max.iter Integer. Maximum of iterations for the Richardson-Lucy deconvolution algorithm.
#'
#' @return Data frame with deconvoluted incidence
#'
#' @keywords internal
#' 
deconv_ww_inc <- function(d, fec, scaling.factor, silent, RL.max.iter){

  d$obs_scal = d$obs * scaling.factor

  start_date = as.Date(dplyr::first(d$date))

  f = get_discrete_dist(fec)

  inc = deconvolution_RL(observed = d$obs_scal,
                         times    = d$t,
                         p_delay  = f,
                         verbose  = !silent,
                         max_iter = RL.max.iter) |>
    # Forces incidence to be a positive integer:
    dplyr::mutate(
      inc.deconvol = as.integer(ifelse(
        RL_result>0, RL_result, 0))) |>
    # Retrieve corresponding dates:
    dplyr::rename(t = time) |>
    dplyr::filter(t > 0) |>
    dplyr::mutate(date = start_date + t)

  res = list(
    inc         = inc,
    ww.smooth   = d,
    fec         = fec,
    scaling.factor = scaling.factor
  )
  return(res)
}
