% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shiny.R
\name{ui_epoxy_mustache}
\alias{ui_epoxy_mustache}
\alias{ui_epoxy_whisker}
\title{Epoxy HTML Mustache Template}
\usage{
ui_epoxy_mustache(id, ..., .file = NULL, .sep = "", .container = "div")

ui_epoxy_whisker(id, ..., .file = NULL, .sep = "", .container = "div")
}
\arguments{
\item{id}{The ID of the output.}

\item{...}{Character strings of HTML or \link[htmltools:builder]{htmltools::tags}. All elements
should be unnamed.}

\item{.file}{A path to a template file. If provided, no other template lines
should be included in \code{...}.}

\item{.sep}{The separator used to concatenate elements in \code{...}.}

\item{.container}{A character tag name, e.g. \code{"div"} or \code{"span"}, or a
function that returns an \code{\link[htmltools:builder]{htmltools::tag()}}.}
}
\value{
Returns a Shiny output UI element.
}
\description{
A Shiny output that uses \href{https://mustache.github.io/}{mustache templating}
to render HTML. Mustache is a powerful template language with minimal
internal logic. The advantage of \code{ui_epoxy_mustache()} is that all parts of
the HTML can be templated -- including element attributes -- whereas
\code{\link[=ui_epoxy_html]{ui_epoxy_html()}} requires that the dynamic template variables appear in the
text portion of the UI. The downside is that the entire template is
re-rendered (in the browser), each time that updated data is sent from the
server -- unlike \code{\link[=ui_epoxy_html]{ui_epoxy_html()}}, whose updates are specific to the parts
of the data that have changed.
}
\section{Functions}{
\itemize{
\item \code{ui_epoxy_whisker()}: An alias for \code{ui_epoxy_mustache()}, provided
because R users are more familiar with this syntax via the \pkg{whisker}
package.

}}
\examples{
\dontshow{if (rlang::is_installed("shiny")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(shiny)

ui <- fluidPage(
  fluidRow(
    style = "max-width: 600px; margin: 0 auto",
    column(
      width = 6,
      ui_epoxy_mustache(
        id = "template",
        h2(class = "{{heading_class}}", "Hello, {{name}}!"),
        "{{#favorites}}",
        p("Your favorite fruits are..."),
        tags$ul(HTML("{{#fruits}}<li>{{.}}</li>{{/fruits}}")),
        "{{/favorites}}",
        "{{^favorites}}<p>Do you have any favorite fruits?</p>{{/favorites}}"
      )
    ),
    column(
      width = 6,
      h2("Inputs"),
      textInput("name", "Your name"),
      textInput("fruits", "Favorite fruits", placeholder = "apple, banana"),
      helpText("Enter a comma-separated list of fruits.")
    )
  )
)

server <- function(input, output, session) {
  user_name <- reactive({
    if (!nzchar(input$name)) return("user")
    input$name
  })

  favorites <- reactive({
    if (!nzchar(input$fruits)) return(NULL)
    list(fruits = strsplit(input$fruits, "\\\\s*,\\\\s*")[[1]])
  })

  output$template <- render_epoxy(
    name = user_name(),
    heading_class = if (user_name() != "user") "text-success",
    favorites = favorites()
  )
}

if (interactive()) {
  shiny::shinyApp(ui, server)
}
\dontshow{\}) # examplesIf}
\dontshow{if (rlang::is_interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
run_epoxy_example_app("ui_epoxy_mustache")
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=ui_epoxy_html]{ui_epoxy_html()}}, \code{\link[=render_epoxy]{render_epoxy()}}
}
