describe("epoxy_html()", {
	it("returns an html glue character", {
		expect_s3_class(
			epoxy_html("word"),
			c("html", "glue", "character")
		)

		expect_s3_class(
			epoxy_html("{{'word'}}"),
			c("html", "glue", "character")
		)
	})

	it("uses html, inline transformers by default", {
		expect_equal(
			epoxy_html("{{ span letters[1:3] }}"),
			html_chr(glue("<span>{x}</span>", x = letters[1:3]))
		)

		expect_equal(
			epoxy_html("{{ span {{ .and letters[1:3] }} }}"),
			html_chr(glue("<span>{and::and(letters[1:3])}</span>"))
		)

		# need to escape one level to access inline formatter
		expect_equal(
			epoxy_html("{{ {{.and letters[1:3] }} }}"),
			html_chr(glue(and::and(letters[1:3])))
		)

		# otherwise `.and` is interpreted as a class
		expect_equal(
			epoxy_html("{{.and letters[1] }}"),
			html_chr(glue('<span class="and">a</span>'))
		)
	})
})

describe("epoxy_transform_set()", {
	it("sets the default epoxy_transform for all engines", {
		opts <- epoxy_transform_set("bold")
		on.exit(options(opts))

		expect_equal(
			epoxy("{1} and {2} is {3}"),
			glue("**1** and **2** is **3**")
		)

		expect_equal(
			epoxy("{1} and {2} is {3}"),
			epoxy("{1} and {2} is {3}", .transformer = "bold")
		)

		expect_equal(
			epoxy_html("{{1}} and {{2}} is {{3}}"),
			html_chr(glue(
				"<strong>1</strong> and <strong>2</strong> is <strong>3</strong>"
			))
		)

		expect_equal(
			epoxy_latex("<1> and <2> is <3>"),
			glue("\\textbf{{1}} and \\textbf{{2}} is \\textbf{{3}}")
		)
	})

	it("sets inline formatters for all engines", {
		opts <- epoxy_transform_set(.bold = function(x) "PASS")
		on.exit(epoxy_transform_set(.bold = rlang::zap()))

		expect_equal(
			epoxy("{.bold 'hi'}"),
			glue("PASS")
		)

		expect_equal(
			epoxy_html("{{ {{.bold 'hi'}} }}"),
			glue("PASS")
		)

		expect_equal(
			epoxy_latex("<.bold 'hi'>"),
			glue("PASS")
		)
	})

	it("set, get, reset with NULL", {
		expect_equal(
			epoxy_transform_get(inline = TRUE),
			list(md = list(), html = list(), latex = list()),
			ignore_attr = TRUE
		)

		epoxy_transform_set(.bold = function(x) "PASS", engine = "md")
		epoxy_transform_set(.mold = function(x) "PASS", engine = "html")
		epoxy_transform_set(.fold = function(x) "PASS", engine = "latex")

		expect_equal(
			epoxy_transform_get(inline = TRUE),
			list(
				md = list(.bold = function(x) "PASS"),
				html = list(.mold = function(x) "PASS"),
				latex = list(.fold = function(x) "PASS")
			)
		)

		epoxy_transform_set(NULL)
		expect_equal(
			epoxy_transform_get(inline = TRUE),
			list(md = list(), html = list(), latex = list())
		)
	})

	it("accepts a spliced list", {
		opts_list <- list(.bold = function(x) "PASS", "inline", "code")
		opts <- epoxy_transform_set(!!!opts_list, engine = "md")
		on.exit({
			# ensure the global settings are reset
			options(epoxy.transformer_default.md = NULL)
			.globals$inline$md <- list(md = list())
		})

		expect_equal(
			epoxy("{.bold 'hi'}"),
			"`PASS`"
		)
	})

	it("sets the default for engine-specific epoxy_transform defaults", {
		opts_md <- epoxy_transform_set("bold", engine = "md")
		on.exit(options(opts_md), add = TRUE)

		opts_html <- epoxy_transform_set("italic", engine = "html")
		on.exit(options(opts_html), add = TRUE)

		opts_latex <- epoxy_transform_set(epoxy_transform_code, engine = "latex")
		on.exit(options(opts_latex), add = TRUE)

		expect_equal(
			epoxy("{1} and {2} is {3}"),
			glue("**1** and **2** is **3**")
		)

		expect_equal(
			epoxy_html("{{1}} and {{2}} is {{3}}"),
			html_chr(glue(
				"<em>1</em> and <em>2</em> is <em>3</em>"
			))
		)

		expect_equal(
			epoxy_latex("<1> and <2> is <3>"),
			glue("\\texttt{{1}} and \\texttt{{2}} is \\texttt{{3}}")
		)
	})
})
