% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dygis.R, R/dynatopGIS-package.R
\docType{package}
\name{dynatopGIS}
\alias{dynatopGIS}
\title{R6 Class for processing a catchment to make a Dynamic TOPMODEL}
\description{
This package contains the code for setting up a dynamic TOPMODEL implementation
}
\examples{
## The vignettes contains more examples of the method calls.

## create temport directory for output
demo_dir <- tempfile("dygis")
dir.create(demo_dir)

## initialise processing
ctch <- dynatopGIS$new(file.path(demo_dir,"meta.json"))

## add digital elevation and channel data
dem_file <- system.file("extdata", "SwindaleDTM40m.tif", package="dynatopGIS", mustWork = TRUE)
dem <- raster::raster(dem_file)
ctch$add_dem(dem)
channel_file <- system.file("extdata", "SwindaleRiverNetwork.shp",
package="dynatopGIS", mustWork = TRUE)
sp_lines <- rgdal::readOGR(channel_file)
property_names <- c(channel_id="identifier",endNode="endNode",startNode="startNode",length="length")
ctch$add_channel(sp_lines,property_names)

## compute properties 
ctch$compute_areas()
ctch$sink_fill() ## fill sinks in the catchment
\donttest{
ctch$compute_properties() # like topograpihc index and contour length
ctch$compute_flow_lengths()
}
## classify and create a model
\donttest{
ctch$classify("atb_20","atb",cuts=20) # classify using the topographic index
ctch$get_class_method("atb_20") ## see the details of the classification
ctch$combine_classes("atb_20_band",c("atb_20","band")) ## combine classes
ctch$create_model("new_model","atb_20_band","band") ## create a model
list.files(demo_dir,pattern="new_model*") ## look at the output files for the model
}
## tidy up
unlink(demo_dir)
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{dynatopGIS$new()}}
\item \href{#method-get_meta}{\code{dynatopGIS$get_meta()}}
\item \href{#method-get_working_directory}{\code{dynatopGIS$get_working_directory()}}
\item \href{#method-set_working_directory}{\code{dynatopGIS$set_working_directory()}}
\item \href{#method-add_dem}{\code{dynatopGIS$add_dem()}}
\item \href{#method-add_channel}{\code{dynatopGIS$add_channel()}}
\item \href{#method-add_layer}{\code{dynatopGIS$add_layer()}}
\item \href{#method-get_layer}{\code{dynatopGIS$get_layer()}}
\item \href{#method-plot_layer}{\code{dynatopGIS$plot_layer()}}
\item \href{#method-sink_fill}{\code{dynatopGIS$sink_fill()}}
\item \href{#method-compute_areas}{\code{dynatopGIS$compute_areas()}}
\item \href{#method-compute_properties}{\code{dynatopGIS$compute_properties()}}
\item \href{#method-compute_flow_lengths}{\code{dynatopGIS$compute_flow_lengths()}}
\item \href{#method-classify}{\code{dynatopGIS$classify()}}
\item \href{#method-combine_classes}{\code{dynatopGIS$combine_classes()}}
\item \href{#method-create_model}{\code{dynatopGIS$create_model()}}
\item \href{#method-get_version}{\code{dynatopGIS$get_version()}}
\item \href{#method-get_class_method}{\code{dynatopGIS$get_class_method()}}
\item \href{#method-clone}{\code{dynatopGIS$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Initialise a project, or reopen an existing project
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$new(meta_file, check = TRUE, verbose = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{meta_file}}{filename and path of the meta data file}

\item{\code{check}}{logical, should checks be performed [TRUE]}

\item{\code{verbose}}{printing of checking output [TRUE]}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
This loads the meta data file found at \code{meta_path}, or creates it with a warning if no file is present. It \code{check} is \code{TRUE} then the meta data file contents are checked with the level of returned information being controlled by \code{verbose}.
}

\subsection{Returns}{
A new `dynatopGIS` object
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_meta"></a>}}
\if{latex}{\out{\hypertarget{method-get_meta}{}}}
\subsection{Method \code{get_meta()}}{
Get project meta data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$get_meta()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_working_directory"></a>}}
\if{latex}{\out{\hypertarget{method-get_working_directory}{}}}
\subsection{Method \code{get_working_directory()}}{
Get current working directory
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$get_working_directory()}\if{html}{\out{</div>}}
}

\subsection{Details}{
Newly generated layers are added to the working directory. By default this is the directory containing the meta date file.
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-set_working_directory"></a>}}
\if{latex}{\out{\hypertarget{method-set_working_directory}{}}}
\subsection{Method \code{set_working_directory()}}{
Set current working directory
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$set_working_directory(file_path, create = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{file_path}}{the path to the new directory to create}

\item{\code{create}}{should the directory be created if it doesn't exist}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Newly generated layers are added to the working directory. By default this is the directory containing the meta date file.
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-add_dem"></a>}}
\if{latex}{\out{\hypertarget{method-add_dem}{}}}
\subsection{Method \code{add_dem()}}{
Import a dem to the `dynatopGIS` object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$add_dem(dem, fill_na = TRUE, verbose = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dem}}{a \code{raster} layer object or the path to file containing one which is the DEM}

\item{\code{fill_na}}{should NA values in dem be filled. See details}

\item{\code{verbose}}{Should additional progress information be printed}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
If not a \code{raster} the DEM is read in using the raster package. If \code{fill_na} is \code{TRUE} all NA values other then those that link to the edge of the dem are filled so they can be identified as sinks.
}

\subsection{Returns}{
suitable for chaining
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-add_channel"></a>}}
\if{latex}{\out{\hypertarget{method-add_channel}{}}}
\subsection{Method \code{add_channel()}}{
Import channel data to the `dynatopGIS` object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$add_channel(
  channel,
  property_names = c(length = "length", startNode = "startNode", endNode = "endNode",
    width = "width"),
  default_width = 2
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{channel}}{a SpatialLinesDataFrame, SpatialPolygonsDataFrame or file path containing the channel information}

\item{\code{property_names}}{named vector of columns of the spatial data frame to use for channel properties - see details}

\item{\code{default_width}}{default width of a channel if not specified in property_names. Defaults to 2 metres.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Takes the input channel converts it a SpatialPolygonDataFrame with properties length, startNode and endNode. The variable names in the sp_object data frame which corresponding to these properties can be specified in the \code{property_names} vector. In the channel is a SpatialLinesDataFrame (or read in as one) an additional property width is used to buffer the lines and create channel polygons. If required the width property is created using the default value. Note that any columns called length, startNode, endNode  and width are overwritten. Any column called id is copied to a column original_id then overwritten.
}

\subsection{Returns}{
suitable for chaining
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-add_layer"></a>}}
\if{latex}{\out{\hypertarget{method-add_layer}{}}}
\subsection{Method \code{add_layer()}}{
Add a layer of geographical information
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$add_layer(layer_name, file_path)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{layer_name}}{name to give to the layer}

\item{\code{file_path}}{the location of the file containing the new layer}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The file given is read by the \code{raster} package and checked against the project meta data. Only layer names not already in use (or reserved) are allowed. If successful the meta data for the project are altered to reflect the new layer name and file location.
}

\subsection{Returns}{
suitable for chaining
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_layer"></a>}}
\if{latex}{\out{\hypertarget{method-get_layer}{}}}
\subsection{Method \code{get_layer()}}{
Get a layer of geographical information or a list of layer names
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$get_layer(layer_name = character(0))}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{layer_name}}{name of the layer give to the layer}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
a `raster` layer of the requested information if layer_name is given else a vector of layer names
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-plot_layer"></a>}}
\if{latex}{\out{\hypertarget{method-plot_layer}{}}}
\subsection{Method \code{plot_layer()}}{
Plot a layer
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$plot_layer(layer_name, add_channel = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{layer_name}}{the name of layer to plot}

\item{\code{add_channel}}{should the channel be added to the plot}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
a plot
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-sink_fill"></a>}}
\if{latex}{\out{\hypertarget{method-sink_fill}{}}}
\subsection{Method \code{sink_fill()}}{
The sink filling algorithm of Planchona and Darboux (2001)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$sink_fill(
  min_grad = 1e-04,
  max_it = 1e+06,
  verbose = FALSE,
  hot_start = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{min_grad}}{Minimum gradient between cell centres}

\item{\code{max_it}}{maximum number of replacement cycles}

\item{\code{verbose}}{print out additional diagnostic information}

\item{\code{hot_start}}{start from filled_dem if it exists}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The algorithm implemented is that described in Planchona and Darboux, "A fast, simple and versatile algorithm to fill the depressions in digital elevation models" Catena 46 (2001). A pdf can be found at (<https://horizon.documentation.ird.fr/exl-doc/pleins_textes/pleins_textes_7/sous_copyright/010031925.pdf>).
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-compute_areas"></a>}}
\if{latex}{\out{\hypertarget{method-compute_areas}{}}}
\subsection{Method \code{compute_areas()}}{
Computes area maps and presence of channel in dem pixels
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$compute_areas()}\if{html}{\out{</div>}}
}

\subsection{Details}{
The algorithm calculates the land and channel area for each DEM pixel assigning a channel_id to each pixel with a channel area.
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-compute_properties"></a>}}
\if{latex}{\out{\hypertarget{method-compute_properties}{}}}
\subsection{Method \code{compute_properties()}}{
Computes statistics e.g. gradient, log(upslope area / gradient) for raster cells
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$compute_properties(min_grad = 1e-04, verbose = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{min_grad}}{gradient that can be assigned to a pixel if it can't be computed}

\item{\code{verbose}}{print out additional diagnostic information}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The algorithm passed through the cells in decreasing height. Min grad is applied to all cells. It is also used for missing gradients in pixels which are partially channel but have no upslope neighbours.
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-compute_flow_lengths"></a>}}
\if{latex}{\out{\hypertarget{method-compute_flow_lengths}{}}}
\subsection{Method \code{compute_flow_lengths()}}{
Computes flow length for each pixel to the channel
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$compute_flow_lengths(verbose = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{verbose}}{print out additional diagnostic information}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The algorithm passed through the cells in increasing height. For measures of flow length to the channel are computed. The shortest length (minimum length to channel through any flow path), the dominant length (the length taking the flow direction with the highest fraction for each pixel on the path) and expected flow length (flow length based on sum of downslope flow lengths based on fraction of flow to each cell) and band (strict sequence to ensure that all contributing cell have a higher band value). By definition cells in the channel that have no land area have a length (or band) of NA.
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-classify"></a>}}
\if{latex}{\out{\hypertarget{method-classify}{}}}
\subsection{Method \code{classify()}}{
Create a catchment classification based cutting an existing layer into classes
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$classify(layer_name, base_layer, cuts)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{layer_name}}{name of the new layer to create}

\item{\code{base_layer}}{name of the layer to be cut into classes}

\item{\code{cuts}}{values on which to cut into classes. These should be numeric and define either the number of bands (single value) or breaks between band (multiple values).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
This applies the given cuts to the supplied landscape layer to produce areal groupings of the catchment. Cuts are implement using \code{raster::cut} with \code{include.lowest = TRUE}. Note that is specifying a vector of cuts values outside the limits will be set to NA.
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-combine_classes"></a>}}
\if{latex}{\out{\hypertarget{method-combine_classes}{}}}
\subsection{Method \code{combine_classes()}}{
Combine any number of classifications based on unique combinations and burns
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$combine_classes(layer_name, pairs, burns = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{layer_name}}{name of the new layer to create}

\item{\code{pairs}}{a vector of layer names to combine into new classes through unique combinations. Names should correspond to raster layers in the project directory.}

\item{\code{burns}}{a vector of layer names which are to be burnt on}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
This applies the given cuts to the supplied landscape layers to produce areal groupings of the catchment. Burns are added directly in the order they are given. Cuts are implement using \code{raster::cut} with \code{include.lowest = TRUE}. Note that is specifying a vector of cuts values outside the limits will be set to NA.
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-create_model"></a>}}
\if{latex}{\out{\hypertarget{method-create_model}{}}}
\subsection{Method \code{create_model()}}{
Compute a Dynamic TOPMODEL
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$create_model(
  layer_name,
  class_layer,
  dist_layer,
  transmissivity = c("exp", "bexp", "cnst", "dexp"),
  channel_solver = c("histogram"),
  dist_delta = 0,
  rain_layer = NULL,
  rain_label = character(0),
  pet_layer = NULL,
  pet_label = character(0),
  verbose = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{layer_name}}{name for the new model and layers}

\item{\code{class_layer}}{the layer defining the topographic classes}

\item{\code{dist_layer}}{the layer defining the distances to the channel}

\item{\code{transmissivity}}{transmissivity profile to use}

\item{\code{channel_solver}}{channel solver to use}

\item{\code{dist_delta}}{used in computing flow linkages see details}

\item{\code{rain_layer}}{the layer defining the rainfall inputs}

\item{\code{rain_label}}{Prepended to rain_layer values to give rainfall series name}

\item{\code{pet_layer}}{the layer defining the pet inputs}

\item{\code{pet_label}}{Prepended to pet_layer values to give pet series name}

\item{\code{verbose}}{print more details of progress}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The \code{class_layer} is used to define the HRUs. Flow between HRUs is based on the distance to a channel. For each HRU the shortest distance to a channel is computed. Flow from a HRU can only go to a HRU with a lower shortest distance to the channel. Flow from a HRU can occur from any raster cell within the HRU whose distance to the channel is within dist_delta of the shortest distance within the HRU.
Setting the transmissivity and channel_solver options ensure the model is set up with the correct parameters present.
The \code{rain_layer} (\code{pet_layer}) can contain the numeric id values of different rainfall (pet) series. If the value of \code{rain_layer} (\code{pet_layer}) is not \code{NULL} the weights used to compute an averaged input value for each HRU are computed, otherwise an input table for the models generated with the value "missing" used in place of the series name.
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_version"></a>}}
\if{latex}{\out{\hypertarget{method-get_version}{}}}
\subsection{Method \code{get_version()}}{
get the version number
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$get_version()}\if{html}{\out{</div>}}
}

\subsection{Details}{
the version number indicates the version of the algorithms within the object
}

\subsection{Returns}{
a numeric version number
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_class_method"></a>}}
\if{latex}{\out{\hypertarget{method-get_class_method}{}}}
\subsection{Method \code{get_class_method()}}{
get the cuts and burns used to classify
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$get_class_method(layer_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{layer_name}}{the name of layer whose classification method is returned}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
a list with two elements, cuts and burns
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{dynatopGIS$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
