% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateConfModel.R
\name{simulateRTConf}
\alias{simulateRTConf}
\alias{rConfModel}
\alias{simulateConfModel}
\title{Simulation of confidence ratings and RTs in sequential sampling confidence models}
\usage{
simulateRTConf(paramDf, n = 10000, model = NULL, gamma = FALSE,
  agg_simus = FALSE, simult_conf = FALSE, stimulus = c(1, 2),
  delta = 0.01, maxrt = 15, seed = NULL)
}
\arguments{
\item{paramDf}{a list or dataframe with one row with the required parameters.}

\item{n}{integer. The number of samples (per condition and stimulus direction) generated.
Total number of samples is \code{n*nConditions*length(stimulus)}.}

\item{model}{character scalar. One of "2DSD", "dynWEV", "IRM", "PCRM", "IRMt", or "PCRMt".}

\item{gamma}{logical. If TRUE, the gamma correlation between confidence ratings, rt and accuracy is
computed.}

\item{agg_simus}{logical. Simulation is done on a trial basis with RTs outcome. If TRUE,
the simulations will be aggregated over RTs to return only the distribution of response and
confidence ratings. Default: FALSE.}

\item{simult_conf}{logical. Whether in the experiment confidence was reported simultaneously
with the decision. If that is the case decision and confidence judgment are assumed to have happened
subsequent before the response. Therefore \code{tau} is included in the response time. If the decision was
reported before the confidence report, \code{simul_conf} should be \code{FALSE}.}

\item{stimulus}{numeric vector. Either 1, 2 or c(1, 2) (default).
Together with condition represents the experimental situation. In a 2AFC task the presented
stimulus belongs to one of two categories. In the default setting trials with
both categories presented are simulated but one can choose to simulate only trials with the
stimulus coming from one category.}

\item{delta}{numerical. Size of steps for the discretized simulation.}

\item{maxrt}{numerical. Maximum reaction time to be simulated. Default: 15.}

\item{seed}{numerical. Seeding for non-random data generation. (Also possible outside of the function.)}
}
\value{
Depending on \code{gamma} and \code{agg_simus}.

If \code{gamma} is \code{FALSE}, returns a \code{data.frame} with columns: \code{condition},
\code{stimulus}, \code{response}, \code{correct}, \code{rt}, \code{conf} (the continuous confidence
measure) and \code{rating} (the discrete confidence rating) or
(if \code{agg_simus=TRUE}): \code{condition}, \code{stimulus},\code{response}, \code{correct},
\code{rating} and \code{p} (for the probability of a response and rating, given
the condition and stimulus).

If \code{gamma} is \code{TRUE}, returns a \code{list} with elements:
\code{simus} (the simulated data frame) and \code{gamma}, which is again a \code{list} with elements
\code{condition}, \code{rt} and \code{correct}, each a \code{tibble} with two columns (see details for more
information).
}
\description{
Simulates the decision responses, reaction times and confidence measure
together with a discrete confidence judgment for the sequential sampling confidence model
specified by the argument \code{model}, given specific parameter constellations.
This function is a wrapper that calls the respective functions for diffusion based
models (dynWEV and 2DSD: \code{\link{simulateWEV}}) and race models (IRM, PCRM,
IRMt, and PCRMt: \code{\link{simulateRM}}. It also computes the Gamma rank correlation
between the confidence ratings and
condition (task difficulty), reaction times and accuracy in the simulated output.
}
\details{
The output of the fitting function \code{\link{fitRTConf}} with the respective model
fits the argument \code{paramDf} for simulation. The function calls the respective simulation
function for diffusion based models, i.e. dynWEV and 2DSD (\code{\link{simulateWEV}}) or race models,
i.e. IRM(t) and PCRM(t), (\code{\link{simulateRM}}). See there for more information.

\strong{Simulation Method:} The simulation is done by simulating normal variables
in discretized steps until
the processes reach the boundary. If no boundary is met within the maximum time,
response is set to 0.

\strong{Gamma correlations:} The Gamma coefficients are computed separately for
correct/incorrect responses for the correlation of confidence ratings with condition and rt
and separately for conditions for the correlation of accuracy and confidence. The resulting
data frames in the output thus have two columns. One for the grouping variable and one for the
Gamma coefficient.
}
\examples{

# The function is particularly useful, when having a collection
# of parameter sets for different models (e.g. output by fitRTConfModels for
# more than one model).
library(dplyr)
# 1. Generate only one parameter set but for two different models
paramDf1 <- data.frame(model="dynWEV", a=1.5,v1=0.2, v2=1, t0=0.1,z=0.52,
                      sz=0.3,sv=0.4, st0=0,  tau=3, w=0.5,
                      theta1=1, svis=0.5, sigvis=0.8)
paramDf2 <- data.frame(model="PCRMt", a=2,b=2, v1=0.5, v2=1, t0=0.1,st0=0,
                      wx=0.6, wint=0.2, wrt=0.2, theta1=4)
paramDf <- full_join(paramDf1, paramDf2)
paramDf  # each model parameters sets hat its relevant parameters
# Split paramDf by model (maybe also other columns) and simulate data
simus <- paramDf |> group_by(model) |>
 reframe(simulateRTConf(cbind(cur_group(), pick(everything())), n=200, simult_conf = TRUE))
head(simus)

}
\author{
Sebastian Hellmann.
}
