\docType{package}
\name{doRNG-package}
\alias{doRNG-package}
\title{Generic Reproducible Parallel Backend for foreach Loops}
\description{
  The \emph{doRNG} package provides functions to perform
  reproducible parallel foreach loops, using independent
  random streams as generated by L'Ecuyer's combined
  multiple-recursive generator [L'Ecuyer (1999)]. It
  enables to easily convert standard %dopar% loops into
  fully reproducible loops, independently of the number of
  workers, the task scheduling strategy, or the chosen
  parallel environment and associated foreach backend. It
  has been tested with the following foreach backend: doMC,
  doSNOW, doMPI.
}
\details{
  \tabular{ll}{ Package: \tab doRNG\cr Type: \tab
  Package\cr Version: \tab 1.2.3\cr Date: \tab
  2012-03-30\cr License: \tab GPL (>= 2)\cr LazyLoad: \tab
  yes\cr }
}
\examples{
\dontshow{# roxygen generated flag
options(R_CHECK_RUNNING_EXAMPLES_=TRUE)
}

# register parallel backend
library(doParallel)
cl <- makeCluster(2)
registerDoParallel(cl)

## standard \%dopar\% loop are not reproducible
set.seed(123)
r1 <- foreach(i=1:4) \%dopar\%{ runif(1) }
set.seed(123)
r2 <- foreach(i=1:4) \%dopar\%{ runif(1) }
identical(r1, r2)
\dontshow{ stopifnot(!identical(r1, r2)) }

## \%dorng\% loops _are_ reproducible
set.seed(123)
r1 <- foreach(i=1:4) \%dorng\%{ runif(1) }
set.seed(123)
r2 <- foreach(i=1:4) \%dorng\%{ runif(1) }
identical(r1, r2)
\dontshow{ stopifnot(identical(r1, r2)) }

# alternative way of seeding
a1 <- foreach(i=1:4, .options.RNG=123) \%dorng\%{ runif(1) }
a2 <- foreach(i=1:4, .options.RNG=123) \%dorng\%{ runif(1) }
identical(a1, a2) && identical(a1, r1)
\dontshow{ stopifnot(identical(a1, a2) && identical(a1, r1)) }

## sequences of \%dorng\% loops _are_ reproducible
set.seed(123)
s1 <- foreach(i=1:4) \%dorng\%{ runif(1) }
s2 <- foreach(i=1:4) \%dorng\%{ runif(1) }
identical(s1, r1) && !identical(s1, s2)
\dontshow{ stopifnot(identical(s1, r1) && !identical(s1, s2)) }

set.seed(123)
s1.2 <- foreach(i=1:4) \%dorng\%{ runif(1) }
s2.2 <- foreach(i=1:4) \%dorng\%{ runif(1) }
identical(s1, s1.2) && identical(s2, s2.2)
\dontshow{ stopifnot(identical(s1, s1.2) && identical(s2, s2.2)) }

## Non-invasive way of converting \%dopar\% loops into reproducible loops
registerDoRNG(123)
s3 <- foreach(i=1:4) \%dopar\%{ runif(1) }
s4 <- foreach(i=1:4) \%dopar\%{ runif(1) }
identical(s3, s1) && identical(s4, s2)
\dontshow{ stopifnot(identical(s3, s1) && identical(s4, s2)) }

stopCluster(cl)
}
\author{
  Renaud Gaujoux \email{renaud@cbio.uct.ac.za},

  Maintainer: Renaud Gaujoux \email{renaud@cbio.uct.ac.za},
}
\references{
  L'Ecuyer P (1999). "Good parameters and implementations
  for combined multiple recursive random number
  generators." _Operations Research_, *47*(1). <URL:
  http://www.jstor.org/stable/10.2307/222902>.
}
\seealso{
  \code{\link{doRNG}}, \code{\link{RNGseq}}
}
\keyword{package}

