% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddml_late.R
\name{ddml_late}
\alias{ddml_late}
\title{Estimator of the Local Average Treatment Effect.}
\usage{
ddml_late(
  y,
  D,
  Z,
  X,
  learners,
  learners_DXZ = learners,
  learners_ZX = learners,
  sample_folds = 10,
  ensemble_type = "nnls",
  shortstack = FALSE,
  cv_folds = 10,
  custom_ensemble_weights = NULL,
  custom_ensemble_weights_DXZ = custom_ensemble_weights,
  custom_ensemble_weights_ZX = custom_ensemble_weights,
  cluster_variable = seq_along(y),
  subsamples_byZ = NULL,
  cv_subsamples_byZ = NULL,
  trim = 0.01,
  silent = FALSE
)
}
\arguments{
\item{y}{The outcome variable.}

\item{D}{The binary endogenous variable of interest.}

\item{Z}{Binary instrumental variable.}

\item{X}{A (sparse) matrix of control variables.}

\item{learners}{May take one of two forms, depending on whether a single
learner or stacking with multiple learners is used for estimation of the
conditional expectation functions.
If a single learner is used, \code{learners} is a list with two named
elements:
\itemize{
\item{\code{what} The base learner function. The function must be
such that it predicts a named input \code{y} using a named input
\code{X}.}
\item{\code{args} Optional arguments to be passed to \code{what}.}
}
If stacking with multiple learners is used, \code{learners} is a list of
lists, each containing four named elements:
\itemize{
\item{\code{fun} The base learner function. The function must be
such that it predicts a named input \code{y} using a named input
\code{X}.}
\item{\code{args} Optional arguments to be passed to \code{fun}.}
\item{\code{assign_X} An optional vector of column indices
corresponding to control variables in \code{X} that are passed to
the base learner.}
\item{\code{assign_Z} An optional vector of column indices
corresponding to instruments in \code{Z} that are passed to the
base learner.}
}
Omission of the \code{args} element results in default arguments being
used in \code{fun}. Omission of \code{assign_X} (and/or \code{assign_Z})
results in inclusion of all variables in \code{X} (and/or \code{Z}).}

\item{learners_DXZ, learners_ZX}{Optional arguments to allow for different
estimators of \eqn{E[D \vert X, Z]}, \eqn{E[Z \vert X]}. Setup is
identical to \code{learners}.}

\item{sample_folds}{Number of cross-fitting folds.}

\item{ensemble_type}{Ensemble method to combine base learners into final
estimate of the conditional expectation functions. Possible values are:
\itemize{
\item{\code{"nnls"} Non-negative least squares.}
\item{\code{"nnls1"} Non-negative least squares with the constraint
that all weights sum to one.}
\item{\code{"singlebest"} Select base learner with minimum MSPE.}
\item{\code{"ols"} Ordinary least squares.}
\item{\code{"average"} Simple average over base learners.}
}
Multiple ensemble types may be passed as a vector of strings.}

\item{shortstack}{Boolean to use short-stacking.}

\item{cv_folds}{Number of folds used for cross-validation in ensemble
construction.}

\item{custom_ensemble_weights}{A numerical matrix with user-specified
ensemble weights. Each column corresponds to a custom ensemble
specification, each row corresponds to a base learner in \code{learners}
(in chronological order). Optional column names are used to name the
estimation results corresponding the custom ensemble specification.}

\item{custom_ensemble_weights_DXZ, custom_ensemble_weights_ZX}{Optional
arguments to allow for different
custom ensemble weights for \code{learners_DXZ},\code{learners_ZX}. Setup
is identical to \code{custom_ensemble_weights}. Note:
\code{custom_ensemble_weights} and
\code{custom_ensemble_weights_DXZ},\code{custom_ensemble_weights_ZX} must
have the same number of columns.}

\item{cluster_variable}{A vector of cluster indices.}

\item{subsamples_byZ}{List of two lists corresponding to the two instrument
levels. Each list contains vectors with sample indices for
cross-fitting.}

\item{cv_subsamples_byZ}{List of two lists, each corresponding to one of the
two instrument levels. Each of the two lists contains lists, each
corresponding to a subsample and contains vectors with subsample indices
for cross-validation.}

\item{trim}{Number in (0, 1) for trimming the estimated propensity scores at
\code{trim} and \code{1-trim}.}

\item{silent}{Boolean to silence estimation updates.}
}
\value{
\code{ddml_late} returns an object of S3 class
\code{ddml_late}. An object of class \code{ddml_late} is a list
containing the following components:
\describe{
\item{\code{late}}{A vector with the average treatment effect
estimates.}
\item{\code{weights}}{A list of matrices, providing the weight
assigned to each base learner (in chronological order) by the
ensemble procedure.}
\item{\code{mspe}}{A list of matrices, providing the MSPE of each
base learner (in chronological order) computed by the
cross-validation step in the ensemble construction.}
\item{\code{psi_a}, \code{psi_b}}{Matrices needed for the computation
of scores. Used in \code{\link[=summary.ddml_late]{summary.ddml_late()}}.}
\item{\code{oos_pred}}{List of matrices, providing the reduced form
predicted values.}
\item{\code{learners},\code{learners_DXZ},\code{learners_ZX},
\code{cluster_variable},\code{subsamples_Z0},
\code{subsamples_Z1},\code{cv_subsamples_list_Z0},
\code{cv_subsamples_list_Z1},\code{ensemble_type}}{Pass-through
of selected user-provided arguments. See above.}
}
}
\description{
Estimator of the local average treatment effect.
}
\details{
\code{ddml_late} provides a double/debiased machine learning
estimator for the local average treatment effect in the interactive model
given by

\eqn{Y = g_0(D, X) + U,}

where \eqn{(Y, D, X, Z, U)} is a random vector such that
\eqn{\operatorname{supp} D = \operatorname{supp} Z = \{0,1\}},
\eqn{E[U\vert X, Z] = 0}, \eqn{E[Var(E[D\vert X, Z]\vert X)] \neq 0},
\eqn{\Pr(Z=1\vert X) \in (0, 1)} with probability 1,
\eqn{p_0(1, X) \geq p_0(0, X)} with probability 1 where
\eqn{p_0(Z, X) \equiv \Pr(D=1\vert Z, X)}, and
\eqn{g_0} is an unknown nuisance function.

In this model, the local average treatment effect is defined as

\eqn{\theta_0^{\textrm{LATE}} \equiv
    E[g_0(1, X) - g_0(0, X)\vert p_0(1, X) > p(0, X)]}.
}
\examples{
# Construct variables from the included Angrist & Evans (1998) data
y = AE98[, "worked"]
D = AE98[, "morekids"]
Z = AE98[, "samesex"]
X = AE98[, c("age","agefst","black","hisp","othrace","educ")]

# Estimate the local average treatment effect using a single base learner,
#     ridge.
late_fit <- ddml_late(y, D, Z, X,
                      learners = list(what = mdl_glmnet,
                                      args = list(alpha = 0)),
                      sample_folds = 2,
                      silent = TRUE)
summary(late_fit)

# Estimate the local average treatment effect using short-stacking with base
#     learners ols, lasso, and ridge. We can also use custom_ensemble_weights
#     to estimate the ATE using every individual base learner.
weights_everylearner <- diag(1, 3)
colnames(weights_everylearner) <- c("mdl:ols", "mdl:lasso", "mdl:ridge")
late_fit <- ddml_late(y, D, Z, X,
                      learners = list(list(fun = ols),
                                      list(fun = mdl_glmnet),
                                      list(fun = mdl_glmnet,
                                           args = list(alpha = 0))),
                      ensemble_type = 'nnls',
                      custom_ensemble_weights = weights_everylearner,
                      shortstack = TRUE,
                      sample_folds = 2,
                      silent = TRUE)
summary(late_fit)
}
\references{
Ahrens A, Hansen C B, Schaffer M E, Wiemann T (2023). "ddml: Double/debiased
machine learning in Stata." \url{https://arxiv.org/abs/2301.09397}

Chernozhukov V, Chetverikov D, Demirer M, Duflo E, Hansen C B, Newey W,
Robins J (2018). "Double/debiased machine learning for treatment and
structural parameters." The Econometrics Journal, 21(1), C1-C68.

Imbens G, Angrist J (1004). "Identification and Estimation of Local Average
Treatment Effects." Econometrica, 62(2), 467-475.

Wolpert D H (1992). "Stacked generalization." Neural Networks, 5(2), 241-259.
}
\seealso{
\code{\link[=summary.ddml_late]{summary.ddml_late()}}

Other ddml: 
\code{\link{ddml_ate}()},
\code{\link{ddml_fpliv}()},
\code{\link{ddml_pliv}()},
\code{\link{ddml_plm}()}
}
\concept{ddml}
