# akde object generator
# list of kde objects with info slots
new.akde <- methods::setClass("akde", representation(info="list",alpha="numeric"), contains="list")


# Slow lag counter
tab.lag.DOF <- function(data,fast=NULL,dt=NULL)
{
  t <- data$t
  # intelligently select algorithm
  n <- length(t)
  if(is.null(fast))
  {
    if(n<100) { fast <- FALSE }
    else { fast <- TRUE }
  }
  
  # calculate lag,n(lag) vectors
  if(fast)
  {
    lag.DOF <- variogram.fast(data,dt=dt,CI="IID") # count all lag pairs
    lag.DOF$SVF <- NULL
    
    # lag==0 should not be doubled
    lag.DOF$DOF[1] <- lag.DOF$DOF[1]/2 
  }
  else
  {
    lag <- outer(t,t,FUN="-")
    lag <- abs(lag) # may have to change far in the future
    lag <- c(lag) # collapse to 1D array
    
    # can we think of a faster sorter?
    r <- lag # to be the remainder
    lag <- NULL
    DOF <- NULL
    while(length(r)>0)
    {
      n <- length(r)
      lag <- c(lag,r[1])
      r <- r[r!=r[1]]
      DOF <- c(DOF,n-length(r))
    }
    
    lag.DOF <- list(lag=lag,DOF=DOF)
  }
  
  return(lag.DOF)
}


##################################
# Bandwidth optimizer
#lag.DOF is an unsupported option for end users
akde.bandwidth <- function(data,CTMM,fast=NULL,dt=NULL)
{
  lag.DOF <- tab.lag.DOF(data,fast=fast,dt=dt)
  # Extract lag data
  DOF <- lag.DOF$DOF
  lag <- lag.DOF$lag
  
  tau <- CTMM$tau
  tau <- tau[tau>0]
  K <- length(tau)
  
  # standardized SVF
  if(K==0)
  {
    g <- Vectorize( function(t) { if(t==0) {0} else {1} } )
  }
  else if(K==1)
  {
    g <- Vectorize( function(t) { 1-exp(-t/tau) } )
  }
  else if(K==2)
  {
    g <- Vectorize( function(t) { 1-diff(tau*exp(-t/tau))/diff(tau) } )
  }
  
  n <- length(data$t)
  
  # Mean Integrated Square Error modulo a constant
  MISE <- function(h)
  {
    if(h<=0) {Inf}
    else { (1/n^2)*sum(DOF/(2*g(lag)+2*h^2)) - 2/(2+h^2) + 1/2 }
  }
  
  h <- 1/n^(1/6) # User Silverman's rule of thumb to place lower bound
  h <- stats::optimize(f=MISE,interval=c(h/2,2))$minimum
  
  H <- h^2*CTMM$sigma
  
  rownames(H) <- c("x","y")
  colnames(H) <- c("x","y")
  
  return(H)
}

#######################################
# wrap the kde function for our telemetry data format and CIs.
akde <- function(data,CTMM,alpha=0.05,fast=NULL,dt=NULL,error=0.001,res=200,grid=NULL)
{
  pb <- utils::txtProgressBar(min=-1,max=3,initial=-1,style=3)
  
  tau <- CTMM$tau
  K <- length(tau)
  
  sigma <- CTMM$sigma
  # covariance area/scale
  GM.sigma <- sqrt(det(sigma))
  # orientation matrix
  R <- sigma/GM.sigma

  # wrapper function to estimate bandwidth matrix
  # par = c(sigma.GM,tau)
  fn.H <- function(par)
  {
    CTMM.par <- CTMM
    CTMM.par$sigma <- par[1]*R
    if(K>0) { CTMM.par$tau <- par[-1] }
    H <- akde.bandwidth(data=data,CTMM=CTMM.par,fast=fast,dt=dt)
    return(H)
  }

  # wrapper function to estimate bandwidth area
  # par = c(sigma.GM,tau)
  fn.GM.H <- function(par)
  {
    H <- fn.H(par)
    GM.H <- sqrt(det(H))
    return(GM.H)
  }
  # How strongly optimal bandwidth varries with parameter estimates
  d.GM.H <- numDeriv::grad(fn.GM.H,c(GM.sigma,tau))
  
  # ML propagated curvature covariance
  COV <- d.GM.H %*% CTMM$COV.tau %*% d.GM.H
  # ML Bandwidth area
  GM.H <- fn.GM.H(c(GM.sigma,tau))
  # confidence intervals from chi^2
  GM.H <- chisq.ci(GM.H,COV,alpha)

  # data formatted for ks::kde
  x <- cbind(data$x,data$y)
  
  # object to store crap
  KDE <- list(low=0,ML=0,high=0)
  
  utils::setTxtProgressBar(pb,0)
  for(i in 1:3)
  {
    H <- GM.H[i]*R
    KDE[[i]] <- kde(data,H,alpha=error,res=res,grid=grid)
    KDE[[i]]$H <- H
    utils::setTxtProgressBar(pb,i)
  }
  
  KDE <- new.akde(KDE,info=attr(data,"info"),alpha=alpha)
  
  close(pb)
  return(KDE)
}



##################################
# construct my own kde objects
# was using ks-package but it has some bugs
# alpha is the error goal in my total probability
kde <- function(data,H,W=rep(1,length(data$x)),alpha=0.001,res=100,grid=NULL)
{
  x <- data$x
  y <- data$y

  # normalize weights
  W <- W/sum(W)
    
  # if a single H matrix is given, make it into an array of H matrices
  n <- length(x)
  if(length(dim(H))==2)
  {
    H <- array(H,c(2,2,n))
    H <- aperm(H,c(3,1,2))
  }

  # design a good grid
  if(is.null(grid))
  {
    # how far to extend range from data as to ensure alpha significance in total probability
    z <- sqrt(-2*log(alpha))
    DX <- z * apply(H,1,function(h){sqrt(h[1,1])})
    DY <- z * apply(H,1,function(h){sqrt(h[2,2])})
    
    # now to find the necessary extent of our grid
    min.x <- min(x - DX)
    max.x <- max(x + DX)
    
    min.y <- min(y - DY)
    max.y <- max(y + DY)
    
    # grid center
    mu.x <- (min.x+max.x)/2
    mu.y <- (min.y+max.y)/2
    
    # grid resolution 
    dx <- (max.x-min.x)/(res)
    dy <- (max.y-min.y)/(res)
    
    # grid locations
    res <- res/2+1 # half resolution
    X <- mu.x + (-res):(res)*dx
    Y <- mu.y + (-res):(res)*dy
  }
  
  cdf <- array(0,c(length(X),length(Y)))
  for(i in 1:n)
  {
    # sub-grid row indices
    r1 <- floor((x[i]-DX[i]-X[1])/dx) + 1
    r2 <- ceiling((x[i]+DX[i]-X[1])/dx) + 1
    
    # sub-grid column indices
    c1 <- floor((y[i]-DY[i]-Y[1])/dy) + 1
    c2 <- ceiling((y[i]+DY[i]-Y[1])/dy) + 1
    
    #if(i==2){ return(list(x=X[r1:r2],y=Y[c1:c2],CDF=pnorm2(X[r1:r2],Y[c1:c2],c(x[i],y[i]),H[i,,],dx,dy),mu=c(x[i],y[i]),sigma=H[i,,])) }
    
    cdf[r1:r2,c1:c2] <- cdf[r1:r2,c1:c2] + W[i]*pnorm2(X[r1:r2]-x[i],Y[c1:c2]-y[i],H[i,,],dx,dy,alpha)
  }

  dA <- dx*dy
  pdf <- cdf/dA
  
  # cdf: cell probability -> probability included in contour
  DIM <- dim(cdf)
  cdf <- c(cdf) # flatten table
  cdf <- sort(cdf,decreasing=TRUE,method="quick",index.return=TRUE)
  IND <- cdf[[2]] # sorted indices
  cdf <- cdf[[1]]
  cdf <- cumsum(cdf)
  cdf[IND] <- cdf # back in spatial order
  cdf <- array(cdf,DIM) # back in table form
  
  result <- list(PDF=pdf,CDF=cdf,x=X,y=Y,dA=dA)
  class(result) <- "kde"
  
  return(result)
}


#######################
# robust bi-variate CDF (mean zero assumed)
#######################
pnorm2 <- function(X,Y,sigma,dx=stats::mean(diff(X)),dy=stats::mean(diff(Y)),alpha=0.001)
{
  cdf <- array(0,c(length(X),length(Y)))
  
  # eigensystem of kernel covariance
  v <- eigen(sigma)
  s <- v$values
  
  # effective degree of degeneracy at best resolution
  ZERO <- sum(s/min(dx,dy)^2 <= 0)

  # correlation
  S <- sqrt(sigma[1,1]*sigma[2,2])
  if(S>0)
  {
    rho <- sigma[1,2]/S
    # prevent some tiny numerical errors just in case
    rho <- clamp(rho,min=-1,max=1)
  }
  else { rho <- 0 }
  
  # main switch
  if(ZERO==0 && abs(rho)<1) # no degeneracy
  {
    # relative grid size (worst case)
    z <- sqrt((dx^2+dy^2)/s[2])
    
    if(z^3/12<=alpha) # midpoint integration
    {
      cdf <- (dx*dy) * Gauss(X,Y,sigma)
    }
    else if(z^5/2880<=alpha) # Simpson integration
    {
      W <- c(1,4,1)
      cdf <- NewtonCotes(X,Y,sigma,W,dx,dy)
    }
    else if(z^7/1935360<=alpha) # Boole integration
    {
      W <- c(7,32,12,32,7)
      cdf <- NewtonCotes(X,Y,sigma,W,dx,dy)
    }
    else # exact calculation
    {
      # offset to corners
      x <- c(X-dx/2,last(X)+dx/2)
      y <- c(Y-dy/2,last(Y)+dy/2)
      
      # standardized all locations
      x <- (x)/sqrt(sigma[1,1])
      y <- (y)/sqrt(sigma[2,2])
      
      # dimensions
      n.x <- length(x)
      n.y <- length(y)
      
      # corner grid of cell probabilities
      CDF <- outer(x,y,function(x,y){pbivnorm::pbivnorm(x,y,rho=rho)})
      
      # integrate over cell and add
      cdf <- CDF[-1,-1] - CDF[-n.x,-1] - CDF[-1,-n.y] + CDF[-n.x,-n.y]
    }
  }
  else if(ZERO==1 || abs(rho)==1) # line degeneracy
  {
    # max variance
    s <- s[1]
    # unit vector of max variance
    v <- v$vectors[,1]
    
    # crossings along X grid
    x.cell <- c()
    y.cross <- c()
    m.y <- v[2]/v[1]
    if(abs(m.y)<Inf)
    {
      x.cell <- c(X-dx/2,last(X)+dx/2)
      y.cross <- (x.cell)*m.y
    }
    
    # crossings along Y grid
    y.cell <- c()
    x.cross <- c()
    m.x <- v[1]/v[2]
    if(abs(m.x)<Inf)
    {
      y.cell <- c(Y-dy/2,last(Y)+dy/2)
      x.cross <- (y.cell)*m.x
    }
    
    # all crossings
    x.cross <- c(x.cell,x.cross)
    y.cross <- c(y.cross,y.cell)
    
    # standardized location along line
    z.cross <- ((x.cross)*v[1] + (y.cross)*v[2])/sqrt(s)
    z.cross <- sort(z.cross,method="quick")
    z.cross <- unique(z.cross)
    
    for(i in 1:(length(z.cross)-1))
    {
      # what cell is this line segment in?
      z.mid <- mean(z.cross[i:(i+1)])
      
      x <- sqrt(s)*v[1]*z.mid
      y <- sqrt(s)*v[2]*z.mid
      
      r <- abs(x-X) <= dx/2
      c <- abs(y-Y) <= dy/2
      
      cdf[r,c] <- (stats::pnorm(z.cross[i+1])-stats::pnorm(z.cross[i]))/(sum(r)*sum(c))
    }
  }
  else if(ZERO==2) # point degeneracy
  {
    # the closest point(s)
    r <- abs(X) <= dx/2
    c <- abs(Y) <= dy/2
    
    # increment the closest point(s)
    cdf[r,c] <- cdf[r,c] + 1/(sum(r)*sum(c))
  }
  else stop("something is wrong with this matrix: sigma == ",sigma)
  
  return(cdf)
}

#################
# Newton-Cotes integrators
NewtonCotes <- function(X,Y,sigma,W,dx=mean(diff(X)),dy=mean(diff(Y)))
{
  W <- W/sum(W)
  
  n <- length(W)
  m <- n-1
  
  # refined grid
  x <- seq(X[1]-dx/2,last(X)+dx/2,dx/m)
  y <- seq(Y[1]-dy/2,last(Y)+dy/2,dy/m)

  # weight arrays
  w.x <- array(W[-n]*dx,length(x)) ; w.x[length(x)] <- w.x[1]
  w.y <- array(W[-n]*dy,length(y)) ; w.y[length(y)] <- w.y[1]

  # weight table
  W <- (w.x %o% w.y)

  cdf <- W * Gauss(x,y,sigma)

  # coarsen grid
  # index order is (x,y)
  cdf <- vapply(1:length(X)-1,function(i){colSums(cdf[1:n+m*i,])},rep(0,length(y)))
  # index order is (y,X)
  cdf <- vapply(1:length(Y)-1,function(i){colSums(cdf[1:n+m*i,])},rep(0,length(X)))
  # index order is (X,Y)
  
  return(cdf)
}

#####################
# gaussian pdf
Gauss <- function(X,Y,sigma)
{
  sigma.inv <- solve(sigma)

  cdf <- outer(X^2*sigma.inv[1,1],Y^2*sigma.inv[2,2],"+")/2
  cdf <- cdf + (X %o% Y)*sigma.inv[1,2]
  cdf <- exp(-cdf)/(2*pi*sqrt(det(sigma)))
  return(cdf)
}

#######################
# summarize details of akde object
summary.akde <- function(object,alpha.HR=0.05,...)
{
  area <- c(0,0,0)
  for(i in 1:3)
  {
    area[i] <- sum(object[[i]]$CDF <= 1-alpha.HR) * object[[i]]$dA
  }
  
  unit.info <- unit(area,"area")
  name <- unit.info$name
  scale <- unit.info$scale
  
  area <- array(area/scale,c(1,3))
  colnames(area) <- c("low","ML","high")
  rownames(area) <- paste("area (",name,")",sep="")
  
  return(area)
}
#methods::setMethod("summary",signature(object="akde"), function(object,...) summary.akde(object,...))


################################
# create a raster of the ML akde
raster.akde <- function(AKDE,CI="ML")
{
  kde <- AKDE[[CI]]
  dx <- kde$x[2]-kde$x[1]
  dy <- kde$y[2]-kde$y[1]
  
  xmn <- kde$x[1]-dx/2
  xmx <- last(kde$x)+dx/2
  
  ymn <- kde$y[1]-dy/2
  ymx <- last(kde$y)+dy/2
  
  Raster <- raster::raster(t(kde$PDF[,dim(kde$PDF)[2]:1]),xmn=xmn,xmx=xmx,ymn=ymn,ymx=ymx,crs=attr(AKDE,"info")$projection)
  
  return(Raster)
}


################
# Is contour A inside contour B
inside <- function(A,B)
{
  result <- mode(sp::point.in.polygon(A$x,A$y,B$x,B$y))
  if(1<=result && result<=2) { return(1) } else { return(0) }
}


##############
SpatialPolygonsDataFrame.akde <- function(AKDE,alpha.HR=0.05)
{
  ID <- paste(AKDE@info$identity," ",names(AKDE)," ",round(100*(1-alpha.HR)),"%",sep="")

  polygons <- list()
  for(i in 1:length(AKDE))
  {
    kde <- AKDE[[i]]
    CL <- grDevices::contourLines(x=kde$x,y=kde$y,z=kde$CDF,levels=1-alpha.HR)
    
    # create contour heirarchy matrix (half of it)
    H <- array(0,c(1,1)*length(CL))    
    for(row in 1:length(CL))
    {
      for(col in row:length(CL))
      {
        H[row,col] <- inside(CL[[row]],CL[[col]]) 
      }
    }
    
    # number of contours that this contour is inside
    I <- rowSums(H)
    
    # if I is odd, then you are a hole inside a positive area
    hole <- is.odd(I)
    
    # polygon
    polygons[[i]] <- list()
    for(j in 1:length(CL))
    {
      polygons[[i]][[j]] <- sp::Polygon( cbind( CL[[j]]$x , CL[[j]]$y ) , hole=hole[j] )
    }

    # polygonS
    polygons[[i]] <- sp::Polygons(polygons[[i]],ID=ID[i])
  }
  names(polygons) <- ID

    # spatial polygons
  polygons <- sp::SpatialPolygons(polygons, proj4string=sp::CRS(attr(AKDE,"info")$projection))

  # spatial polygons data frame  
  data <- data.frame(name=rev(ID))
  rownames(data) <- rev(ID)
  polygons <- sp::SpatialPolygonsDataFrame(polygons,data)
  
  return(polygons)
}


################
writeShapefile.akde <- function(AKDE, folder, file=AKDE@info$identity, alpha.HR=0.05,  ...)
{
  SP <- SpatialPolygonsDataFrame.akde(AKDE,alpha.HR=alpha.HR)
  
  rgdal::writeOGR(SP, dsn=folder, layer=file, driver="ESRI Shapefile",...)
}

