#' Calculates AIC for all objects of class crwFit listed as arguments
#' 
#' AIC, delta AIC, and Akaike weights for all models listed as arguments.
#' 

#' 
#' The function can either be executed with a series of 'crwFit' objects (see
#' \code{\link{crwMLE}}) without the '.crwFit' suffix or the function can be
#' called without any arguments and it will search out all 'crwFit' objects in
#' the current workspace and produce the model selection table for all 'crwFit'
#' objects in the workspace. Caution should be used when executing the function
#' in this way. ALL 'crwFit' objects will be included whether ot not the same
#' locations are used!  For all of the models listed as arguments (or in the
#' workspace), AIC, delta AIC, and Akaike weights will be calculated.
#' 
#' @param \dots a series of crwFit objects
#' @return A table, sorted from lowest AIC value to highest.
#' @author Devin S. Johnson
#' @export
"AIC.crwFit" <- function(...)
{
  lnms <- NULL
  models <- list(...)
  if(length(models) == 0) {
      lnms <- list()
      lx <- ls(envir=parent.frame(2))
      for (i in 1:length(lx)) {
          classval <- class(eval(parse(text=lx[i]), envir=parent.frame(2)))
          if("crwFit" %in% classval) lnms <- append(lnms,list(lx[i]))
      }
      models <- eval(parse(text=paste("list(",
                             paste(paste(lnms, "=", lnms, sep=""),
                                   collapse=","), ")")), envir=parent.frame())
      vnms <- do.call("c", lnms)
  } else {
      models <- list(...)
      vnms <- all.vars(match.call())
  }
  num.mod <- length(vnms)
  AIC.vec <- numeric(num.mod)
  ks <- numeric(num.mod)
  for (i in 1:num.mod) {
      AIC.vec[i] <- round(models[[i]]$aic, 2)
      ks[i] <- length(models[[i]]$fixPar) - sum(!is.na(models[[i]]$fixPar))
  }
  deltaAIC <- round(AIC.vec - min(AIC.vec), 2)
  wAIC <- round(exp(-0.5 * deltaAIC) / sum(exp(-0.5 * deltaAIC)), 2)
  ord <- order(deltaAIC)
  out <- data.frame(Name=vnms, k=ks, AIC=AIC.vec,
                    dAIC=deltaAIC, weight=wAIC)
  return(out[ord, ])
}




#' 'Flattening' a list-form crwPredict object into a data.frame
#' 
#' \dQuote{Flattens} a list form \code{\link{crwPredict}} object into a flat
#' data.frame.
#' 
#' 
#' @param predObj A crwPredict object
#' @return a \code{\link{data.frame}} version of a crwPredict list with columns
#' for the state standard errors
#' @author Devin S. Johnson
#' @seealso \code{\link{northernFurSeal}} for use example
#' @export
"as.flat" <- function(predObj)
{
    se.y <- sqrt(t(apply(predObj$V.hat.y, 3, diag)))
    se.x <- sqrt(t(apply(predObj$V.hat.x, 3, diag)))
    colnames(se.y) <- paste("se", names(predObj$alpha.hat.y), sep=".")
    colnames(se.x) <- paste("se", names(predObj$alpha.hat.x), sep=".")
    flat <- cbind(predObj$originalData, predObj$alpha.hat.y, se.y,
                  predObj$alpha.hat.x, se.x)
    if (!is.null(predObj$speed)) flat <- cbind(flat, predObj$speed)
    class(flat) <- c("crwPredict", "data.frame")
    attr(flat, "coord") <- attr(predObj, "coord")
    attr(flat, "random.drift") <- attr(predObj, "random.drift")
    attr(flat, "stop.model") <- attr(predObj, "stop.model")
    attr(flat, "polar.coord") <- attr(predObj, "polar.coord")
    attr(flat, "Time.name") <- attr(predObj, "Time.name")
    attr(flat, "flat") <- TRUE
    return(flat)
}

#' @S3method print crwFit
"print.crwFit" <- function(x, ...)
{
    fit <- x
    cat("\n\n")
    cat("Continuous-Time Correlated Random Walk fit\n\n")
    cat('Models:\n')
    cat("--------\n")
    cat("Movement   "); cat(as.character(fit$mov.model)); cat("\n")
    cat("Error   "); cat(as.character(fit$err.model)); cat("\n")
    if (fit$random.drift | !is.null(fit$stop.model)) cat("with ")
    if (fit$random.drift) cat("Random Drift")
    if (fit$random.drift & !is.null(fit$stop.model)) cat(" and ")
    if (!is.null(fit$stop.model)) cat("Movement Stops")
    cat("\n\n")
    out <- as.data.frame(round(cbind(fit$par, fit$se, fit$ci[, 1], fit$ci[, 2]), 3))
    colnames(out) <- c("Parameter Est.", "St. Err.", "95% Lower", "95% Upper")
    rownames(out) <- fit$nms
    out[!is.na(fit$fixPar), 2:4] <- "."
    print(out)
    cat("\n\n")
    cat(paste("Log Likelihood =", round(fit$loglik, 3),"\n", collapse=""))
    cat(paste("AIC =", round(fit$aic, 3),"\n", collapse=""))
    cat("\n\n\n")
}
