#' @title Extract Weights from Preprocessing Objects
#' 
#' @description Extracts weights from the outputs of preprocessing functions.
#' 
#' @param x output from the corresponding preprocessing packages.
#' @param stop.method the name of the stop method used in the original call to `ps()` or `mnps()` in \pkg{twang}, e.g., `"es.mean"`. If empty, will return weights from all stop method available into a data.frame. Abbreviations allowed.
#' @param estimand if weights are computed using the propensity score (i.e., for the `ps` and `CBPS` methods), which estimand to use to compute the weights. If `"ATE"`, weights will be computed as `1/ps` for the treated group and `1/(1-ps)` for the control group. If `"ATT"`, weights will be computed as `1` for the treated group and `ps/(1-ps)` for the control group. If not specified, `get.w()` will try to figure out which estimand is desired based on the object.
#' 
#' If weights are computed using subclasses/matching strata (i.e., for the `cem` and `designmatch` methods), which estimand to use to compute the weights. First, a subclass propensity score is computed as the proportion of treated units in each subclass, and the one of the formulas above will be used based on the estimand requested. If not specified, `"ATT"` is assumed.
#' @param treat a vector of treatment status for each unit. This is required for methods that include `treat` as an argument. The treatment variable that was used in the original preprocessing function call should be used.
#' @param s.weights whether the sampling weights included in the original call to the fitting function should be included in the weights. If `TRUE`, the returned weights will be the product of the balancing weights estimated by the fitting function and the sampling weights. If `FALSE`, only the balancing weights will be returned.
#' @param ... arguments passed to other methods.
#' 
#' @returns A vector or data frame of weights for each unit. These may be matching weights or balancing weights.
#' 
#' @details The output of `get.w()` can be used in calls to the formula and data frame methods of [bal.tab()] (see example below). In this way, the output of multiple preprocessing packages can be viewed simultaneously and compared. The weights can also be used in `weights` statements in regression methods to compute weighted effects.
#' 
#' \pkg{twang} has a function called `get.weights()` that performs the same function on `ps` objects but offers slightly finer control. Note that the weights generated by `get.w()` for `ps` objects do not include sampling weights by default.
#' 
#' When sampling weights are used with `CBPS()` in \pkg{CBPS}, the returned weights will already have the sampling weights incorporated. To retrieve the balancing weights on their own, divide the returned weights by the original sampling weights. For other packages, the balancing weights are returned separately unless `s.weights = TRUE`, which means they must be multiplied by the sampling weights for effect estimation.
#' 
#' When `Match()` in \pkg{Matching} is used with `CommonSupport = TRUE`, the returned weights will be incorrect. This option is not recommended by the package authors.
#' 
#' @examplesIf all(sapply(c("WeightIt", "MatchIt"), requireNamespace, quietly = TRUE))
#' data("lalonde", package = "cobalt")
#' 
#' m.out <- MatchIt::matchit(treat ~ age + educ + race,
#'                           data = lalonde,
#'                           estimand = "ATT") 
#' 
#' w.out <- WeightIt::weightit(treat ~ age + educ + race,
#'                             data = lalonde,
#'                             estimand = "ATT")
#' 
#' bal.tab(treat ~ age + educ + race, data = lalonde,
#'         weights = data.frame(matched = get.w(m.out),
#'                              weighted = get.w(w.out)),
#'         method = c("matching", "weighting"), 
#'         estimand = "ATT")

#' @rdname get.w
#' @export 
get.w <- function(x, ...) {
  if (!inherits(x, "cobalt.processed.obj")) {
    x <- process_obj(x)
    return(get.w(x, ...))
  }
  
  UseMethod("get.w")
}

#' @rdname get.w
#' @exportS3Method get.w matchit
get.w.matchit <- function(x,...) {
  x[["weights"]]
}

#' @rdname get.w
#' @exportS3Method get.w ps
get.w.ps <- function(x, stop.method = NULL, estimand, s.weights = FALSE, ...) {
  
  if (missing(estimand)) estimand <- NULL
  else estimand <- tolower(estimand)
  
  if (is_null(stop.method)) {
    rule1 <- names(x[["w"]])
  }
  else if (any(is.character(stop.method))) {
    rule1 <- names(x$w)[vapply(tolower(names(x$w)), function(x) any(startsWith(x, tolower(stop.method))), logical(1L))]
    if (is_null(rule1)) {
      .wrn(sprintf("`stop.method` should be %s. Using all available stop methods instead",
                   word_list(names(x$w), and.or = "or", quotes = 2L)))
      rule1 <- names(x$w)
    }
  }
  else if (is.numeric(stop.method) && any(stop.method %in% seq_along(names(x$w)))) {
    if (!all(stop.method %in% seq_along(names(x$w)))) {
      .wrn(sprintf("there are %s stop methods available, but you requested %s", 
                   length(names(x$w)),
                   word_list(setdiff(stop.method, seq_along(names(x$w))), and.or = "and")))
    }
    rule1 <- names(x$w)[stop.method %in% seq_along(names(x$w))]
  }
  else {
    .wrn(sprintf("`stop.method` should be %s. Using all available stop methods instead",
                 word_list(names(x$w), and.or = "or", quotes = 2L)))
    rule1 <- names(x$w)
  }
  
  s <- names(x$w)[match(tolower(rule1), tolower(names(x$w)))]
  criterion <- substr(tolower(s), 1L, nchar(s) - 4L)
  allowable.estimands <- c("ATT", "ATE", "ATC")
  
  if (is_null(estimand)) {
    estimand <- setNames(substr(toupper(s), nchar(s) - 2L, nchar(s)), s)
  }
  else if (!all(toupper(estimand) %in% allowable.estimands)) {
    .err(sprintf("`estimand` must be %s", word_list(allowable.estimands, "or", quotes = 1L)))
  }
  else if (length(estimand) == 1L) {
    estimand <- setNames(toupper(rep.int(estimand, length(s))), s)
  }
  else if (length(estimand) >= length(s)) {
    estimand <- setNames(toupper(estimand[seq_along(s)]), s)
  }
  else {
    .err("`estimand` must be the same length as the number of sets of weights requested")
  }
  
  w <- make_df(s, nrow(x$ps))
  w[] <- 1
  t1 <- x$treat == 1
  
  for (p in s) {
    if (estimand[p] == "ATE") {
      w[[p]][t1] <- 1 / x$ps[t1, p]
      w[[p]][!t1] <- 1 / (1 - x$ps[!t1, p])
    }
    else if (estimand[p] == "ATT") {
      w[[p]][!t1] <- x$ps[!t1, p] / (1 - x$ps[!t1, p])
    }
    else if (estimand[p] == "ATT") {
      w[[p]][t1] <- (1 - x$ps[t1, p]) / x$ps[t1, p]
    }
    else {
      w[[p]] <- x$w[,p]
    }
    
    if (s.weights) w[[p]] <- w[[p]] * x$sampw
  }
  
  if (ncol(w) == 1L) {
    return(w[[1L]])
  }
  
  names(w) <- ifelse(toupper(substr(s, nchar(s) - 2L, nchar(s))) == estimand,
                     criterion,
                     sprintf("%s (%s)", criterion, estimand))
  
  w
}

#' @rdname get.w
#' @exportS3Method get.w mnps
get.w.mnps <- function(x, stop.method = NULL, s.weights = FALSE, ...) {
  
  if (is_null(stop.method)) {
    rule1 <- x$stopMethods
  }
  else if (is.character(stop.method)) {
    rule1 <- names(x$w)[vapply(tolower(names(x$w)), function(x) any(startsWith(x, tolower(stop.method))), logical(1L))]
    if (is_null(rule1)) {
      .wrn(sprintf("`stop.method` should be %s. Using all available stop methods instead",
                   word_list(x$stopMethods, and.or = "or", quotes = 2L)))
      rule1 <- x$stopMethods
    }
  }
  else if (is.numeric(stop.method) && any(stop.method %in% seq_along(x$stopMethods))) {
    if (!all(stop.method %in% seq_along(x$stopMethods))) {
      .wrn(sprintf("there are %s stop methods available, but you requested %s",
                   length(x$stopMethods), 
                   word_list(setdiff(stop.method, seq_along(x$stopMethods)), and.or = "and")))
    }
    rule1 <- x$stopMethods[stop.method %in% seq_along(x$stopMethods)]
  }
  else {
    .wrn(sprintf("`stop.method` should be %s. Using all available stop methods instead",
                 word_list(x$stopMethods, and.or = "or", quotes = 2L)))
    rule1 <- x$stopMethods
  }
  
  s <- paste.(x$stopMethods[match(tolower(rule1), tolower(x$stopMethods))],
              x$estimand)
  
  estimand <- x$estimand
  criterion <- x$stopMethods[match(tolower(rule1), tolower(x$stopMethods))]
  
  w <- make_df(criterion, length(x$treatVar))
  w[] <- 1
  
  if (estimand == "ATT") {
    for (i in x$levExceptTreatATT) {
      w[x$treatVar == i, criterion] <- {
        if (length(s) > 1L)
          get.w.ps(x$psList[[i]], stop.method = criterion)[x$psList[[i]]$treat == 0, criterion]
        else
          get.w.ps(x$psList[[i]], stop.method = criterion)[x$psList[[i]]$treat == 0]
      }
    }
  }
  else if (estimand == "ATE") {
    for (i in x$treatLev) {
      w[x$treatVar == i, criterion] <- {
        if (length(s) > 1L)
          get.w.ps(x$psList[[i]], stop.method = criterion)[x$psList[[i]]$treat == 1, criterion]
        else
          get.w.ps(x$psList[[i]], stop.method = criterion)[x$psList[[i]]$treat == 1]
      }
    }
  }
  
  if (s.weights) {
    w <- w * x[["sampw"]]
  }
  
  if (ncol(w) == 1L) {
    return(w[[1L]])
  }
  
  names(w) <- ifelse(toupper(substr(s, nchar(s) - 2L, nchar(s))) == estimand, criterion,
                     paste0(criterion, " (", estimand, ")"))
  
  w
}

#' @rdname get.w
#' @exportS3Method get.w ps.cont
get.w.ps.cont <- function(x, s.weights = FALSE, ...) {
  if (isTRUE(s.weights)) {
    return(x$w * x$sampw)
  }
  
  x$w
}

#' @rdname get.w
#' @exportS3Method get.w iptw
get.w.iptw <- function(x, stop.method = NULL, s.weights = FALSE, ...) {
  
  if (is_null(stop.method)) {
    rule1 <- names(x$psList[[1L]]$ps)
  }
  else if (any(is.character(stop.method))) {
    rule1 <- names(x$psList[[1L]]$ps)[vapply(tolower(names(x$psList[[1L]]$ps)),
                                             function(x) any(startsWith(x, tolower(stop.method))), logical(1L))]
    if (is_null(rule1)) {
      .wrn(sprintf("`stop.method` should be %s. Using all available stop methods instead",
                   word_list(names(x$psList[[1L]]$ps), and.or = "or", quotes = 2)))
      rule1 <- names(x$psList[[1L]]$ps)
    }
  }
  else if (is.numeric(stop.method) && any(stop.method %in% seq_along(names(x$psList[[1L]]$ps)))) {
    if (!all(stop.method %in% seq_along(names(x$psList[[1L]]$ps)))) {
      .wrn(sprintf("there are %s stop methods available, but you requested %s",
           length(names(x$psList[[1L]]$ps)), 
           word_list(setdiff(stop.method, seq_along(names(x$psList[[1L]]$ps))), and.or = "and")))
    }
    rule1 <- names(x$psList[[1L]]$ps)[stop.method %in% seq_along(names(x$psList[[1L]]$ps))]
  }
  else {
    .wrn(sprintf("`stop.method` should be %s. Using all available stop methods instead",
                 word_list(names(x$psList[[1L]]$ps), and.or = "or", quotes = 2L)))
    rule1 <- names(x$psList[[1L]]$ps)
  }
  
  w <- make_df(rule1, nrow = nrow(x$psList[[1L]]$ps))
  
  for (i in rule1) {
    w[i] <- Reduce("*", lapply(x$psList, get.w.ps, stop.method = i))
  }
  
  if (s.weights) {
    w <- w * x$psList[[1L]]$sampw
  }
  
  w
}

#' @rdname get.w
#' @exportS3Method get.w Match
get.w.Match <- function(x, ...) {
  # x$weights <- x$weights / ave(x$weights, x$index.treated, FUN = sum)
  vapply(seq_len(x$orig.nobs), function(i) {
    sum(x$weights[x$index.treated == i | x$index.control == i])
  }, numeric(1L))
}

#' @rdname get.w
#' @exportS3Method get.w CBPS
get.w.CBPS <- function(x, estimand, ...) {
  
  if (inherits(x, "CBPSContinuous") || inherits(x, "npCBPS") || is.factor(x$y)) { #continuous, multi, or npCBPS
    return(x$weights)
  }
  
  use.weights <- isTRUE(...get("use.weights", TRUE))
  
  if (use.weights) {
    return(x$weights)
  }
  
  if (missing(estimand)) estimand <- NULL
  else estimand <- tolower(estimand)
  
  ps <- x$fitted.values
  tr <- x$y 
  t1 <- tr == 1
  
  if (is_null(estimand)) {
    estimand <- {
      if (all_the_same(x$weights[t1])) "att"
      else if (all_the_same(x$weights[!t1])) "atc"
      else "ate"
    }
  }
  
  estimand <- match_arg(tolower(estimand), c("att", "atc", "ate"))
  
  w <- rep.int(1, length(tr))
  
  if (estimand == "att") {
    w[!t1] <- ps[!t1] / (1 - ps[!t1])
  }
  else if (estimand == "atc") {
    w[t1] <- (1 - ps[t1]) / ps[t1]
  }
  else {
    w[t1] <- 1 / ps[t1]
    w[!t1] <- 1 / (1 - ps[!t1])
  }
  
  w
}

#' @rdname get.w
#' @exportS3Method get.w CBMSM
get.w.CBMSM <- function(x, ...) {
  x$weights[sort(unique(x$id))]
}

#' @rdname get.w
#' @exportS3Method get.w ebalance
get.w.ebalance <- function(x, treat, ...) {
  .chk_not_missing(treat, "`treat`")
  
  if (!inherits(treat, "processed.treat")) {
    treat <- process_treat(treat)
  }
  
  if (length(x$w) != sum(treat == treat_vals(treat)["Control"])) {
    .err("there are more control units in `treat` than weights in the `ebalance` object.")
  }
  
  weights <- rep.int(1, length(treat))
  
  weights[treat == treat_vals(treat)["Control"]] <- x$w
  
  weights
}

#' @rdname get.w
#' @exportS3Method get.w optmatch
get.w.optmatch <- function(x, estimand, ...) {
  if (missing(estimand) || is_null(estimand)) estimand <- "ATT"
  treat <- as.numeric(attr(x, "contrast.group"))
  strata2weights(x, treat = treat, estimand = estimand)
}

#' @rdname get.w
#' @exportS3Method get.w cem.match
get.w.cem.match <- function(x, estimand, ...) {
  if (missing(estimand) || is_null(estimand)) {
    estimand <- "ATT"
  }
  
  if (isTRUE(...get("use.match.strata"))) {
    if (inherits(x, "cem.match.list")) {
      return(unlist(lapply(x[vapply(x, inherits, logical(1L), "cem.match")],
                           function(cm) strata2weights(cm[["mstrata"]], treat = cm[["groups"]], estimand = estimand)),
                    use.names = FALSE))
    }
    
    return(strata2weights(x[["mstrata"]], treat = x[["groups"]], estimand = estimand))
  }
  
  if (inherits(x, "cem.match.list")) {
    return(unlist(grab(x[vapply(x, inherits, logical(1L), "cem.match")], "w"), use.names = FALSE))
  }
  
  x[["w"]]
}

#' @rdname get.w
#' @exportS3Method get.w weightit
get.w.weightit <- function(x, s.weights = FALSE, ...) {
  if (isTRUE(s.weights)) {
    return(x[["weights"]] * x[["s.weights"]])
  }
  
  x[["weights"]]
}

#' @rdname get.w
#' @exportS3Method get.w designmatch
get.w.designmatch <- function(x, treat, estimand, ...) {
  .chk_not_missing(treat, "`treat`")
  
  if (missing(estimand) || is_null(estimand)) {
    estimand <- "ATT"
  }
  
  if (length(x[["group_id"]]) != length(x[["t_id"]]) + length(x[["c_id"]])) {
    .err("`designmatch` objects without 1:1 matching cannot be used")
  }
  
  q <- merge(data.frame(id = seq_along(treat)), 
             data.frame(id = c(x[["t_id"]], x[["c_id"]]),
                        group = factor(x[["group_id"]])),
             all.x = TRUE, by = "id")
  
  q <- q[order(q$id), , drop = FALSE]
  
  strata2weights(q$group, treat, estimand)
}

#' @rdname get.w
#' @exportS3Method get.w mimids
get.w.mimids <- function(x, ...) {
  old_version <- !all(c("object", "models", "approach") %in% names(x))
  
  weights <- {
    if (old_version) unlist(lapply(x[["models"]][-1L], get.w.matchit))
    else unlist(lapply(x[["models"]], get.w.matchit))
  }
  
  if (anyNA(weights)) {
    weights[is.na(weights)] <- 0
  }
  
  weights
}

#' @rdname get.w
#' @exportS3Method get.w wimids
get.w.wimids <- function(x, ...) {
  old_version <- !all(c("object", "models", "approach") %in% names(x))
  
  weights <- {
    if (old_version) unlist(lapply(x[["models"]][-1L], get.w.weightit))
    else unlist(lapply(x[["models"]], get.w.weightit))
  }
  
  if (anyNA(weights)) {
    weights[is.na(weights)] <- 0
  }
  
  weights
}

#' @rdname get.w
#' @exportS3Method get.w sbwcau
get.w.sbwcau <- function(x, ...) {
  x[["dat_weights"]][[ncol(x[["dat_weights"]])]]
}
