% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/varselbest.R
\name{varselbest}
\alias{varselbest}
\title{Variable selection for specifying conditional imputation models}
\usage{
varselbest(
  data.na = NULL,
  res.imputedata = NULL,
  listvar = NULL,
  nb.clust = NULL,
  nnodes = 1,
  sizeblock = 5,
  method.select = "knockoff",
  B = 200,
  r = 0.3,
  graph = TRUE,
  printflag = TRUE,
  path.outfile = NULL,
  mar = c(2, 4, 2, 0.5) + 0.1,
  cex.names = 0.7,
  modelNames = NULL
)
}
\arguments{
\item{data.na}{a dataframe with only numeric variables}

\item{res.imputedata}{an output from \code{\link{imputedata}}}

\item{listvar}{a character vector indicating for which subset of incomplete variables variable selection must be performed. By default all column names.}

\item{nb.clust}{the number of clusters used for imputation}

\item{nnodes}{number of CPU cores for parallel computing. By default, nnodes = 1}

\item{sizeblock}{an integer indicating the number of variables sampled at each iteration}

\item{method.select}{a single string indicating the variable selection method applied on each subset of variables}

\item{B}{number of iterations, by default B = 200}

\item{r}{a numerical vector (or a single real number) indicating the threshold used for each variable in listvar. Each value of r should be between 0 and 1. See details.}

\item{graph}{a boolean. If TRUE two graphics are plotted per variable in \code{listvar}: a graphic reporting the variable importance measure of each explanatory variable and a graphic reporting the influence of the number iterations (B) on the importance measures}

\item{printflag}{a boolean. If TRUE, a message is printed at each iteration. Use printflag = FALSE for silent selection.}

\item{path.outfile}{a vector of strings indicating the path for redirection of print messages. Default value is NULL, meaning that silent imputation is performed. Otherwise, print messages are saved in the files path.outfile/output.txt. One file per node is generated.}

\item{mar}{a numerical vector of the form c(bottom, left, top, right). Only used if graph = TRUE}

\item{cex.names}{expansion factor for axis names (bar labels) (only used if graph = TRUE)}

\item{modelNames}{a vector of character strings indicating the models to be fitted in the EM phase of clustering}
}
\value{
a list of four objects
\item{predictormatrix}{a numeric matrix containing 0 and 1 specifying on each line the set of predictors to be used for each target column of the incomplete dataset.}
\item{res.varsel}{a list given details on the variable selection procedure (only required for checking convergence by the \code{chooseB} function)}
\item{proportion}{a numeric matrix of proportion indicating on each line the variable importance of each predictor}
\item{call}{the matching call}
}
\description{
\code{varselbest} performs variable selection from an incomplete dataset (see Bar-Hen and Audigier (2022) <doi:10.1080/00949655.2022.2070621>) in order to specify the imputation models to use for FCS imputation methods
}
\details{
\code{varselbest} performs variable selection on random subsets of variables and, then, combines them to recover which explanatory variables are related to the response.
More precisely, the outline of the algorithm are as follows: let consider a random subset of \code{sizeblock} among p variables.
By choosing \code{sizeblock} small, this subset is low dimensional, allowing treatment of missing values by standard imputation method for clustered individuals.
Then, any selection variable scheme can be applied (lasso, stepwise and knockoff are proposed by tuning the \code{method.select} argument).
By resampling \code{B} times, a sample of size \code{sizeblock} among the p variables, we may count how many times, a variable is considered as significantly related to the response and how many times it is not.
We need to define a threshold (\code{r}) to conclude if a given variable is significantly related to the response.
}
\examples{
data(wine)

require(parallel)
set.seed(123456)
ref <- wine$cult
nb.clust <- 3
wine.na <- wine
wine.na$cult <- NULL
wine.na <- as.matrix(wine.na)
wine.na[sample(seq(length(wine.na)), size = ceiling(length(wine.na)/3))] <- NA

\donttest{
nnodes <- 2 # parallel::detectCores()
B <- 150 #  Number of iterations
m <- 5 # Number of imputed data sets

# variable selection
res.varsel <- varselbest(data.na = wine.na,
                         nb.clust = nb.clust,
                         listvar = c("alco","malic"),
                         B = B,
                         nnodes = nnodes)
predictmat <- res.varsel$predictormatrix

# imputation
res.imp.select <- imputedata(data.na = wine.na, method = "FCS-homo",
                     nb.clust = nb.clust, predictmat = predictmat, m = m)
}

}
\references{
Bar-Hen, A. and Audigier, V., An ensemble learning method for variable selection: application to high dimensional data and missing values, Journal of Statistical Computation and Simulation, <doi:10.1080/00949655.2022.2070621>, 2022.
}
\seealso{
\code{\link[mice]{mice}},\code{\link{clusterMI}}, \code{\link{imputedata}},\code{\link[knockoff]{knockoff}},\code{\link[glmnet]{glmnet}},\code{\link[mix]{imp.mix}}
}
