% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cdcov.R
\name{cdcov.test}
\alias{cdcov.test}
\title{Conditional Distance Covariance Independence Test}
\usage{
cdcov.test(x, y, z, num.bootstrap = 99, width, distance = FALSE,
  index = 1, seed = 1, num.threads = 1)
}
\arguments{
\item{x}{a numeric vector, matrix, or \code{dist} object}

\item{y}{a numeric vector, matrix, or \code{dist} object}

\item{z}{\code{z} is a numeric vector or matrix. It is the variable being conditioned.}

\item{num.bootstrap}{the number of local bootstrap procedure replications. Default: \code{num.bootstrap = 99}.}

\item{width}{a user-specified positive value (univariate conditional variable) or vector (multivariate conditional variable) for 
gaussian kernel bandwidth. Its default value is relies on \code{stats::bw.nrd0} function when conditional variable is univariate, 
\code{ks::Hpi.diag} when conditional variable with at most trivariate, and \code{stats::bw.nrd} on the other cases.}

\item{distance}{if \code{distance = TRUE}, \code{x} and \code{y} will be considered as distance matrices. Default: \code{distance = FALSE}.}

\item{index}{exponent on Euclidean distance, in \eqn{(0,2]}}

\item{seed}{the random seed}

\item{num.threads}{number of threads. Default \code{num.threads = 1}.}
}
\value{
\code{cdcov.test} returns a list with class "htest" containing the following components:
\item{\code{statistic}}{conditional distance covariance statistic.}            
\item{\code{p.value}}{the \eqn{p}-value for the test.}
\item{\code{replicates}}{the number of local bootstrap procedure replications.}
\item{\code{size}}{sample sizes.}
\item{\code{alternative}}{a character string describing the alternative hypothesis.}
\item{\code{method}}{a character string indicating what type of test was performed.}
\item{\code{data.name}}{description of data.}
}
\description{
Performs the nonparametric conditional distance covariance test for conditional independence assumption
}
\examples{
\dontrun{

library(cdcsis)
set.seed(1)
num <- 50
################# Conditional Independent #################
## Case 1:
cov_mat <- matrix(c(1, 0.36, 0.6, 0.36, 1, 0.6, 0.6, 0.6, 1), nrow = 3)
dat <- mvtnorm::rmvnorm(n = num, sigma = cov_mat)
x <- dat[, 1]
y <- dat[, 2]
z <- dat[, 3]
cdcov.test(x, y, z)
## Case 2:
z <- rnorm(num)
x <- 0.5 * (z^3 / 7 + z / 2) + tanh(rnorm(num))
x <- x + x^3 / 3
y <- (z^3 + z) / 3 + rnorm(num)
y <- y + tanh(y / 3)
cdcov.test(x, y, z, num.bootstrap = 99)

################# Conditional Dependent #################
## Case 3:
cov_mat <- matrix(c(1, 0.7, 0.6, 0.7, 1, 0.6, 0.6, 0.6, 1), nrow = 3)
dat <- mvtnorm::rmvnorm(n = num, sigma = cov_mat)
x <- dat[, 1]
y <- dat[, 2]
z <- dat[, 3]
cdcov.test(x, y, z, width = 0.5)
## Case 4:
z <- matrix(rt(num * 4, df = 2), nrow = num)
x <- z
y <- cbind(sin(z[, 1]) + cos(z[, 2]) + (z[, 3])^2 + (z[, 4])^2, 
           (z[, 1])^2 + (z[, 2])^2 + z[, 3] + z[, 4])
z <- z[, 1:2]
cdcov.test(x, y, z, seed = 2)

################# Distance Matrix Input #################
x <- dist(x)
y <- dist(y)
cdcov.test(x, y, z, seed = 2, distance = TRUE)
}
}
\references{
Wang, X., Pan, W., Hu, W., Tian, Y. and Zhang, H., 2015. Conditional distance correlation. Journal of the American Statistical Association, 110(512), pp.1726-1734.
}
\seealso{
\code{\link{cdcov}}
}
