#' @title export.detections
#'
#' @description Exports detection table from `call.detect` into a txt file
#' that can be read by Raven Lite. All columns other than those containing
#' start and end times are filled with 0 or ''.
#'
#' @param detections data.frame, the object generated by `call.detect`.
#' @param sr numeric, the sampling rate of the wave on which detections were
#' run. Default to `1`, which allows users to transform the start and end times
#' before feeding the data.frame to this function.
#' @param path_out character, the path including file name where to store the
#' txt file. Default is `'out.txt'`.
#'
#' @return Stores a Raven Lite readable selection table.
#'
#' @export
#' @importFrom utils "write.table"

export.detections = function(detections, sr = 1, path_out = 'out.txt'){

  if(path_out == 'out.txt')
    warning('No path supplied.
            Storing as out.txt in current working directory.')
  if(!str_detect(path_out, '.txt'))
    stop('Path needs to end on `.txt`.')

  # Make empty data frame if no detections
  if(nrow(detections) == 0) {
    out = data.frame(numeric(),
                     character(),
                     numeric(),
                     numeric(),
                     numeric(),
                     numeric(),
                     numeric(),
                     numeric(),
                     numeric(),
                     numeric(),
                     character())
  } else {
    # Else make actual thing
    out = data.frame(c(seq_len(nrow(detections)), seq_len(nrow(detections))),
                     c('Waveform 1', 'Spectrogram 1'),
                     c(1, 1),
                     c(detections$start/sr, detections$start/sr),
                     c(detections$end/sr, detections$end/sr),
                     c(0, 0),
                     c(0, 0),
                     c(detections$end/sr - detections$start/sr,
                       detections$end/sr - detections$start/sr),
                     c(0, 0),
                     c(0, 0),
                     c('', ''))
  }
  names(out) = c('Selection', 'View', 'Channel', 'Begin Time (s)',
                 'End Time (s)', 'Low Freq (Hz)', 'High Freq (Hz)',
                 'Delta Time (s)', 'Delta Freq (Hz)',
                 'Avg Power Density (dB FS/Hz)', 'Annotation')

  write.table(out, path_out, sep = '\t', row.names = FALSE, quote = FALSE)

}
