% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bspline.R
\name{smbsp}
\alias{smbsp}
\alias{fitsmbsp}
\title{Smoothing B-spline of order n >= 0}
\usage{
smbsp(x, y, n = 3L, xki = NULL, nki = 1L, lieq = NULL, monotone = 0)

fitsmbsp(
  x,
  y,
  n = 3L,
  xki = NULL,
  nki = 1L,
  lieq = NULL,
  monotone = 0,
  control = list()
)
}
\arguments{
\item{x}{Numeric vector, abscissa points}

\item{y}{Numeric vector or matrix or data.frame, ordinate values to be smoothed
(one set per column in case of matrix or data.frame)}

\item{n}{Integer scalar, polynomial order of B-splines (3 by default)}

\item{xki}{Numeric vector, strictly internal B-spline knots, i.e. lying strictly
inside of \code{x} bounds. If NULL (by default), they are
estimated with the help of \code{iknots()}. This vector is used as initial approximation
during optimization process. Must be non decreasing if not NULL.}

\item{nki}{Integer scalar, internal knot number (1 by default). When
nki==0, it corresponds to polynomial regression. If \code{xki}
is not NULL, this parameter is ignored.}

\item{lieq}{List, equality constraints to respect by the smoothing spline,
one list item per y column. By default (NULL), no constraint is imposed.
Constraints are given as a 2-column matrix \code{(xe, ye)} where
for each xe, an ye value is imposed. If a list item is NULL, no constraint
is imposed on corresponding y column.}

\item{monotone}{Numeric scalar, if monotone > 0, resulting B-spline
must be increasing;
if monotone < 0, B-spline must be decreasing; if monotone == 0 (default), no
constraint on monotonicity is imposed.}

\item{control}{List, passed through to \code{nlsic()} call}
}
\value{
Function, smoothing B-splines
  respecting optional constraints (generated by \code{par2bsp()}).
}
\description{
Optimize smoothing B-spline coefficients (smbsp) and knot positions (fitsmbsp)
such that residual squared sum is minimized for all y columns.
}
\details{
If constraints are set, we use \code{nlsic::lsie_ln()} to solve a
  least squares
  problem with equality constraints in least norm sens for each y column.
  Otherwise, \code{nlsic::ls_ln_svd()} is used for the whole y matrix.
  The solution of least squares problem is a vector of B-splines coefficients \code{qw},
  one vector per \code{y} column. These vectors are used to define B-spline function
  which is returned as the result.\cr\cr
  NB. When \code{nki >= length(x)-n-1} (be it from direct setting or calculated
  from \code{length(xki)}), it corresponds
  to spline interpolation, i.e. the resulting spline will pass
  exactly by (x,y) points (well, up to numerical precision).

Border and external knots are fixed, only strictly internal knots can move
during optimization. The optimization process is constrained to respect a minimal
distance between knots as well as to bound them to x range.
This is done to avoid knots getting unsorted during iterations and/or going
outside of a meaningful range.
}
\examples{
  x=seq(0, 1, length.out=11)
  y=sin(pi*x)+rnorm(x, sd=0.1)
  # constraint B-spline to be 0 at the interval ends
  fsm=smbsp(x, y, nki=1, lieq=list(rbind(c(0, 0), c(1, 0))))
  # check parameters of found B-splines
  bsppar(fsm)
  plot(x, y) # original "measurements"
  # fine grained x
  xfine=seq(0, 1, length.out=101)
  lines(xfine, fsm(xfine)) # fitted B-splines
  lines(xfine, sin(pi*xfine), col="blue") # original function
  # visualize knot positions
  xk=bsppar(fsm)$xk
  points(xk, fsm(xk), pch="x", col="red")
 # fit broken line with linear B-splines
 x1=seq(0, 1, length.out=11)
 x2=seq(1, 3, length.out=21)
 x3=seq(3, 4, len=11)
 y1=x1+rnorm(x1, sd=0.1)
 y2=-2+3*x2+rnorm(x2, sd=0.1)
 y3=4+x3+rnorm(x3, sd=0.1)
 x=c(x1, x2[-1], x3[-1])
 y=c(y1, y2[-1], y3[-1])
 plot(x, y)
 if (requireNamespace("numDeriv", quietly=TRUE)) {
    f=fitsmbsp(x, y, n=1, nki=2)
    lines(x, f(x))
 }
}
\seealso{
\code{bsppar} for retrieving parameters of B-spline functions; \code{par2bsp}
  for generating B-spline function; \code{iknots} for estimation of knot positions
}
