% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/draw_all_admix.R
\name{draw_all_admix}
\alias{draw_all_admix}
\title{Simulate random allele frequencies and genotypes from the BN-PSD admixture model}
\usage{
draw_all_admix(
  admix_proportions,
  inbr_subpops,
  m_loci,
  want_genotypes = TRUE,
  want_p_ind = FALSE,
  want_p_subpops = FALSE,
  want_p_anc = TRUE,
  verbose = FALSE,
  require_polymorphic_loci = TRUE,
  beta = NA,
  p_anc = NULL
)
}
\arguments{
\item{admix_proportions}{The \code{n}-by-\code{k} matrix of admixture proportions.}

\item{inbr_subpops}{The length-\code{k} vector (or scalar) of intermediate subpopulation FST values.}

\item{m_loci}{The number of loci to draw.}

\item{want_genotypes}{If \code{TRUE} (default), includes the matrix of random genotypes in the return list.}

\item{want_p_ind}{If \code{TRUE} (NOT default), includes the matrix of individual-specific allele frequencies in the return list.
Note that by default \code{p_ind} is not constructed in full at all, instead a fast low-memory algorithm constructs it in parts as needed only; beware that setting \code{want_p_ind = TRUE} increases memory usage in comparison.}

\item{want_p_subpops}{If \code{TRUE} (NOT default), includes the matrix of random intermediate subpopulation allele frequencies in the return list.}

\item{want_p_anc}{If \code{TRUE} (default), includes the vector of random ancestral allele frequencies in the return list.}

\item{verbose}{If \code{TRUE}, prints messages for every stage in the algorithm.}

\item{require_polymorphic_loci}{If TRUE (default), returned genotype matrix will not include any fixed loci (loci that happened to be fixed are drawn again, starting from their ancestral allele frequencies, and checked iteratively until no fixed loci remain, so that the final number of polymorphic loci is exactly \code{m_loci}).}

\item{beta}{Shape parameter for a symmetric Beta for ancestral allele frequencies \code{p_anc}.
If \code{NA} (default), \code{p_anc} is uniform with range in [0.01, 0.5].
Otherwise, \code{p_anc} has a symmetric Beta distribution with range in [0, 1].}

\item{p_anc}{If provided, it is used as the ancestral allele frequencies (instead of drawing random ones).  Must either be a scalar or a length-\code{m_loci} vector.
If scalar and \code{want_p_anc = TRUE}, then the returned \code{p_anc} is the scalar value repeated \code{m_loci} times (it is always a vector).}
}
\value{
A named list with the following items (which may be missing depending on options):
\itemize{
\item \code{X}: An \code{m}-by-\code{n} matrix of genotypes.
Included if \code{want_genotypes = TRUE}.
\item \code{p_anc}: A length-\code{m} vector of ancestral allele frequencies.
Included if \code{want_p_anc = TRUE}.
\item \code{p_subpops}: An \code{m}-by-\code{k} matrix of intermediate subpopulation allele frequencies
Included if \code{want_p_subpops = TRUE}.
\item \code{p_ind}: An \code{m}-by-\code{n} matrix of individual-specific allele frequencies.
Included if \code{want_p_ind = TRUE}.
}
}
\description{
This function returns simulated ancestral, intermediate, and individual-specific allele frequencies and genotypes given the admixture structure, as determined by the admixture proportions and the vector of intermediate subpopulation FST values.
The function is a wrapper around \verb{\link{draw_p_anc}}, \verb{\link{draw_p_subpops}}, \verb{\link{make_p_ind_admix}}, and \verb{\link{draw_genotypes_admix}} with additional features such as requiring polymorphic loci.
Importantly, by default fixed loci (where all individuals were homozygous for the same allele) are re-drawn from the start (starting from the ancestral allele frequencies) so no fixed loci are in the output and no biases are introduced by re-drawing genotypes conditional on any of the previous allele frequencies (ancestral, intermediate, or individual-specific).
Below \code{m_loci} (also \code{m}) is the number of loci, \code{n} is the number of individuals, and \code{k} is the number of intermediate subpopulations.
}
\examples{
# dimensions
# number of loci
m_loci <- 10
# number of individuals
n_ind <- 5
# number of intermediate subpops
k_subpops <- 2

# define population structure
# FST values for k = 2 subpopulations
inbr_subpops <- c(0.1, 0.3)
# admixture proportions from 1D geography
admix_proportions <- admix_prop_1d_linear(n_ind, k_subpops, sigma = 1)

# draw all random allele freqs and genotypes
out <- draw_all_admix(admix_proportions, inbr_subpops, m_loci)

# return value is a list with these items:

# genotypes
X <- out$X

# ancestral AFs
p_anc <- out$p_anc

# # these are excluded by default, but would be included if ...
# # ... `want_p_subpops == TRUE`
# # intermediate subpopulation AFs
# p_subpops <- out$p_subpops
# 
# # ... `want_p_ind == TRUE`
# # individual-specific AFs
# p_ind <- out$p_ind

}
