#' Retrieve Data From the U.S. Bureau Of Labor Statistics API v2
#'
#' @description
#'
#' `bls_request()`  will execute queries against the BLS API. Queries are
#' generated using one of the following query-generating functions:
#' [query_series()], [query_n_series()], [query_popular_series()],
#' [query_all_surveys()], [query_survey_info()], [query_latest_observation()].
#' The result is the "Results" block as defined in the API v2 signatures at
#' <https://www.bls.gov/developers/api_signature_v2.htm>
#'
#' @param query list generated by one of the query generating functions
#' @param api_key string, only necessary for retrieving multiple series in one
#' request, requesting calculations, or custom time frames and catalog data
#' @param user_agent string, optional
#' @param process_response function, optional. processes the `httr` response
#'  object. The default function will return the JSON payload parsed into a list
#' @param ... further arguments will be passed to `process_response` when called
#'
#' @return a list of information returned by the API request
#'
#' @family blsR-requests
#'
#' @export
#'
#' @examples
#' \dontrun{
#' library(blsR)
#' uer_query <- query_series('LNS14000000') #monthly unemployment rate series
#' uer_results <- bls_request(uer_query) #API response
#' }


bls_request <- function(
  query,
  api_key = NA,
  user_agent = 'http://github.com/groditi/blsR',
  process_response = .process_response,
  ...
){
  ua <-  httr::user_agent(user_agent)
  url <- httr::parse_url(query$url)
  #the query object should contain all it needs to make the request except
  #the BLS API key. simple requests that don't need it should be run even if
  #its missing but for complex ones with multiple series requested it should
  #inject the  key into the payload

  if(query$is_complex == FALSE){
    if(!is.na(api_key))
      url <- httr::modify_url(url, query = c(url$query, list(registrationkey = api_key)))

    response <- httr::GET(url, ua)
  } else{
    if('payload' %in% names(query)){
      if('seriesid' %in% names(query$payload)){
        if(is.na(api_key))
          warning('api_key is required for multiple series requests.')
        query$payload[['registrationkey']] <- api_key
      }
    }

    response <- httr::POST(url=url, ua, body=query$payload, encode="json")
  }

  return(
    tryCatch(
      process_response(response, ...),
      error = function(e) stop(paste(c('Error processing', url, e), ': '))
    )
  )
}

.process_response <- function(response, ...){

  #die if the format of the response content isnt json
  if(httr::http_type(response) != "application/json"){
    stop("http request did not return json", call. = FALSE)
  }

  json_response <- httr::content(response, simplifyVector=FALSE, ...)

  #die if request wasn't successful
  if(json_response$status != 'REQUEST_SUCCEEDED') {
    stop(paste(json_response$message, '; '), call. = FALSE)
  }
  return(json_response$Results)
}

