% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netclu_leiden.R
\name{netclu_leiden}
\alias{netclu_leiden}
\title{Finding communities using the Leiden algorithm}
\usage{
netclu_leiden(
  net,
  weight = TRUE,
  index = names(net)[3],
  objective_function = c("CPM", "modularity"),
  resolution_parameter = 1,
  beta = 0.01,
  initial_membership = NULL,
  n_iterations = 2,
  vertex_weights = NULL,
  bipartite = FALSE,
  site_col = 1,
  species_col = 2,
  return_node_type = "both",
  algorithm_in_output = TRUE
)
}
\arguments{
\item{net}{the output object from \code{\link[=similarity]{similarity()}} or
\code{\link[=dissimilarity_to_similarity]{dissimilarity_to_similarity()}}. If a \code{data.frame} is used, the first two
columns represent pairs of sites (or any pair of nodes), and the next
column(s) are the similarity indices.}

\item{weight}{a \code{boolean} indicating if the weights should be considered
if there are more than two columns.}

\item{index}{name or number of the column to use as weight. By default,
the third column name of \code{net} is used.}

\item{objective_function}{Whether to use the Constant Potts Model (CPM) or
modularity. Must be either "CPM" or "modularity".}

\item{resolution_parameter}{The resolution parameter to use. Higher
resolutions lead to more smaller communities, while lower resolutions lead
to fewer larger communities.}

\item{beta}{Parameter affecting the randomness in the Leiden algorithm. This
affects only the refinement step of the algorithm.}

\item{initial_membership}{If provided, the Leiden algorithm will try to
improve this provided membership. If no argument is provided, the aglorithm
simply starts from the singleton partition.}

\item{n_iterations}{the number of iterations to iterate the Leiden
algorithm. Each iteration may improve the partition further.}

\item{vertex_weights}{the vertex weights used in the Leiden algorithm. If
this is not provided, it will be automatically determined on the basis of
the objective_function. Please see the details of this function how to
interpret the vertex weights.}

\item{bipartite}{a \code{boolean} indicating if the network is bipartite
(see Details).}

\item{site_col}{name or number for the column of site nodes
(i.e. primary nodes).}

\item{species_col}{name or number for the column of species nodes
(i.e. feature nodes).}

\item{return_node_type}{a \code{character} indicating what types of nodes
("sites", "species" or "both") should be returned in the output
(\code{keep_nodes_type="both"} by default).}

\item{algorithm_in_output}{a \code{boolean} indicating if the original output
of \code{communities} should be returned in the output (see Value).}
}
\value{
A \code{list} of class \code{bioregion.clusters} with five slots:
\enumerate{
\item{\strong{name}: \verb{character string} containing the name of the algorithm}
\item{\strong{args}: \code{list} of input arguments as provided by the user}
\item{\strong{inputs}: \code{list} of characteristics of the clustering process}
\item{\strong{algorithm}: \code{list} of all objects associated with the
clustering procedure, such as original cluster objects (only if
\code{algorithm_in_output = TRUE})}
\item{\strong{clusters}: \code{data.frame} containing the clustering results}}

In the \code{algorithm} slot, if \code{algorithm_in_output = TRUE}, users can
find an "communities" object, output of
\link[igraph:cluster_leiden]{cluster_leiden}.
}
\description{
This function finds communities in a (un)weighted undirected network based
on the Leiden algorithm of Traag, van Eck & Waltman.
}
\details{
This function is based on the Leiden algorithm
\insertCite{Traag2019}{bioregion} as implemented in the
\href{https://cran.r-project.org/package=igraph}{igraph}
package (\link[igraph:cluster_leiden]{cluster_leiden}).
}
\note{
Although this algorithm was not primarily designed to deal with bipartite
network, it is possible to consider the bipartite network as unipartite
network (\code{bipartite = TRUE}).

Do not forget to indicate which of the first two columns is
dedicated to the site nodes (i.e. primary nodes) and species nodes (i.e.
feature nodes) using the arguments \code{site_col} and \code{species_col}.
The type of nodes returned in the output can be chosen with the argument
\code{return_node_type} equal to \code{"both"} to keep both types of nodes,
\code{"sites"} to preserve only the sites nodes and \code{"species"} to
preserve only the species nodes.
}
\examples{
comat <- matrix(sample(1000, 50), 5, 10)
rownames(comat) <- paste0("Site", 1:5)
colnames(comat) <- paste0("Species", 1:10)

net <- similarity(comat, metric = "Simpson")
com <- netclu_leiden(net)

net_bip <- mat_to_net(comat, weight = TRUE)
clust2 <- netclu_leiden(net_bip, bipartite = TRUE)

}
\references{
\insertRef{Traag2019}{bioregion}
}
\author{
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}),
Pierre Denelle (\email{pierre.denelle@gmail.com}) and
Boris Leroy (\email{leroy.boris@gmail.com})
}
