% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bigleaf_physiology.r
\name{stomatal.slope}
\alias{stomatal.slope}
\title{Stomatal Slope Parameter "g1"}
\usage{
stomatal.slope(data, Tair = "Tair", pressure = "pressure", GPP = "GPP",
  Gs = "Gs", VPD = "VPD", Ca = "Ca", Rleaf = NULL, model = c("USO",
  "Ball&Berry", "Leuning"), robust.nls = FALSE, nmin = 40, fitg0 = FALSE,
  g0 = 0, fitD0 = FALSE, D0 = 1.5, Gamma = 50,
  missing.Rleaf.as.NA = FALSE, constants = bigleaf.constants(), ...)
}
\arguments{
\item{data}{Data.frame or matrix containing all required columns}

\item{Tair}{Air temperature (deg C)}

\item{pressure}{Atmospheric pressure (kPa)}

\item{GPP}{Gross primary productivity (umol CO2 m-2 s-1)}

\item{Gs}{Surface conductance to water vapor (mol m-2 s-1)}

\item{VPD}{Vapor pressure deficit (kPa)}

\item{Ca}{Atmospheric CO2 concentration (umol mol-1)}

\item{Rleaf}{Ecosystem respiration stemming from leaves (umol CO2 m-2 s-1); defaults to 0}

\item{model}{Stomatal model used. One of \code{"USO","Ball&Berry","Leuning"}.}

\item{robust.nls}{Use robust nonlinear regression (\code{\link[robustbase]{nlrob}})? Default is \code{FALSE}.}

\item{nmin}{Minimum number of data required to perform the fit; defaults to 40.}

\item{fitg0}{Should g0 and g1 be fitted simultaneously?}

\item{g0}{Minimum stomatal conductance (mol m-2 s-1); ignored if \code{fitg0 = TRUE}.}

\item{fitD0}{Should D0 be fitted along with g1 (and g0 if \code{fitg0 = TRUE})?; only used if \code{model = "Leuning"}.}

\item{D0}{Stomatal sensitivity parameter to VPD; only used if \code{model = "Leuning"} and \code{fitD0 = FALSE}.}

\item{Gamma}{Canopy CO2 compensation point (umol mol-1); only used if \code{model = "Leuning"}. 
Can be a constant or a variable. Defaults to 50 umol mol-1.}

\item{missing.Rleaf.as.NA}{if Rleaf is provided, should missing values be treated as \code{NA} (\code{TRUE})
or set to 0 (\code{FALSE}, the default)?}

\item{constants}{Kelvin - conversion degree Celsius to Kelvin \cr
Rgas - universal gas constant (J mol-1 K-1)}

\item{...}{Additional arguments to \code{\link[stats]{nls}} or \code{\link[robustbase]{nlrob}} if \code{robust.nls = TRUE}.}
}
\value{
A \code{nls} model object, containing information on the fitted parameters, their uncertainty range,
        model fit, etc.
}
\description{
Estimation of the intrinsic WUE metric "g1" (stomatal slope) 
             from nonlinear regression.
}
\details{
All stomatal models were developed at leaf-level, but its parameters 
         can also be estimated at ecosystem level (but be aware of caveats).
         
         The unified stomatal optimization (USO) model is given by (Medlyn et al. 2011):
     
            \deqn{gs = g0 + 1.6*(1.0 + g1/sqrt(VPD)) * GPP/Ca}
         
         The semi-empirical model by Ball et al. 1987 is defined as:
         
            \deqn{gs = g0 + g1* ((An * rH) / Ca)}
         
         Leuning 1995 suggested a revised version of the Ball&Berry model:
         
            \deqn{gs = g0 + g1*GPP / ((Ca - \Gamma) * (1 + VPD/D0))}
         
         where \eqn{\Gamma} is by default assumed to be constant, but likely varies with temperature and among
         plant species. 
         The parameters in the models are estimated using nonlinear regression (\code{\link[stats]{nls}}) if
         \code{robust.nls = FALSE} and weighted nonlinear regression if \code{robust.nls = TRUE}.
         The weights are calculated from \code{\link[robustbase]{nlrob}}, and \code{\link[stats]{nls}}
         is used for the actual fitting.
         Alternatively to measured VPD and Ca (i.e. conditions at instrument height), conditions at 
         the big-leaf surface can be provided. Those can be calculated using \code{\link{surface.conditions}}.
}
\examples{
## filter data to ensure that Gs is a meaningful proxy to canopy conductance (Gc)
DE_Tha_Jun_2014_2 <- filter.data(DE_Tha_Jun_2014,quality.control=FALSE,
                                 vars.qc=c("Tair","precip","VPD","H","LE"),
                                 filter.growseas=FALSE,filter.precip=TRUE,
                                 filter.vars=c("Tair","PPFD","ustar","LE"),
                                 filter.vals.min=c(5,200,0.2,0),
                                 filter.vals.max=c(NA,NA,NA,NA),NA.as.invalid=TRUE,
                                 quality.ext="_qc",good.quality=c(0,1),
                                 missing.qc.as.bad=TRUE,GPP="GPP",doy="doy",
                                 year="year",tGPP=0.5,ws=15,min.int=5,precip="precip",
                                 tprecip=0.1,precip.hours=24,records.per.hour=2)

# calculate Gs from the the inverted PM equation
Ga <- aerodynamic.conductance(DE_Tha_Jun_2014_2,Rb_model="Thom_1972")[,"Ga_h"]

# if G and/or S are available, don't forget to indicate (they are ignored by default).
Gs_PM <- surface.conductance(DE_Tha_Jun_2014_2,Tair="Tair",pressure="pressure",
                             Rn="Rn",G="G",S=NULL,VPD="VPD",Ga=Ga,
                             formulation="PenmanMonteith")[,"Gs_mol"]
                             
### Estimate the stomatal slope parameter g1 using the USO model
mod_USO <- stomatal.slope(DE_Tha_Jun_2014_2,model="USO",GPP="GPP",Gs=Gs_PM,
                          robust.nls=FALSE,nmin=40,fitg0=FALSE)
                          
### Use robust regression to minimize influence of outliers in Gs                           
mod_USO <- stomatal.slope(DE_Tha_Jun_2014_2,model="USO",GPP="GPP",Gs=Gs_PM,
                          robust.nls=TRUE,nmin=40,fitg0=FALSE)

### Estimate the same parameter from the Ball&Berry model and prescribe g0
mod_BB <- stomatal.slope(DE_Tha_Jun_2014_2,model="Ball&Berry",GPP="GPP",
                         robust.nls=FALSE,Gs=Gs_PM,g0=0.01,nmin=40,fitg0=FALSE)

## same for the Leuning model, but this time estimate both g1 and g0 (but fix D0)
mod_Leu <- stomatal.slope(DE_Tha_Jun_2014_2,model="Leuning",GPP="GPP",Gs=Gs_PM,
                          robust.nls=FALSE,nmin=40,fitg0=FALSE,D0=1.5,fitD0=FALSE)

}
\references{
Medlyn B.E., et al., 2011: Reconciling the optimal and empirical approaches to
            modelling stomatal conductance. Global Change Biology 17, 2134-2144.
            
            Ball T.J., Woodrow I.E., Berry J.A. 1987: A model predicting stomatal conductance
            and its contribution to the control of photosynthesis under different environmental conditions.
            In: Progress in Photosynthesis Research, edited by J.Biggins, pp. 221-224, Martinus Nijhoff Publishers,
            Dordrecht, Netherlands.
            
            Leuning R., 1995: A critical appraisal of a combined stomatal-photosynthesis
            model for C3 plants. Plant, Cell and Environment 18, 339-355.
            
            Knauer, J. et al., 2018: Towards physiologically meaningful water-use efficiency estimates
            from eddy covariance data. Global Change Biology 24, 694-710.
}
\seealso{
\code{\link{surface.conductance}}
}
