#' Estimate Squared Partial Correlation Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using Nonparametric Bootstrapping
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of squared partial correlation coefficients
#'   (\eqn{r^{2}_{p}})
#'   is estimated from bootstrap samples.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{r^{2}_{p}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betanb` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{object}{The function argument `object`.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{r^{2}_{p}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{r^{2}_{p}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{r^{2}_{p}}.}
#'     \item{fun}{Function used ("PCorMC").}
#'   }
#'
#' @inheritParams BetaNB
#'
#' @examples
#' # Fit the regression model
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#' # Generate the sampling distribution of sample covariances
#' # (use a large R, for example, R = 5000 for actual research)
#' nb <- NB(object, R = 50)
#' # Generate confidence intervals for standardized regression slopes
#' rp <- PCorNB(nb)
#' # Methods --------------------------------------------------------
#' print(rp)
#' summary(rp)
#' coef(rp)
#' vcov(rp)
#' confint(rp, level = 0.95)
#' @export
#' @family Beta Nonparametric Bootstrap Functions
#' @keywords betaNB pcor
PCorNB <- function(object) {
  stopifnot(
    methods::is(
      object,
      "nb"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  fun <- "PCorNB"
  est <- .PCorSq(
    srsq = .SPCor(
      betastar = object$lm_process$betastar,
      sigmacapx = object$lm_process$sigmacapx
    )^2,
    rsq = object$lm_process$summary_lm$r.squared
  )
  names(est) <- object$lm_process$xnames
  foo <- function(x) {
    sr <- .SPCor(
      betastar = .BetaStarofSigma(
        sigmacap = x,
        q = 1 / sqrt(diag(x)),
        k = object$lm_process$k
      ),
      sigmacapx = x[
        2:object$lm_process$k,
        2:object$lm_process$k,
        drop = FALSE
      ]
    )
    rsq <- .RSqofSigma(
      sigmacap = x,
      k = object$lm_process$k
    )
    return(
      .PCorSq(
        srsq = sr^2,
        rsq = rsq
      )
    )
  }
  thetahatstar <- lapply(
    X = object$thetahatstar,
    FUN = foo
  )
  vcov <- stats::var(
    do.call(
      what = "rbind",
      args = thetahatstar
    )
  )
  colnames(vcov) <- rownames(vcov) <- names(est)
  out <- list(
    call = match.call(),
    object = object,
    thetahatstar = thetahatstar,
    jackknife = lapply(
      X = object$jackknife,
      FUN = foo
    ),
    vcov = vcov,
    est = est,
    fun = fun
  )
  class(out) <- c(
    "betanb",
    class(out)
  )
  return(
    out
  )
}
