#' Automatically threshold an image.
#'
#' These functions apply the ImageJ "Auto Threshold" plugin's image thresholding
#' methods. The available methods are "IJDefault", "Huang", "Intermodes",
#' "IsoData", "Li", "MaxEntropy", "Mean", "MinErrorI", "Minimum", "Moments",
#' "Otsu", "Percentile", "RenyiEntropy", "Shanbhag", "Triangle", "Yen". Read
#' about them at \url{http://imagej.net/Auto_Threshold}.
#'
#' \itemize{ \item{\code{NA} values are automatically ignored.} \item{For
#' \code{ignore.white = TRUE}, if the maximum value in the array is one of
#' \code{2^8-1}, \code{2^12-1}, \code{2^16-1} or \code{2^32-1}, then those max
#' values are ignored. That's because they're the white values in 8, 12, 16 and
#' 32-bit images respectively (and these are the common image bit sizes to work
#' with). This guesswork has to be done because \code{R} does not know how many
#' bits the image was on disk. This guess is very unlikely to be wrong, and if
#' it is, the consequences are negligible anyway. If you're very concerned, then
#' just specify the max value in the \code{ignore.white} argument.} \item{If you
#' have set \code{ignore.black = TRUE} and/or \code{ignore.white = TRUE} but you
#' are still getting error/warning messages telling you to try them, then your
#' chosen method is not working for the given array, so you should try a
#' different method.} }
#'
#' @param int_arr An array (or vector) of \emph{integers}.
#' @param method The name of the method you wish to use (e.g. "Huang"). Partial
#'   matching is performed i.e. \code{method = "h"} is enough to get you "Huang"
#'   and \code{method = "in"} is enough to get you "Intermodes".
#' @param ignore.black Ignore black pixels/elements (zeros) when performing the
#'   thresholding?
#' @param ignore.white Ignore white pixels when performing the thresholding? If
#'   set to \code{TRUE}, the function makes a good guess as to what the white
#'   (saturated) value would be (see "Details"). If this is set to a number, all
#'   pixels with value greater than or equal to that number are ignored.
#' @param fail When using \code{auto_thresh_apply_mask}, to what value do you
#'   wish to set the pixels which fail to exceed the threshold.
#'
#' @return \code{auto_thresh} returns an integer, the image threshold value.
#'   Pixels exceeding this threshold are passed, but pixels at or below this
#'   level are failed.
#'
#'   \code{auto_thresh_mask} returns a binarized version of the input, with a
#'   value of \code{TRUE} at points which exceed the threshold and \code{FALSE}
#'   at those which do not. This has an attribute "threshold" to tell you what
#'   the threshold value was.
#'
#'   \code{auto_thresh_apply_mask} returns the original input masked by the
#'   threshold, i.e. all points not exceeding the threshold are set to a
#'   user-defined value (default \code{NA}). This has an attribute "threshold"
#'   to tell you what the threshold value was.
#'
#' @references \itemize{ \item{Huang, L-K & Wang, M-J J (1995), "Image
#'   thresholding by minimizing the measure of fuzziness", Pattern Recognition
#'   28(1): 41-51} \item{Prewitt, JMS & Mendelsohn, ML (1966), "The analysis of
#'   cell images", Annals of the New York Academy of Sciences 128: 1035-1053}
#'   \item{Ridler, TW & Calvard, S (1978), "Picture thresholding using an
#'   iterative selection method", IEEE Transactions on Systems, Man and
#'   Cybernetics 8: 630-632} \item{Li, CH & Lee, CK (1993), "Minimum Cross
#'   Entropy Thresholding", Pattern Recognition 26(4): 617-625} \item{Li, CH &
#'   Tam, PKS (1998), "An Iterative Algorithm for Minimum Cross Entropy
#'   Thresholding", Pattern Recognition Letters 18(8): 771-776} \item{Sezgin, M
#'   & Sankur, B (2004), "Survey over Image Thresholding Techniques and
#'   Quantitative Performance Evaluation", Journal of Electronic Imaging 13(1):
#'   146-165} \item{Kapur, JN; Sahoo, PK & Wong, ACK (1985), "A New Method for
#'   Gray-Level Picture Thresholding Using the Entropy of the Histogram",
#'   Graphical Models and Image Processing 29(3): 273-285} \item{Glasbey, CA
#'   (1993), "An analysis of histogram-based thresholding algorithms", CVGIP:
#'   Graphical Models and Image Processing 55: 532-537} \item{Kittler, J &
#'   Illingworth, J (1986), "Minimum error thresholding", Pattern Recognition
#'   19: 41-47} \item{Prewitt, JMS & Mendelsohn, ML (1966), "The analysis of
#'   cell images", Annals of the New York Academy of Sciences 128: 1035-1053}
#'   \item{Tsai, W (1985), "Moment-preserving thresholding: a new approach",
#'   Computer Vision, Graphics, and Image Processing 29: 377-393} \item{Otsu, N
#'   (1979), "A threshold selection method from gray-level histograms", IEEE
#'   Trans. Sys., Man., Cyber. 9: 62-66, doi:10.1109/TSMC.1979.4310076}
#'   \item{Doyle, W (1962), "Operation useful for similarity-invariant pattern
#'   recognition", Journal of the Association for Computing Machinery 9:
#'   259-267, doi:10.1145/321119.321123} \item{Kapur, JN; Sahoo, PK & Wong, ACK
#'   (1985), "A New Method for Gray-Level Picture Thresholding Using the Entropy
#'   of the Histogram", Graphical Models and Image Processing 29(3): 273-285}
#'   \item{Shanbhag, Abhijit G. (1994), "Utilization of information measure as a
#'   means of image thresholding", Graph. Models Image Process. (Academic Press,
#'   Inc.) 56 (5): 414--419, ISSN 1049-9652} \item{Zack GW, Rogers WE, Latt SA
#'   (1977), "Automatic measurement of sister chromatid exchange frequency", J.
#'   Histochem. Cytochem. 25 (7): 74153, PMID 70454} \item{Yen JC, Chang FJ,
#'   Chang S (1995), "A New Criterion for Automatic Multilevel Thresholding",
#'   IEEE Trans. on Image Processing 4 (3): 370-378, ISSN 1057-7149,
#'   doi:10.1109/83.366472} \item{Sezgin, M & Sankur, B (2004), "Survey over
#'   Image Thresholding Techniques and Quantitative Performance Evaluation",
#'   Journal of Electronic Imaging 13(1): 146-165} }
#'
#' @section Acknowledgements: Gabriel Landini coded all of these functions
#'   together in one Java file for the ImageJ plugin.
#'
#' @examples
#' img <- EBImage::readImage(system.file("extdata", "eg.tif",
#' package = "autothresholdr"), as.is = TRUE)
#' auto_thresh(img, "h")
#' auto_thresh(img, "tri")
#' auto_thresh(img, "Otsu")
#' mask <- auto_thresh_mask(img, "h")
#' EBImage::display(mask, method = "r")
#' masked <- auto_thresh_apply_mask(img, "h")
#' EBImage::display(EBImage::normalize(masked), method = "r")
#'
#' @export
auto_thresh <- function(int_arr, method,
                        ignore.black = FALSE, ignore.white = FALSE) {
  available_methods <- c("IJDefault", "Huang", "Intermodes", "IsoData", "Li",
                         "MaxEntropy", "Mean", "MinErrorI", "Minimum",
                         "Moments", "Otsu", "Percentile", "RenyiEntropy",
                         "Shanbhag", "Triangle", "Yen")
  method <- RSAGA::match.arg.ext(method, available_methods,
                                 ignore.case = TRUE, numeric = TRUE) %>%
                                 {available_methods[.]}
  if ((!CanBeInteger(int_arr)) || any(int_arr < 0, na.rm = TRUE)) {
    stop("int_arr must be a matrix of non-negative integers.")
  }
  if (ignore.black) int_arr[int_arr == 0] <- NA
  if (ignore.white) {
    if (isTRUE(ignore.white)) {
      mx <- max(int_arr)
      if (mx %in% (2 ^ c(8, 12, 16, 32) - 1)) int_arr[int_arr == mx] <- NA
    } else {
      int_arr[int_arr >= ignore.white] <- NA
    }
  }
  rim <- range(int_arr, na.rm = TRUE)
  im_hist <- factor(int_arr, levels = rim[1]:rim[2]) %>%
    table %>% as.vector
  if (length(im_hist) < 2) {
    stop("The image you're trying to threshold has only one unique value, ",
         "to perform thresholding, it needs at least two unique values.")
  }
  autothresh_class <- rJava::.jnew("Auto_Threshold_R")
  rJava::.jcall(autothresh_class, "I", method, im_hist) + rim[1]
}

#' @rdname auto_thresh
#' @export
auto_thresh_mask <- function(int_arr, method,
                             ignore.black = FALSE, ignore.white = FALSE) {
  thresh <- auto_thresh(int_arr, method,
              ignore.black = ignore.black, ignore.white = ignore.white)
  mask <- int_arr > thresh
  attr(mask, "threshold") <- thresh
  mask
}

#' @rdname auto_thresh
#' @export
auto_thresh_apply_mask <- function(int_arr, method, fail = NA,
                                   ignore.black = FALSE, ignore.white = FALSE) {
  mask <- auto_thresh_mask(int_arr, method,
            ignore.black = ignore.black, ignore.white = ignore.white)
  int_arr[!mask] <- fail
  attr(int_arr, "threshold") <- attr(mask, "threshold")
  int_arr
}

