% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confint_adjust.R
\name{confint_adjust}
\alias{confint_adjust}
\title{Adjust confidence intervals for multiple comparisons}
\usage{
confint_adjust(object, parm, level = 0.95, method = "none")
}
\arguments{
\item{object}{a fitted model object.}

\item{parm}{a specification of which parameters are to be given
    confidence intervals, either a vector of numbers or a vector of
    names.  If missing, all parameters are considered.}

\item{level}{the confidence level required.}

\item{method}{A character string indicating the type of
adjustment to make. The default choice is
\code{"none"}. The other option is
\code{"bonferroni"}.}
}
\value{
A \code{confint_adjust} object, which is simply a
  a \code{data.frame} with columns \code{term},
  \code{lwr} (the lower confidence limit), and \code{upr}
  (the upper confidence limit).
}
\description{
A function to produce adjusted confidence intervals with a family-wise
confidence level of at least \code{level} for
\code{lm} objects (not applicable if no adjustment is used).
Internally, the function is a slight revision of the code
used in the \code{\link[stats]{confint.lm}} function.
}
\details{
Let \code{a = 1 - level}. Let \code{p} be the number of
estimated coefficients in the fitted model. All intervals are computed
using the formula \code{estimate +/- m * ese}, where
\code{m} is a multiplier and \code{ese} is the estimated
standard error of the \code{estimate}.

\code{method = "none"} (no correction) produces the
standard t-based confidence intervals with multiplier
\code{qt(1 - a/2, df = object$df.residual)}.

\code{method = "bonferroni"} produces Bonferroni-adjusted
intervals that use the multiplier \code{m = qt(1 - a/(2 *
k), df = object$df.residual)}, where \code{k} is the
number of intervals being produced.

\code{method = "wh"} produces Working-Hotelling-adjusted
intervals that are valid for all linear combinations of
the regression coefficients, which uses the multiplier
\code{m = sqrt(p * qf(level, df1 = p, df2 =
object$df.residual))}.
}
\examples{
## an extension of the documentation for confint.lm
fit <- lm(100/mpg ~ disp + hp + wt + am, data = mtcars)
# standard intervals
confint_adjust(fit)
# bonferroni-adjusted intervals
(cib <- confint_adjust(fit, method = "b"))
# plot results
plot(cib)
plot(cib, parm = c("hp", "disp"))
if (require(ggplot2)) {
  autoplot(cib)
  autoplot(cib, parm = c("hp", "disp"))
}
#' working-hotelling-adjusted intervals
(ciwh <- confint_adjust(fit, method = "wh"))
}
\references{
Bonferroni, C. (1936). Teoria statistica
  delle classi e calcolo delle probabilita. Pubblicazioni
  del R Istituto Superiore di Scienze Economiche e
Commericiali di Firenze, 8, 3-62.

Working, H., & Hotelling, H. (1929). Applications of the
theory of error to the interpretation of trends. Journal
of the American Statistical Association, 24(165A), 73-85.
doi:10.1080/01621459.1929.10506274

Kutner, M. H., Nachtsheim, C. J., Neter, J., & Li, W.
(2004). Applied Linear Statistical Models, 5th edition.
New York: McGraw-Hill/Irwin. (p. 230)
}
\seealso{
\code{\link[stats]{confint.lm}}
}
