#Copyright © 2016 RTE Réseau de transport d’électricité

#' Load factors of clusters
#'
#' This function computes the load factor and other related statistics
#' for cluster of a study.
#'
#' @param x
#'   Object of class \code{antaresData} created with function
#'   \code{\link[antaresRead]{readAntares}}. It must contain hourly detailed
#'   results for clusters and has to contain the columns
#'   \code{minGenModulation}.
#' @inheritParams surplus
#' @inheritParams surplusClusters
#'
#' @return
#' a data.table of class \code{antaresDataTable}containing the following
#' columns:
#' \item{area}{Area name}
#' \item{cluster}{Cluster name}
#' \item{mcYear}{Only if \code{synthesis=FALSE}}. Id of the Monte-carlo scenario
#' \item{timeId}{Time id and other time variables}
#' \item{loadFactor}{
#'   Load factor of the cluster. It represent the proportion of
#'   the installed capacity of a cluster that is effectively generate
#'
#'   Formula: production / (unitcount * nominalcapacity)
#' }
#' \item{propHoursMinGen}{
#'   Proportion of hours when production is positive and
#'   all units of a cluster are either off, either producing at their minimum. This
#'   situation occurs when units are kept producing above the optimal level to avoid
#'   future startup costs or to satisfy the constraints generated by parameters
#'   "Min. up Time" or "Min gen. modulation".
#'
#'   Formula: mean(1 if production > 0 and production = max(min.stable.power * unitcount,
#'   minGenModulation * nominalcapacity * unitcount) else 0)
#' }
#' \item{propHoursMaxGen}{
#'   Proportion of hours when all units started produce at
#'   their maximal capacity.
#'
#'   Formula: mean(1 if production > 0 and production = NODU * nominalcapacity *
#'   (1 - spinning / 100))
#' }
#'
#' @examples
#' \dontrun{
#' # data required by the function
#' showAliases("loadfactor")
#'
#' mydata <- readAntares(select = "loadfactor")
#' loadFactor(mydata, synthesis = TRUE)
#' }
#'
#' @export
#'
loadFactor <- function(x, timeStep = "annual", synthesis = FALSE,
                       clusterDesc = NULL) {

  .checkAttrs(x, timeStep = "hourly", synthesis = FALSE)
  x <- .checkColumns(x, list(clusters = c("production", "NODU", "minGenModulation")))
  x$cluster[is.na(minGenModulation), minGenModulation := 0]
  opts <- simOptions(x)
  idVars <- .idCols(x$clusters)

  if (is.null(clusterDesc)) clusterDesc <- readClusterDesc(opts)
  .fillClusterDesc(clusterDesc, min.stable.power = 0, spinning = 0)

  tmp <- merge(x$cluster[, c(idVars, "production", "NODU", "minGenModulation"), with = FALSE],
               clusterDesc[, .(area, cluster, min.stable.power, nominalcapacity, unitcount, spinning)],
               by = c("area", "cluster"))

  tmp[, `:=`(
    loadFactor = production / (nominalcapacity * unitcount),
    propHoursMinGen = ifelse(
      production > 0 & production == pmax(min.stable.power * NODU,
                                          minGenModulation * nominalcapacity * unitcount),
      1,
      0
    ),
    propHoursMaxGen = ifelse(
      production > 0 & production == round(NODU * nominalcapacity * (1 - spinning /100)),
      1,
      0
    )
  )]

  res <- tmp[, c(idVars, "loadFactor", "propHoursMinGen", "propHoursMaxGen"), with = FALSE]
  res <- .addClassAndAttributes(res, FALSE, "hourly", opts, type = "loadFactor")

  res <- changeTimeStep(res, timeStep, fun = "mean")
  if (synthesis) res <- synthesize(res)

  res
}
