#' @title BFCA balanced floating catchment area
#'
#' @description
#' Calculates accessibility levels using the balanced floating catchment area
#' (BFCA) measure proposed by Paez et al. (2019). The BFCA metric calculate
#' accessibility accounting for the competition of resources and simultaneously
#' correcting for issues of inflation of demand and service levels that are
#' present in other accessibility metrics in the floating catchment area family.
#'
#' @template input_data
#' @template arguments_fca
#' @template decay_function
#' @template opportunity_col
#' @template travel_cost_col
#'
#' @return A `numeric` estimate of accessibility.
#'
#' @keywords internal
fca_bfca <- function(data,
                     orig_col,
                     dest_col,
                     population_col,
                     opportunity_col,
                     decay_function,
                     travel_cost_col){

  # orig_col <- 'from_id'
  # dest_col <- 'to_id'
  # opportunity_col <- 'jobs'
  # population_col <- 'population'


  # check inputs ------------------------------------------------------------
  checkmate::assert_function(decay_function)


  # calculate access -----------------------------------------------------------
  dt <- data.table::copy(data)

  # orig_col <- 'from_id'
  # dest_col <- 'to_id'
  # opportunity_col <- 'jobs'
  # population_col <- 'population'

  # calculate impedance
  dt[, impedance := decay_function(t_ij = get(travel_cost_col)),]

  # calculate balanced impedance i (normalizing impedance by origin)
  dt[, balanced_impedance_i := impedance/sum(impedance),
       by= c(orig_col)]


  # calculate balanced impedance j (normalizing impedance by destination)
  dt[, balanced_impedance_j := impedance/sum(impedance),
       by= c(dest_col)]


  ## Step 1 - allocate the demand to each destination proportionally to weight i
  dt[, pop_served := sum( get(population_col) * balanced_impedance_i, na.rm = TRUE),
       by= c(dest_col)]

  ## Step 2 - calculate provider-to-population ration (ppr) at each destination
  # The level of service of an area is the number of opportunities/resources in the area, divided by the population it serves:
  # level of service == provider-to-population ratio (PPR)
  dt[, ppr := data.table::first( get(opportunity_col)) / pop_served,
       by= c(dest_col)]


  ## Step 3 - reaportion ppr at each origin proportionally to weight j
  bfca <- dt[, .(access_bfca = sum(ppr * balanced_impedance_j, na.rm=T)),
               by= c(orig_col)]

  return(bfca)
}
