% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xEnricher.r
\name{xEnricher}
\alias{xEnricher}
\title{Function to conduct enrichment analysis given the input data and the ontology and its annotation}
\usage{
xEnricher(data, annotation, g, background = NULL, size.range = c(10,
2000),
min.overlap = 3, which.distance = NULL, test = c("hypergeo", "fisher",
"binomial"), p.adjust.method = c("BH", "BY", "bonferroni", "holm",
"hochberg", "hommel"), ontology.algorithm = c("none", "pc", "elim",
"lea"),
elim.pvalue = 0.01, lea.depth = 2, path.mode = c("all_paths",
"shortest_paths", "all_shortest_paths"), true.path.rule = TRUE,
verbose = T)
}
\arguments{
\item{data}{an input vector containing a list of genes or SNPs of
interest}

\item{annotation}{the vertices/nodes for which annotation data are
provided. It can be a sparse Matrix of class "dgCMatrix" (with
variants/genes as rows and terms as columns), or a list of nodes/terms
each containing annotation data, or an object of class 'GS' (basically
a list for each node/term with annotation data)}

\item{g}{an object of class "igraph" to represent DAG. It must have
node/vertice attributes: "name" (i.e. "Term ID"), "term_id" (i.e. "Term
ID"), "term_name" (i.e "Term Name") and "term_distance" (i.e. Term
Distance: the distance to the root; always 0 for the root itself)}

\item{background}{a background vector. It contains a list of genes or
SNPs as the test background. If NULL, by default all annotatable are
used as background}

\item{size.range}{the minimum and maximum size of members of each term
in consideration. By default, it sets to a minimum of 10 but no more
than 2000}

\item{min.overlap}{the minimum number of overlaps. Only those terms
with members that overlap with input data at least min.overlap (3 by
default) will be processed}

\item{which.distance}{which terms with the distance away from the
ontology root (if any) is used to restrict terms in consideration. By
default, it sets to 'NULL' to consider all distances}

\item{test}{the test statistic used. It can be "fisher" for using
fisher's exact test, "hypergeo" for using hypergeometric test, or
"binomial" for using binomial test. Fisher's exact test is to test the
independence between gene group (genes belonging to a group or not) and
gene annotation (genes annotated by a term or not), and thus compare
sampling to the left part of background (after sampling without
replacement). Hypergeometric test is to sample at random (without
replacement) from the background containing annotated and non-annotated
genes, and thus compare sampling to background. Unlike hypergeometric
test, binomial test is to sample at random (with replacement) from the
background with the constant probability. In terms of the ease of
finding the significance, they are in order: hypergeometric test >
fisher's exact test > binomial test. In other words, in terms of the
calculated p-value, hypergeometric test < fisher's exact test <
binomial test}

\item{p.adjust.method}{the method used to adjust p-values. It can be
one of "BH", "BY", "bonferroni", "holm", "hochberg" and "hommel". The
first two methods "BH" (widely used) and "BY" control the false
discovery rate (FDR: the expected proportion of false discoveries
amongst the rejected hypotheses); the last four methods "bonferroni",
"holm", "hochberg" and "hommel" are designed to give strong control of
the family-wise error rate (FWER). Notes: FDR is a less stringent
condition than FWER}

\item{ontology.algorithm}{the algorithm used to account for the
hierarchy of the ontology. It can be one of "none", "pc", "elim" and
"lea". For details, please see 'Note' below}

\item{elim.pvalue}{the parameter only used when "ontology.algorithm" is
"elim". It is used to control how to declare a signficantly enriched
term (and subsequently all genes in this term are eliminated from all
its ancestors)}

\item{lea.depth}{the parameter only used when "ontology.algorithm" is
"lea". It is used to control how many maximum depth is used to consider
the children of a term (and subsequently all genes in these children
term are eliminated from the use for the recalculation of the
signifance at this term)}

\item{path.mode}{the mode of paths induced by vertices/nodes with input
annotation data. It can be "all_paths" for all possible paths to the
root, "shortest_paths" for only one path to the root (for each node in
query), "all_shortest_paths" for all shortest paths to the root (i.e.
for each node, find all shortest paths with the equal lengths)}

\item{true.path.rule}{logical to indicate whether the true-path rule
should be applied to propagate annotations. By default, it sets to
true}

\item{verbose}{logical to indicate whether the messages will be
displayed in the screen. By default, it sets to true for display}
}
\value{
an object of class "eTerm", a list with following components:
\itemize{
\item{\code{term_info}: a matrix of nTerm X 4 containing snp/gene set
information, where nTerm is the number of terms, and the 4 columns are
"id" (i.e. "Term ID"), "name" (i.e. "Term Name"), "namespace" and
"distance"}
\item{\code{annotation}: a list of terms containing annotations, each
term storing its annotations. Always, terms are identified by "id"}
\item{\code{g}: an igraph object to represent DAG}
\item{\code{data}: a vector containing input data in consideration. It
is not always the same as the input data as only those mappable are
retained}
\item{\code{background}: a vector containing the background data. It is
not always the same as the input data as only those mappable are
retained}
\item{\code{overlap}: a list of overlapped snp/gene sets, each storing
snps/genes overlapped between a snp/gene set and the given input data
(i.e. the snps/genes of interest). Always, gene sets are identified by
"id"}
\item{\code{fc}: a vector containing fold changes}
\item{\code{zscore}: a vector containing z-scores}
\item{\code{pvalue}: a vector containing p-values}
\item{\code{adjp}: a vector containing adjusted p-values. It is the p
value but after being adjusted for multiple comparisons}
\item{\code{cross}: a matrix of nTerm X nTerm, with an on-diagnal cell
for the overlapped-members observed in an individaul term, and
off-diagnal cell for the overlapped-members shared between two terms}
\item{\code{call}: the call that produced this result}
}
}
\description{
\code{xEnricher} is supposed to conduct enrichment analysis given the
input data and the ontology direct acyclic graph (DAG) and its
annotation. It returns an object of class "eTerm". Enrichment analysis
is based on either Fisher's exact test or Hypergeometric test. The test
can respect the hierarchy of the ontology.
}
\note{
The interpretation of the algorithms used to account for the hierarchy
of the ontology is:
\itemize{
\item{"none": does not consider the ontology hierarchy at all.}
\item{"lea": estimates the significance of a term in terms of the
significance of its children at the maximum depth (e.g. 2). Precisely,
once snps/genes are already annotated to any children terms with a more
signficance than itself, then all these snps/genes are eliminated from
the use for the recalculation of the signifance at that term. The final
p-values takes the maximum of the original p-value and the recalculated
p-value.}
\item{"elim": estimates the significance of a term in terms of the
significance of its all children. Precisely, once snps/genes are
already annotated to a signficantly enriched term under the cutoff of
e.g. pvalue<1e-2, all these snps/genes are eliminated from the
ancestors of that term).}
\item{"pc": requires the significance of a term not only using the
whole snps/genes as background but also using snps/genes annotated to
all its direct parents/ancestors as background. The final p-value takes
the maximum of both p-values in these two calculations.}
\item{"Notes": the order of the number of significant terms is: "none"
> "lea" > "elim" > "pc".}
}
}
\examples{
\dontrun{
# 1) SNP-based enrichment analysis using GWAS Catalog traits (mapped to EF)
# 1a) ig.EF (an object of class "igraph" storing as a directed graph)
g <- xRDataLoader('ig.EF')

# 1b) load GWAS SNPs annotated by EF (an object of class "dgCMatrix" storing a spare matrix)
anno <- xRDataLoader(RData='GWAS2EF')

# 1c) optionally, provide the test background (if not provided, all annotatable SNPs)
background <- rownames(anno)

# 1d) provide the input SNPs of interest (eg 'EFO:0002690' for 'systemic lupus erythematosus')
ind <- which(colnames(anno)=='EFO:0002690')
data <- rownames(anno)[anno[,ind]==1]
data

# 1e) perform enrichment analysis
eTerm <- xEnricher(data=data, annotation=anno, background=background,
g=g, path.mode=c("all_paths"))

# 1f) view enrichment results for the top significant terms
xEnrichViewer(eTerm)

# 1f') save enrichment results to the file called 'EF_enrichments.txt'
res <- xEnrichViewer(eTerm, top_num=length(eTerm$adjp), sortBy="adjp",
details=TRUE)
output <- data.frame(term=rownames(res), res)
utils::write.table(output, file="EF_enrichments.txt", sep="\\t",
row.names=FALSE)

# 1g) barplot of significant enrichment results
bp <- xEnrichBarplot(eTerm, top_num="auto", displayBy="adjp")
print(bp)

# 1h) visualise the top 10 significant terms in the ontology hierarchy
# color-code terms according to the adjust p-values (taking the form of 10-based negative logarithm)
xEnrichDAGplot(eTerm, top_num=10, displayBy="adjp",
node.info=c("full_term_name"))
# color-code terms according to the z-scores
xEnrichDAGplot(eTerm, top_num=10, displayBy="zscore",
node.info=c("full_term_name"))
}
}
\seealso{
\code{\link{xDAGanno}}, \code{\link{xEnricherGenes}},
\code{\link{xEnricherSNPs}}
}
