#'@title 
#'Retrieves a list of category members.
#'
#'@description
#'wiki_catpages retrieves a list of pages, subcategories, files or all of the above
#'in a specified category (or series of specified categories)
#'
#'@param con A connector object, generated by \code{\link{wiki_con}}, that corresponds
#'to the project you're trying to query.
#'
#'@param categories The names of the categories you want to gather information for.
#'
#'@param properties The properties you want to gather for each member of the category. 
#'Options are "title" (the name of the member, including namespace), 
#'"id" (the unique numeric identifier of the member), "sortkey" 
#'(the hexadecimal key used to sort that member within the category), 
#'"sortkeyprefix" (the human-readable sort key), "type"
#'(whether the member is a page, a subcategory or a file) and 
#'"timestamp" (when the member was added to the category)
#'
#'@param type The type of member you're interested in returning;
#'options are any permutation of "page" (pages), "subcat" (subcategories) and "file" (files).
#'
#'@section warnings:
#'Because of the way MediaWiki stores this data, both "the category you asked for doesn't exist"
#'and "the category you asked for exists, but has no members" return in the same way.
#'
#'@seealso \code{\link{wiki_pagecats}} for finding categories that a specified page is a member of.
#'
#'@export

#Retrieves pages in a particular category, or categories
wiki_catpages <- function(con, categories, properties = c("title","ids","sortkey","sortkeyprefix","type","timestamp"),
                          type = c("page","subcat","file")){
  
  #Add 'Category'
  categories <- gsub(x = categories, pattern = "^", replacement = "Category:")
  
  #Check categories against the limit
  categories <- handle_limits(categories, 50)
  
  #Match and standardise properties
  properties <- match.arg(properties, several.ok = TRUE)
  properties <- paste(properties, collapse = "|")
    
  #Match and standardise types
  type <- match.arg(type, several.ok = TRUE)
  type <- paste(type, collapse = "|")
  
  #Construct URL
  catpage_url <- paste(con$URL,"&action=query&list=categorymembers&cmtitle=",categories,"&cmprop=",properties,"&cmtype=",type, sep = "")

  #Query
  catpage_response <- wiki_call(URL = catpage_url, con$Config)
  
  #Return
  return(catpage_response)
}