% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/veWaningVariant.R
\name{veWaningVariant}
\alias{veWaningVariant}
\title{Estimation of Vaccine Efficacy Over Time - Variant Aware}
\usage{
veWaningVariant(
  data,
  L,
  ...,
  phase = "ub",
  lag = 6,
  modelEntry = NULL,
  modelUnblind0 = NULL,
  modelUnblind1 = NULL,
  modelPsi = NULL,
  modelCensor0 = NULL,
  modelCensor1 = NULL,
  gFunc = NULL,
  variant = 0L,
  v = NULL,
  minWgt = NULL,
  maxWgt = NULL,
  txName = "A",
  U = "U",
  entryTime = "E",
  Gamma = "Gam",
  R = "R",
  Psi = "Psi",
  Delta = "Delta"
)
}
\arguments{
\item{data}{A data.frame object containing all relevant data. 
Records with missing data will be removed prior to initiating the analysis.}

\item{L}{A numeric object. The analysis time.}

\item{...}{Ignored. Used only to require named inputs.}

\item{phase}{A character object. The phase(s) to include in the analysis.
Default is ="ub", indicating that both blinded and unblinded phases will be
used to estimate theta values if possible. If ="b", only the blinded
phase will be used. If ="u", only the unblinded phase will be used.}

\item{lag}{A scalar numeric object. The lag time between 
the initial vaccine dose and full efficacy for all participants. 
The default value is 6 weeks (42 days) -- NOTE this assumes that the
data are on the scale of weeks.}

\item{modelEntry}{A formula object or NULL. The coxph model for entry times.
The LHS is set as the appropriate Surv() object internally. If a LHS
is provided, it is ignored. If NULL, inputs modelPsi, modelUnblind0,
and modelUnblind1 must also be NULL. See Details for further information.}

\item{modelUnblind0}{A formula object or NULL. The coxph model for 
unblinding/crossover for placebo (A=0) participants. If NULL, inputs 
modelEntry, modelPsi, and modelUnblind1 must also be NULL. See Details 
for further information.}

\item{modelUnblind1}{A formula object or NULL. The coxph model for unblinding
for vaccinated (A=1) participants. If NULL, inputs modelEntry, modelPsi, 
and modelUnblind0 must also be NULL. See Details for further information.}

\item{modelPsi}{A formula object or NULL. The logistic model for the
probability that a placebo participant (A = 0) is unblinded at R
(Gamma = 1) and agrees to take the study vaccine (Psi = 1).
If a LHS is provided, it is ignored. If NULL, inputs modelEntry, 
modelUnblind0, and modelUnblind1 must also be NULL. See Details for 
further information.}

\item{modelCensor0}{A formula object or NULL. The coxph model for censoring
for placebo (A=0) participants. The LHS is set as the appropriate Surv() 
object internally. If a LHS is provided, it is ignored. If NULL, all other 
models must be NULL. See Details for further information.}

\item{modelCensor1}{A formula object or NULL. The coxph model for censoring
for vaccinated (A=1) participants. The LHS is set as the appropriate Surv() 
object internally. If a LHS is provided, it is ignored. If NULL, all other 
models must be NULL. See Details for further information.}

\item{gFunc}{A character vector object. The model of infection rates.
Must be one of {'lin', 'piece', 'splin', 'spcub'} for the linear,
piecewise constant, linear spline, and cubic spline models, respectively.}

\item{variant}{An integer object. The variant for the analysis. If 0,
all variants are included in the analysis.}

\item{v}{A numeric vector object. 
The knots or cut-offs to be used for input gFunc.
If gFunc = 'lin', this input is ignored. For 'splin' and 'spcub', the
knots of the spline on (0,L). For 'piece', the cut-offs on 
(0,L). Note that this input should not include the extremes 0 and L.}

\item{minWgt}{A numeric object. If not NULL, the minimum non-zero value a 
weight can have, i.e., weight = max(minWgt, weight). If NULL, no
lower truncation of weights is performed.}

\item{maxWgt}{A numeric object. If not NULL, the maximum value a 
weight can have, i.e., weight = min(maxWgt, weight). If NULL, no
upper truncation of weights is performed.}

\item{txName}{A character object. The header of the column of data 
containing the treatment variable. Default value is 'A'.
Treatment must be coded as 0/1, where 1 indicates that participant
was vaccinated; 0 otherwise.}

\item{U}{A character object. The header of the column of data 
containing the minimum of time to infection, time to refusing study 
vaccine after unblinding (placebo participants only), or time to censoring
(due to loss to follow up or administrative censoring).}

\item{entryTime}{A character object. The header of the column of data 
containing the time of entry into the study on the scale of the
calendar time. Default value is 'E'.}

\item{Gamma}{A character object. The header of the column of data 
containing the category for the R dynamic. Default value is 'Gam'.
Data must be 0/1, where 1 indicates that R is the time to unblinding;
0 indicates that R is the infection time or the censoring time.}

\item{R}{A character object. The header of the column of data 
containing the minimum of: time to unblinding, time to infection, or
time to censoring.}

\item{Psi}{A character object. The header of the column of data 
containing the indicator of whether a participant received study 
vaccine, where 1 indicates that a participant was assigned to placebo is 
unblinded and decides to get the study vaccine or that a participant
was assigned to vaccine; 0 otherwise.  Default value is 'Psi'.}

\item{Delta}{A character object. The header of the column of data 
containing the infection-variant category indicator.}
}
\value{
A an S3 object of class "VEwaningVariant", which comprises a list
  object containing 
  \item{theta}{A vector object containing the estimated theta parameters.}
  \item{cov}{The covariance estimated using the sandwich estimator.}
  \item{SE}{The standard error estimated using the sandwich estimator.}
  and attributes  "gFunc", "maxTau", "lag", "v", "phaseType", and "wgtType", 
  which store
  details of the original analysis that are required for post-processing
  convenience functions ve() and plot(). See details for further
  information.
}
\description{
Implements methods for inference on potential waning of vaccine
   efficacy and for estimation of vaccine efficacy at a user-specified time
   after vaccination based on data from a randomized, double-blind,
   placebo-controlled vaccine trial in which participants may be unblinded
   and placebo subjects may be crossed over to the study vaccine.  The
   method allows for variant specification and adjustment for possible 
   confounding via inverse
   probability weighting through specification of models for the trial
   entry process, unblinding mechanisms, censoring process, and the 
   probability an unblinded placebo participant accepts study vaccine.
}
\details{
Analysis data must include the following information:
  \describe{
  \item{E}{The study entry time.}
  \item{A}{Binary treatment assignment, where A=0 indicates placebo; 
           A = 1 otherwise.}
  \item{U}{The minimum of: time to infection, time to refusing study 
     vaccine after unblinding (placebo participants only), or time to censoring 
     (loss to followup or administrative censoring due to analysis at time L).}
  \item{Delta}{Infection-variant category indicator, where: 
    -1 indicates a placebo participant was unblinded and refused study vaccine;
    0 indicates censoring; and
    = ve indicates infection of variant ve (ve = 1, ..., nV).}
  \item{R}{The minimum of: time to unblinding, time to infection, or
    time to censoring.}
  \item{Gamma}{Indicator of R dynamic, where 1 indicates that R is the
    time to unblinding and 0 indicates that R is either the time to infection
    or the time to censoring.}
  \item{Psi}{Indicator of whether a subject received study vaccine, where 1
    indicates that a participant was assigned to vaccine or was assigned to 
    placebo is unblinded and decides to get the study vaccine; 0 otherwise.}
}
Data can include baseline covariates. Methods for time-dependent covariates
 are not currently available.

There are 3 possible weighting options, the selection of which is determined
 by the combination of model inputs. 
  \describe{
    \item{No weighting: }{No models are provided as input, i.e., 
          inputs modelEntry, modelUnblind0, modelUnblind1, modelPsi, 
          modelCensor0, and modelCensor1 are not provided or are NULL.}
    \item{Weighting depends only on the censoring dynamic: }{Models
          modelCensor0 and modelCensor1 must be provided and models 
          modelEntry, modelUnblind0,  modelUnblind1, modelPsi must be NULL.}
    \item{Weighting depends on the entry, unblinding, and censoring 
          dynamics: }{All models must be provided.}
 }

The returned S3 object has 6 attributes needed for post-processing
 tools ve() and plot(). Specifically, "gFunc" is an integer object
 specifying the model selected for the infection rate (based on input gFunc);
 "v", the knots or cut-offs to be used by gFunc (input v);
 "maxTau", the maximum vaccination time included in the analysis; 
 "lag", the lag time between the initial vaccine dose and full efficacy;
 "phaseType", =1 only unblinded phase, =2, only blinded phase, =3 both phases,
 and "wgtType", =0 no weighting, =1 censor weighting, =2 full weighting.
}
\examples{
data(variantData)

set.seed(1234)

ind <- sample(1:nrow(variantData), 2000)
# NOTE: This sample size is chosen for example only -- larger data sets
# should be used.

# no weighting -- variant 1 infection only

res_noWgt <- veWaningVariant(data = variantData[ind,], 
                             L = 52.0,  
                             variant = 1L,
                             gFunc = 'piece', 
                             v = c(5.0,10.0))

# censoring only weighting -- variant 1 infection only

res_cens <- veWaningVariant(data = variantData[ind,], 
                            L = 52.0,  
                            variant = 1L,
                            modelCensor0 = ~ X1+X2, 
                            modelCensor1 = ~ X1+X2, 
                            gFunc = 'piece', 
                            v = c(5.0,10.0))

# full weighting -- variant 1 infection only

\dontrun{res_full <- veWaningVariant(data = variantData[ind,], 
                            L = 52.0,  
                            variant = 1L,
                            modelEntry = ~ X1,
                            modelUnblind0 = ~X1+X2,
                            modelUnblind1 = ~X2,
                            modelPsi = ~X1*X2,
                            modelCensor0 = ~ X1+X2, 
                            modelCensor1 = ~ X1+X2, 
                            gFunc = 'piece', 
                            v = c(5.0,10.0))}

}
