#' export
#'
#' Export andromeda generated by \link[TreatmentPatterns]{computePathways}
#' object to sharable csv-files and/or a zip archive.
#'
#' @export
#'
#' @template param_andromeda
#' @template param_outputPath
#' @template param_ageWindow
#' @template param_minFreq
#' @template param_archiveName
#'
#' @return (`invisible(NULL)`)
#'
#' @examples
#' \donttest{
#'   ableToRun <- invisible(all(
#'     require("Eunomia", character.only = TRUE),
#'     require("CirceR", character.only = TRUE),
#'     require("CohortGenerator", character.only = TRUE),
#'     require("dplyr", character.only = TRUE)
#'   ))
#'   
#'   if (ableToRun) {
#'     # CohortGenerator example
#'     connectionDetails <- Eunomia::getEunomiaConnectionDetails()
#'     cdmDatabaseSchema <- "main"
#'     resultSchema <- "main"
#'     cohortTable <- "CohortTable"
#' 
#'     cohortsToCreate <- CohortGenerator::createEmptyCohortDefinitionSet()
#'   
#'     cohortJsonFiles <- list.files(
#'       system.file(
#'         package = "TreatmentPatterns",
#'         "exampleCohorts"),
#'         full.names = TRUE)
#' 
#'     for (i in seq_len(length(cohortJsonFiles))) {
#'       cohortJsonFileName <- cohortJsonFiles[i]
#'       cohortName <- tools::file_path_sans_ext(basename(cohortJsonFileName))
#'       cohortJson <- readChar(cohortJsonFileName, file.info(
#'         cohortJsonFileName)$size)
#'
#'       cohortExpression <- CirceR::cohortExpressionFromJson(cohortJson)
#' 
#'       cohortSql <- CirceR::buildCohortQuery(
#'         cohortExpression,
#'         options = CirceR::createGenerateOptions(generateStats = FALSE))
#'     
#'       cohortsToCreate <- rbind(
#'         cohortsToCreate,
#'         data.frame(
#'           cohortId = i,
#'           cohortName = cohortName,
#'           sql = cohortSql,
#'           stringsAsFactors = FALSE))
#'     }
#'
#'     cohortTableNames <- CohortGenerator::getCohortTableNames(
#'       cohortTable = cohortTable)
#'
#'     CohortGenerator::createCohortTables(
#'       connectionDetails = connectionDetails,
#'       cohortDatabaseSchema = resultSchema,
#'       cohortTableNames = cohortTableNames)
#'
#'     # Generate the cohorts
#'     cohortsGenerated <- CohortGenerator::generateCohortSet(
#'       connectionDetails = connectionDetails,
#'       cdmDatabaseSchema = cdmDatabaseSchema,
#'       cohortDatabaseSchema = resultSchema,
#'       cohortTableNames = cohortTableNames,
#'       cohortDefinitionSet = cohortsToCreate)
#'     
#'     # Select Viral Sinusitis
#'     targetCohorts <- cohortsGenerated %>%
#'       filter(cohortName == "ViralSinusitis") %>%
#'       select(cohortId, cohortName)
#' 
#'     # Select everything BUT Viral Sinusitis cohorts
#'     eventCohorts <- cohortsGenerated %>%
#'       filter(cohortName != "ViralSinusitis" & cohortName != "Death") %>%
#'       select(cohortId, cohortName)
#' 
#'     exitCohorts <- cohortsGenerated %>%
#'       filter(cohortName == "Death") %>%
#'       select(cohortId, cohortName)
#' 
#'     cohorts <- dplyr::bind_rows(
#'       targetCohorts %>% mutate(type = "target"),
#'       eventCohorts %>% mutate(type = "event"),
#'       exitCohorts %>% mutate(type = "exit")
#'     )
#'
#'     andromeda <- computePathways(
#'       cohorts = cohorts,
#'       cohortTableName = cohortTable,
#'       connectionDetails = connectionDetails,
#'       cdmSchema = cdmDatabaseSchema,
#'       resultSchema = resultSchema
#'     )
#'     
#'     try(
#'       TreatmentPatterns::export(
#'         andromeda = andromeda,
#'         outputPath = tempdir(),
#'         ageWindow = 2,
#'         minFreq = 5,
#'         archiveName = "output.zip"
#'       )
#'     )
#'   }
#' }
export <- function(andromeda, outputPath, ageWindow = 10, minFreq = 5, archiveName = NULL) {
  collection <- checkmate::makeAssertCollection()
  checkmate::assertTRUE(Andromeda::isAndromeda(andromeda), add = collection)
  checkmate::assertPathForOutput(outputPath, overwrite = TRUE, add = collection)
  checkmate::assertIntegerish(ageWindow, min.len = 1, any.missing = FALSE, unique = TRUE, add = collection)
  checkmate::assertIntegerish(minFreq, len = 1, lower = 1, add = collection)
  checkmate::assertCharacter(archiveName, len = 1, add = collection, null.ok = TRUE)
  checkmate::reportAssertions(collection)
  
  if (!dir.exists(outputPath)) {
    dir.create(outputPath)
  }
  
  treatmentHistory <- andromeda$treatmentHistory %>%
    dplyr::collect()
  
  if (nrow(treatmentHistory) == 0) {
    message("Treatment History table is empty. Nothing to export.")
    return(invisible(NULL))
  }

  # metadata
  metadataPath <- file.path(outputPath, "metadata.csv")
  message(sprintf("Writing metadata to %s", metadataPath))
  metadata <- andromeda$metadata %>% dplyr::collect()
  write.csv(metadata, file = metadataPath, row.names = FALSE)

  # Treatment Pathways
  treatmentPathwaysPath <- file.path(outputPath, "treatmentPathways.csv")
  message(sprintf("Writing treatmentPathways to %s", treatmentPathwaysPath))
  treatmentPathways <- computeTreatmentPathways(treatmentHistory, ageWindow, minFreq)

  nTotal <- andromeda$currentCohorts %>%
    dplyr::summarise(n = dplyr::n_distinct(.data$personId)) %>%
    dplyr::pull()

  nTreated <- treatmentHistory %>%
    dplyr::summarise(n = dplyr::n_distinct(.data$personId)) %>%
    dplyr::pull()

  treatmentPathways <- treatmentPathways %>%
    dplyr::add_row(data.frame(
      path = "None",
      freq = nTotal - nTreated,
      sex = "all",
      age = "all",
      indexYear = "all"
    ))

  write.csv(treatmentPathways, file = treatmentPathwaysPath, row.names = FALSE)

  # Summary statistics duration
  statsTherapyPath <- file.path(outputPath, "summaryStatsTherapyDuraion.csv")
  message(sprintf("Writing summaryStatsTherapyDuraion to %s", statsTherapyPath))
  statsTherapy <- computeStatsTherapy(treatmentHistory)
  write.csv(statsTherapy, file = statsTherapyPath, row.names = FALSE)

  # Counts
  counts <- computeCounts(treatmentHistory, minFreq)

  countsYearPath <- file.path(outputPath, "countsYear.csv")
  message(sprintf("Writing countsYearPath to %s", countsYearPath))
  write.csv(counts$year, file = countsYearPath, row.names = FALSE)

  countsAgePath <- file.path(outputPath, "countsAge.csv")
  message(sprintf("Writing countsAgePath to %s", countsAgePath))
  write.csv(counts$age, file = countsAgePath, row.names = FALSE)

  countsSexPath <- file.path(outputPath, "countsSex.csv")
  message(sprintf("Writing countsSexPath to %s", countsSexPath))
  write.csv(counts$sex, file = countsSexPath, row.names = FALSE)

  if (!is.null(archiveName)) {
    zipPath <- file.path(outputPath, archiveName)

    message(sprintf("Zipping files to %s", zipPath))

    utils::zip(
      zipfile = zipPath,
      files = c(
        treatmentPathwaysPath, countsYearPath, countsAgePath, countsSexPath,
        statsTherapyPath
      )
    )
  }
  return(invisible(NULL))
}

#' computeStatsTherapy
#'
#' @template param_treatmentHistory
#'
#' @return (`data.frame()`)
computeStatsTherapy <- function(treatmentHistory) {
  stats <- treatmentHistory %>%
    mutate(treatmentType = dplyr::case_when(
      nchar(.data$eventCohortId) > 1 ~ "combination",
      .default = "monotherapy"
    )) %>%
    group_by(.data$treatmentType) %>%
    summarise(
      avgDuration = mean(.data$durationEra),
      medianDuration = stats::median(.data$durationEra),
      sd = stats::sd(.data$durationEra),
      min = min(.data$durationEra),
      max = max(.data$durationEra),
      count = n()
    )

  return(stats)
}

#' computeCounts
#'
#' @template param_treatmentHistory
#' @template param_minFreq
#'
#' @return (`list()`)
computeCounts <- function(treatmentHistory, minFreq) {
  # n per Year
  countYear <- treatmentHistory %>%
    dplyr::group_by(.data$indexYear) %>%
    dplyr::count() %>%
    dplyr::ungroup() %>%
    dplyr::mutate(n = case_when(
      .data$n < minFreq ~ sprintf("<%s", minFreq),
      .default = as.character(.data$n)
    ))

  # n per sex
  countSex <- treatmentHistory %>%
    dplyr::group_by(.data$sex) %>%
    dplyr::count() %>%
    dplyr::ungroup() %>%
    dplyr::mutate(n = case_when(
      .data$n < minFreq ~ sprintf("<%s", minFreq),
      .default = as.character(.data$n)
    ))

  # n per age
  countAge <- treatmentHistory %>%
    group_by(.data$age) %>%
    dplyr::count() %>%
    dplyr::ungroup() %>%
    dplyr::mutate(n = case_when(
      .data$n < minFreq ~ sprintf("<%s", minFreq),
      .default = as.character(.data$n)
    ))

  return(list(year = countYear, age = countAge, sex = countSex))
}


#' computeTreatmentPathways
#'
#' @template param_treatmentHistory
#' @template param_ageWindow
#' @template param_minFreq
#'
#' @return (`data.frame()`)
computeTreatmentPathways <- function(treatmentHistory, ageWindow, minFreq) {
  years <- c("all", treatmentHistory$indexYear %>% unique())

  if (length(ageWindow) > 1) {
    treatmentHistory <- treatmentHistory %>%
      rowwise() %>%
      dplyr::mutate(
        ageBin = paste(
          unlist(stringr::str_extract_all(as.character(cut(.data$age, ageWindow)), "\\d+")),
          collapse = "-"
        )
      )
  } else {
    treatmentHistory <- treatmentHistory %>%
      rowwise() %>%
      dplyr::mutate(
        ageBin = paste(
          unlist(stringr::str_extract_all(as.character(cut(.data$age, seq(0, 150, ageWindow))), "\\d+")),
          collapse = "-"
        )
      )
  }

  ages <- treatmentHistory$ageBin %>% unique()

  # Per year
  treatmentPathways <- stratisfy(treatmentHistory, years, ages)

  treatmentPathways <- treatmentPathways %>%
    mutate(indexYear = as.character(.data$indexYear))

  treatmentPathways[is.na(treatmentPathways)] <- "all"

  a <- nrow(treatmentPathways)

  treatmentPathways <- treatmentPathways %>%
    dplyr::filter(.data$freq >= minFreq)
  b <- nrow(treatmentPathways)

  message(sprintf("Removed %s pathways with a frequency < %s.", a - b, minFreq))
  return(treatmentPathways)
}

#' stratisfy
#'
#' @template param_treatmentHistory
#' @param years (`vector("character")`)
#' @param ages (`vector("character")`)
#'
#' @return (`data.frame()`)
stratisfy <- function(treatmentHistory, years, ages) {
  outDf <- dplyr::bind_rows(lapply(years, function(y) {
    all <- prepData(treatmentHistory = treatmentHistory, year = y)

    sex <- dplyr::bind_rows(
      treatmentHistory %>%
        filter(.data$sex == "MALE") %>%
        prepData(y) %>%
        mutate(sex = "male"),
      treatmentHistory %>%
        filter(.data$sex == "FEMALE") %>%
        prepData(y) %>%
        mutate(sex = "female")
    )

    age <- dplyr::bind_rows(lapply(ages, function(ageRange) {
      treatmentHistory %>%
        filter(.data$ageBin == ageRange) %>%
        prepData(y) %>%
        mutate(age = ageRange)
    }))

    return(dplyr::bind_rows(all, sex, age))
  }))
  return(outDf)
}
