#
# plot_final_year_time_series_data.R
#
#' Plots of inshore and offshore low trophic level state variables for the final year of a model run
#'
#' Multi-panel time series plots of inshore and offshore low trophic level state variables for the final year of a model run generated by the e2e_run() function
#'
#' The function plots a multi-panel page of time series plots of daily values of nutrients, detritus, macrophyte, phytoplankton, zooplankton and suspension/deposit feeding benthos (including larvae) for the inshore zone and upper and lower levels of the offshore zone, for the final year of a model run.
#'
#' Units of the plotted varaiables are mass concentrations, scaled to seas surface area or layer volume, ie. mMN/m2 or mMN/m3
#'
#' @param model R-list object defining the model configuration compiled by the e2e_read() function
#' @param results R-list object containing model results generated by the e2e_run() function
#' @param selection Text string from a list identifying the group of model output variables to be plotted. Select from: "NUT_PHYT", "SEDIMENT", "ZOOPLANKTON", "FISH", "BENTHOS", "PREDATORS", "CORP_DISC", "MACROPHYTE", Remember to include the phrase within "" quotes.
#'
#' @return Graphical display in a new graphics window
#'
#' @noRd
#
# ------------------------------------------------------------------------------

plot_final_year_time_series_data <- function(model, results, selection="NUT_PHYT") {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	build		<- elt(results, "build")
	run		<- elt(build, "run")
	nyears		<- elt(run, "nyears")
	ndays		<- elt(run, "ndays")

	data		<- elt(model, "data")
	physical.parameters	<- elt(data, "physical.parameters")
	si_depth	<- elt(physical.parameters, "si_depth")
	so_depth	<- elt(physical.parameters, "so_depth")
	d_depth		<- elt(physical.parameters, "d_depth")
	x_shallowprop	<- elt(physical.parameters, "x_shallowprop")

	x_poros_s1		<- elt(physical.parameters, "x_poros_s1")
	x_poros_s2		<- elt(physical.parameters, "x_poros_s2")
	x_poros_s3		<- elt(physical.parameters, "x_poros_s3")
	x_area_s1		<- elt(physical.parameters, "x_area_s1")
	x_area_s2		<- elt(physical.parameters, "x_area_s2")
	x_area_s3		<- elt(physical.parameters, "x_area_s3")
	x_depth_s1		<- elt(physical.parameters, "x_depth_s1")
	x_depth_s2		<- elt(physical.parameters, "x_depth_s2")
	x_depth_s3		<- elt(physical.parameters, "x_depth_s3")

	x_poros_d1		<- elt(physical.parameters, "x_poros_d1")
	x_poros_d2		<- elt(physical.parameters, "x_poros_d2")
	x_poros_d3		<- elt(physical.parameters, "x_poros_d3")
	x_area_d1		<- elt(physical.parameters, "x_area_d1")
	x_area_d2		<- elt(physical.parameters, "x_area_d2")
	x_area_d3		<- elt(physical.parameters, "x_area_d3")
	x_depth_d1		<- elt(physical.parameters, "x_depth_d1")
	x_depth_d2		<- elt(physical.parameters, "x_depth_d2")
	x_depth_d3		<- elt(physical.parameters, "x_depth_d3")


	output		<- elt(results, "output")
	corpse_d1	<- elt(output, "corpse_d1")
	corpse_d2	<- elt(output, "corpse_d2")
	corpse_d3	<- elt(output, "corpse_d3")
	corpse_s1	<- elt(output, "corpse_s1")
	corpse_s2	<- elt(output, "corpse_s2")
	corpse_s3	<- elt(output, "corpse_s3")
	kelpdebris	<- elt(output, "kelpdebris")
	detritus_so	<- elt(output, "detritus_so")
	detritus_si	<- elt(output, "detritus_si")
	detritus_d	<- elt(output, "detritus_d")
	nitrate_so	<- elt(output, "nitrate_so")
	nitrate_si	<- elt(output, "nitrate_si")
	nitrate_d	<- elt(output, "nitrate_d")
	ammonia_so	<- elt(output, "ammonia_so")
	ammonia_si	<- elt(output, "ammonia_si")
	ammonia_d	<- elt(output, "ammonia_d")
	phyt_so		<- elt(output, "phyt_so")
	phyt_si		<- elt(output, "phyt_si")
	phyt_d		<- elt(output, "phyt_d")
	omni_o		<- elt(output, "omni_o")
	omni_i		<- elt(output, "omni_i")
	carn_o		<- elt(output, "carn_o")
	carn_i		<- elt(output, "carn_i")
	benths_o	<- elt(output, "benths_o")
	benths_i	<- elt(output, "benths_i")
	kelpN		<- elt(output, "kelpN")
	benthslar_o	<- elt(output, "benthslar_o")
	benthslar_i	<- elt(output, "benthslar_i")

	benthc_o	<- elt(output, "benthc_o")
	benthc_i	<- elt(output, "benthc_i")
	benthclar_o	<- elt(output, "benthclar_o")
	benthclar_i	<- elt(output, "benthclar_i")
	fishp_o		<- elt(output, "fishp_o")
	fishp_i		<- elt(output, "fishp_i")
	fishplar_o	<- elt(output, "fishplar_o")
	fishplar_i	<- elt(output, "fishplar_i")
	fishd_o		<- elt(output, "fishd_o")
	fishd_i		<- elt(output, "fishd_i")
	fishdlar_o	<- elt(output, "fishdlar_o")
	fishdlar_i	<- elt(output, "fishdlar_i")
	fishm_o		<- elt(output, "fishm_o")
	fishm_i		<- elt(output, "fishm_i")
	bird_o		<- elt(output, "bird_o")
	bird_i		<- elt(output, "bird_i")
	ceta_o		<- elt(output, "ceta_o")
	ceta_i		<- elt(output, "ceta_i")
	seal_o		<- elt(output, "seal_o")
	seal_i		<- elt(output, "seal_i")
	discard_o	<- elt(output, "discard_o")
	discard_i	<- elt(output, "discard_i")

	x_ammonia_s1	<- elt(output, "x_ammonia_s1")
	x_ammonia_s2	<- elt(output, "x_ammonia_s2")
	x_ammonia_s3	<- elt(output, "x_ammonia_s3")
	x_ammonia_d1	<- elt(output, "x_ammonia_d1")
	x_ammonia_d2	<- elt(output, "x_ammonia_d2")
	x_ammonia_d3	<- elt(output, "x_ammonia_d3")
	x_nitrate_s1	<- elt(output, "x_nitrate_s1")
	x_nitrate_s2	<- elt(output, "x_nitrate_s2")
	x_nitrate_s3	<- elt(output, "x_nitrate_s3")
	x_nitrate_d1	<- elt(output, "x_nitrate_d1")
	x_nitrate_d2	<- elt(output, "x_nitrate_d2")
	x_nitrate_d3	<- elt(output, "x_nitrate_d3")
	x_detritus_s1	<- elt(output, "x_detritus_s1")
	xR_detritus_s1	<- elt(output, "xR_detritus_s1")
	x_detritus_s2	<- elt(output, "x_detritus_s2")
	xR_detritus_s2	<- elt(output, "xR_detritus_s2")
	x_detritus_s3	<- elt(output, "x_detritus_s3")
	xR_detritus_s3	<- elt(output, "xR_detritus_s3")
	x_detritus_d1	<- elt(output, "x_detritus_d1")
	xR_detritus_d1	<- elt(output, "xR_detritus_d1")
	x_detritus_d2	<- elt(output, "x_detritus_d2")
	xR_detritus_d2	<- elt(output, "xR_detritus_d2")
	x_detritus_d3	<- elt(output, "x_detritus_d3")
	xR_detritus_d3	<- elt(output, "xR_detritus_d3")


	aggregates	<- elt(results, "aggregates")
	x_poros		<- elt(aggregates, "x_poros")
	x_depth		<- elt(aggregates, "x_depth")

	xvolume_si<-si_depth*x_shallowprop
	xvolume_so<-so_depth*(1-x_shallowprop)
	xd_volume<-d_depth*(1-x_shallowprop)

# ...................................................................


if(selection=="NUT_PHYT"){

	#Plots the final year of output for water column detritus, nutrients and phytoplankton

#	par(mfrow=c(3,4))
	par(mfrow=c(2,2))

	l1<-detritus_so[((nyears-1)*360+1):ndays]/xvolume_so
	l2<-detritus_si[((nyears-1)*360+1):ndays]/xvolume_si
	l3<-detritus_d[((nyears-1)*360+1):ndays]/xd_volume
	fyplot3("Suspended bact. & detritus","","S-offshore","S-inshore","Deep",l1,l2,l3)
	mtext(bquote("Concentration mMN.m"^-3),cex=0.7,side=2,line=2.5)


	l1<-ammonia_so[((nyears-1)*360+1):ndays]/xvolume_so
	l2<-ammonia_si[((nyears-1)*360+1):ndays]/xvolume_si
	l3<-ammonia_d[((nyears-1)*360+1):ndays]/xd_volume
	fyplot3("Ammonia","","S-offshore","S-inshore","Deep",l1,l2,l3)
	mtext(bquote("Concentration mMN.m"^-3),cex=0.7,side=2,line=2.5)

	l1<-nitrate_so[((nyears-1)*360+1):ndays]/xvolume_so
	l2<-nitrate_si[((nyears-1)*360+1):ndays]/xvolume_si
	l3<-nitrate_d[((nyears-1)*360+1):ndays]/xd_volume
	fyplot3("Nitrate","","S-offshore","S-inshore","Deep",l1,l2,l3)
	mtext(bquote("Concentration mMN.m"^-3),cex=0.7,side=2,line=2.5)

	l1<-phyt_so[((nyears-1)*360+1):ndays]/xvolume_so
	l2<-phyt_si[((nyears-1)*360+1):ndays]/xvolume_si
	l3<-phyt_d[((nyears-1)*360+1):ndays]/xd_volume
	fyplot3("Phytoplankton","","S-offshore","S-inshore","Deep",l1,l2,l3)
	mtext(bquote("Concentration mMN.m"^-3),cex=0.7,side=2,line=2.5)


#...........................................................

} else if(selection=="SEDIMENT"){


	#Plot the final year of output for the sediment habitats separately

	#Plots are:
	#    Shallow layer - wc ammonia, hab1 pw ammonia, hab2 pw ammonia, hab3 pw ammonia
	#    Deep layer    - wc ammonia, hab1 pw ammonia, hab2 pw ammonia, hab3 pw ammonia

	#    Shallow layer - wc nitrate, hab1 pw nitrate, hab2 pw nitrate, hab3 pw nitrate
	#    Deep layer    - wc nitrate, hab1 pw nitrate, hab2 pw nitrate, hab3 pw nitrate

	#    Shallow layer - wc detritus, hab1 detritus, hab2 detritus, hab3 pw detritus
	#    Deep layer    - wc detritus, hab1 detritus, hab2 detritus, hab3 pw detritus

	par(mfrow=c(4,2))

	#Shallow sediment ammonia

	#Converts the sediment ammonia into units of N /m3 in the pore water)

	if(x_poros_s1>0 && x_area_s1>0){
		l1<-x_ammonia_s1[((nyears-1)*360+1):ndays]/(x_area_s1*x_depth_s1*x_poros_s1)
	} else {
		l1<-rep(NA,361)
	}

	if(x_poros_s2>0 && x_area_s2>0){
		l2<-x_ammonia_s2[((nyears-1)*360+1):ndays]/(x_area_s2*x_depth_s2*x_poros_s2)
	} else {
		l2<-rep(NA,361)
	}

	if(x_poros_s3>0 && x_area_s3>0){
		l3<-x_ammonia_s3[((nyears-1)*360+1):ndays]/(x_area_s3*x_depth_s3*x_poros_s3)
	} else {
		l3<-rep(NA,361)
	}

	fyplot3_hab("Inshore ammonia","","Area_s1 porewater","Area_s2 porewater","Area_s3 porewater",l1,l2,l3)
	mtext(bquote("Concentration mMN.m"^-3),cex=0.7,side=2,line=2.5)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	#Deep sediment ammonia

	#Converts the sediment ammonia into units of N /m3 in the pore water)

	if(x_poros_d1>0 && x_area_d1>0){
		l1<-x_ammonia_d1[((nyears-1)*360+1):ndays]/(x_area_d1*x_depth_d1*x_poros_d1)
	} else {
		l1<-rep(NA,361)
	}

	if(x_poros_d2>0 && x_area_d2>0){
		l2<-x_ammonia_d2[((nyears-1)*360+1):ndays]/(x_area_d2*x_depth_d2*x_poros_d2)
	} else {
		l2<-rep(NA,361)
	}

	if(x_poros_d3>0 && x_area_d3>0){
		l3<-x_ammonia_d3[((nyears-1)*360+1):ndays]/(x_area_d3*x_depth_d3*x_poros_d3)
	} else {
		l3<-rep(NA,361)
	}

	fyplot3_hab("Offshore ammonia","","Area_d1 porewater","Area_d2 porewater","Area_d3 porewater",l1,l2,l3)
	mtext(bquote("Concentration mMN.m"^-3),cex=0.7,side=2,line=2.5)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	#Shallow sediment nitrate

	#Converts the sediment nitrate into units of N /m3 in the pore water)

	if(x_poros_s1>0 && x_area_s1>0){
	l1<-x_nitrate_s1[((nyears-1)*360+1):ndays]/(x_area_s1*x_depth_s1*x_poros_s1)
	} else {
	l1<-rep(NA,361)
	}

	if(x_poros_s2>0 && x_area_s2>0){
	l2<-x_nitrate_s2[((nyears-1)*360+1):ndays]/(x_area_s2*x_depth_s2*x_poros_s2)
	} else {
	l3<-rep(NA,361)
	}

	if(x_poros_s3>0 && x_area_s3>0){
	l3<-x_nitrate_s3[((nyears-1)*360+1):ndays]/(x_area_s3*x_depth_s3*x_poros_s3)
	} else {
	l3<-rep(NA,361)
	}

	fyplot3_hab("Inshore nitrate","","Area_s1 porewater","Area_s2 porewater","Area_s3 porewater",l1,l2,l3)
	mtext(bquote("Concentration mMN.m"^-3),cex=0.7,side=2,line=2.5)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	#Deep sediment nitrate

	#Converts the sediment nitrate into units of N /m3 in the pore water)

	if(x_poros_d1>0 && x_area_d1>0){
	l1<-x_nitrate_d1[((nyears-1)*360+1):ndays]/(x_area_d1*x_depth_d1*x_poros_d1)
	} else {
	l1<-rep(NA,361)
	}

	if(x_poros_d2>0 && x_area_d2>0){
	l2<-x_nitrate_d2[((nyears-1)*360+1):ndays]/(x_area_d2*x_depth_d2*x_poros_d2)
	} else {
	l2<-rep(NA,361)
	}

	if(x_poros_d3>0 && x_area_d3>0){
	l3<-x_nitrate_d3[((nyears-1)*360+1):ndays]/(x_area_d3*x_depth_d3*x_poros_d3)
	} else {
	l3<-rep(NA,361)
	}

	fyplot3_hab("Offshore nitrate","","Area_d1 porewater","Area_d2 porewater","Area_d3 porewater",l1,l2,l3)
	mtext(bquote("Concentration mMN.m"^-3),cex=0.7,side=2,line=2.5)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	#Shallow sediment total detritus

	#Converts the sediment detritus into units of %N by dry wt (100*gN/g-drysediment) (density of dry solid matter = 2.65g/cm3)

	if(x_poros_s1>0 && x_area_s1>0){
	l1a<-100*(((x_detritus_s1[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s1*x_depth_s1*(((1-x_poros_s1)*(2650*1000))))
	l1b<-100*(((xR_detritus_s1[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s1*x_depth_s1*(((1-x_poros_s1)*(2650*1000))))
	} else {
	l1a<-rep(NA,361)
	l1b<-rep(NA,361)
	}

	if(x_poros_s2>0 && x_area_s2>0){
	l2a<-100*(((x_detritus_s2[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s2*x_depth_s2*(((1-x_poros_s2)*(2650*1000))))
	l2b<-100*(((xR_detritus_s2[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s2*x_depth_s2*(((1-x_poros_s2)*(2650*1000))))
	} else {
	l2a<-rep(NA,361)
	l2b<-rep(NA,361)
	}

	if(x_poros_s3>0 && x_area_s3>0){
	l3a<-100*(((x_detritus_s3[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s3*x_depth_s3*(((1-x_poros_s3)*(2650*1000))))
	l3b<-100*(((xR_detritus_s3[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s3*x_depth_s3*(((1-x_poros_s3)*(2650*1000))))
	} else {
	l3a<-rep(NA,361)
	l3b<-rep(NA,361)
	}
 
	l1<-l1a+l1b
	l2<-l2a+l2b
	l3<-l3a+l3b
	fyplot3_hab("Inshore detritus","","Area_s1 sediment","Area_s2 sediment","Area_s3 sediment",l1,l2,l3)
	mtext(bquote("Nitrogen.DW"^-1 ~ " g.g"^-1),cex=0.7,side=2,line=2.5)


	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	#Deep sediment total detritus

	#Converts the sediment detritus into units of %N by dry wt (100*gN/g-drysediment) (density of dry solid matter = 2.65g/cm3)

	if(x_poros_d1>0 && x_area_d1>0){
	l1a<-100*(((x_detritus_d1[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d1*x_depth_d1*(((1-x_poros_d1)*(2650*1000))))
	l1b<-100*(((xR_detritus_d1[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d1*x_depth_d1*(((1-x_poros_d1)*(2650*1000))))
	} else {
	l1a<-rep(NA,361)
	l1b<-rep(NA,361)
	}

	if(x_poros_d2>0 && x_area_d2>0){
	l2a<-100*(((x_detritus_d2[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d2*x_depth_d2*(((1-x_poros_d2)*(2650*1000))))
	l2b<-100*(((xR_detritus_d2[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d2*x_depth_d2*(((1-x_poros_d2)*(2650*1000))))
	} else {
	l2a<-rep(NA,361)
	l2b<-rep(NA,361)
	}

	if(x_poros_d3>0 && x_area_d3>0){
	l3a<-100*(((x_detritus_d3[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d3*x_depth_d3*(((1-x_poros_d3)*(2650*1000))))
	l3b<-100*(((xR_detritus_d3[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d3*x_depth_d3*(((1-x_poros_d3)*(2650*1000))))
	} else {
	l3a<-rep(NA,361)
	l3b<-rep(NA,361)
	}

	l1<-l1a+l1b
	l2<-l2a+l2b
	l3<-l3a+l3b

	fyplot3_hab("Offshore detritus","","Area_d1 sediment","Area_d2 sediment","Area_d3 sediment",l1,l2,l3)
	mtext(bquote("Nitrogen.DW"^-1 ~ " g.g"^-1),cex=0.7,side=2,line=2.5)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	#Shallow sediment corpses

	#Converts the sediment corpse mass into units of N /m2 of sediment surface)

	if(x_area_s1>0){
	l1<-corpse_s1[((nyears-1)*360+1):ndays]/(x_area_s1)
	} else {
	l1<-rep(NA,361)
	}

	if(x_area_s2>0){
	l2<-corpse_s2[((nyears-1)*360+1):ndays]/(x_area_s2)
	} else {
	l2<-rep(NA,361)
	}

	if(x_area_s3>0){
	l3<-corpse_s3[((nyears-1)*360+1):ndays]/(x_area_s3)
	} else {
	l3<-rep(NA,361)
	}

	fyplot3_hab("Inshore corpses","","Area_s1","Area_s2","Area_s3",l1,l2,l3)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	#Deep sediment corpses

	#Converts the sediment corpse mass into units of N /m2 of sediment surface)

	if(x_area_d1>0){
	l1<-corpse_d1[((nyears-1)*360+1):ndays]/(x_area_d1)
	} else {
	l1<-rep(NA,361)
	}

	if(x_area_d2>0){
	l2<-corpse_d2[((nyears-1)*360+1):ndays]/(x_area_d2)
	} else {
	l2<-rep(NA,361)
	}

	if(x_area_d3>0){
	l3<-corpse_d3[((nyears-1)*360+1):ndays]/(x_area_d3)
	} else {
	l3<-rep(NA,361)
	}

	fyplot3_hab("Offshore corpses","","Area_d1","Area_d2","Area_d3",l1,l2,l3)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)



#...........................................................

} else if(selection=="ZOOPLANKTON"){

	par(mfrow=c(2,2))

	l1<-omni_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-omni_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Omnivorous zooplankton","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-carn_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-carn_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Carnivorous zooplankton","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)




#...........................................................

} else if(selection=="FISH"){


#	par(mfrow=c(4,2))
	par(mfrow=c(2,2))

	l1<-fishp_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishp_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Planktivorous fish","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-fishplar_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishplar_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Planktivorous fish larvae","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-fishd_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishd_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Demersal fish","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-fishdlar_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishdlar_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Demersal fish larvae","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)


#...........................................................

} else if(selection=="BENTHOS"){

#	par(mfrow=c(4,2))
	par(mfrow=c(2,2))

	l1<-benths_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-benths_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Benthos susp/dep feeders","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-benthslar_o[((nyears-1)*360+1):ndays]/(xvolume_so+xd_volume)
	l2<-benthslar_i[((nyears-1)*360+1):ndays]/xvolume_si
	fyplot2("Benthos susp/dep feeders larvae","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-benthc_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-benthc_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Benthos carn/scav feeders","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-benthclar_o[((nyears-1)*360+1):ndays]/(xvolume_so+xd_volume)
	l2<-benthclar_i[((nyears-1)*360+1):ndays]/xvolume_si
	fyplot2("Benthos carn/scav feeders larvae","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)


#...........................................................

} else if(selection=="PREDATORS"){

#	par(mfrow=c(4,2))
	par(mfrow=c(2,2))

	l1<-bird_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-bird_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Birds","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-seal_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-seal_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Pinnipeds","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-ceta_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-ceta_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Cetaceans","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-fishm_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishm_i[((nyears-1)*360+1):ndays]/x_shallowprop
	fyplot2("Migratory fish","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)


#...........................................................

} else if(selection=="CORP_DISC"){

	par(mfrow=c(2,2))

	l1<-(corpse_d1[((nyears-1)*360+1):ndays]+corpse_d2[((nyears-1)*360+1):ndays]+corpse_d3[((nyears-1)*360+1):ndays])/(1-x_shallowprop)
	l2<-(corpse_s1[((nyears-1)*360+1):ndays]+corpse_s2[((nyears-1)*360+1):ndays]+corpse_s3[((nyears-1)*360+1):ndays])/x_shallowprop
	fyplot2("Corpses","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-(discard_o[((nyears-1)*360+1):ndays])/(1-x_shallowprop)
	l2<-(discard_i[((nyears-1)*360+1):ndays])/x_shallowprop
	fyplot2("Discards","","Offshore","Inshore",l1,l2)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

#...........................................................

} else if(selection=="MACROPHYTE"){

	par(mfrow=c(2,2))

	l1<-kelpN[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	fyplot1("Inshore macrophytes","",l1)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)

	l1<-(kelpdebris[((nyears-1)*360+1):ndays])/x_shallowprop
	fyplot1("Inshore macrophyte debris","",l1)
	mtext(bquote("Area density mMN.m"^-2),cex=0.7,side=2,line=2.5)


#...........................................................

} else {
	stop("Error: unknown selection '", selection, "' !\n")
}

#...........................................................



}

