% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regtab.R
\name{regtab}
\alias{regtab}
\title{Multi-Outcome Regression Table}
\usage{
regtab(
  data,
  outcomes,
  predictors,
  family = poisson(link = "log"),
  robust = TRUE,
  exponentiate = NULL,
  labels = NULL,
  d = 2,
  conf.level = 0.95,
  include_intercept = FALSE,
  p_values = FALSE
)
}
\arguments{
\item{data}{Data.frame containing all variables for analysis.}

\item{outcomes}{Character vector of dependent variable names. Each outcome is
modeled separately with the same set of predictors.}

\item{predictors}{Formula or character string specifying predictors. Can be:
\itemize{
\item Formula: \code{~ x1 + x2 + x3}
\item Character: \code{"~ x1 + x2 + x3"} or \code{"x1 + x2 + x3"}
}}

\item{family}{GLM family specification. Options:
\itemize{
\item \code{poisson(link = "log")} - For count outcomes (default)
\item \code{binomial(link = "logit")} - For binary outcomes
\item \code{gaussian(link = "identity")} - For continuous outcomes
\item \code{quasipoisson()}, \code{quasibinomial()} - For overdispersed data
\item Or character: "poisson", "binomial", "gaussian"
}}

\item{robust}{Logical. If TRUE (default), calculates heteroskedasticity-consistent
(HC0) robust standard errors via the sandwich package. CIs are based on robust SEs.}

\item{exponentiate}{Logical. If TRUE, exponentiates coefficients and CIs:
\itemize{
\item Poisson: IRR (Incidence Rate Ratios)
\item Binomial: OR (Odds Ratios)
\item Gaussian: Not typically used (stays on linear scale)
}
If NULL (default), automatically detects: TRUE for Poisson/Binomial,
FALSE for Gaussian.}

\item{labels}{Named character vector for renaming outcome columns in output.
Format: \code{c("raw_name" = "Pretty Label")}. Useful for publication tables.}

\item{d}{Integer. Number of decimal places for rounding estimates and CIs. Default: 2.}

\item{conf.level}{Numeric. Confidence level for intervals (0-1). Default: 0.95.}

\item{include_intercept}{Logical. If TRUE, includes intercept in output table.
Default: FALSE (typically excluded from publication tables).}

\item{p_values}{Logical. If TRUE, adds p-values as separate column. Default: FALSE.}
}
\value{
A data.frame in wide format with:
\itemize{
\item \strong{Variable}: Predictor names (first column)
\item \strong{Outcome columns}: One column per outcome with formatted estimates and CIs
}

Can be directly exported to Excel, Word, or LaTeX for publication.
}
\description{
Fits generalized linear models (GLMs) for multiple outcome variables and generates
a formatted wide-format table with point estimates and confidence intervals.
Supports robust standard errors, automatic exponentiation for count/binary outcomes,
and custom labeling for publication-ready tables.
}
\details{
\subsection{Model Fitting}{

For each outcome, the function fits:
\code{glm(outcome ~ predictors, family = family, data = data)}
}

\subsection{Robust Standard Errors}{

When \code{robust = TRUE}, the function:
\enumerate{
\item Fits the model with standard GLM.
\item Computes sandwich covariance matrix (HC0 estimator).
\item Calculates Wald-type CIs based on robust SEs.
}

This provides protection against heteroskedasticity and mild model misspecification.
}

\subsection{Exponentiation}{
\itemize{
\item \strong{Poisson regression}: exp(beta) = Incidence Rate Ratio
\itemize{
\item IRR = 1: No association
\item IRR > 1: Increased rate
\item IRR < 1: Decreased rate
}
\item \strong{Logistic regression}: exp(beta) = Odds Ratio
\itemize{
\item OR = 1: No association
\item OR > 1: Increased odds
\item OR < 1: Decreased odds
}
}
}

\subsection{Output Format}{

Returns a wide-format data.frame:
\preformatted{
Variable    | Outcome1          | Outcome2          | ...
------------|-------------------|-------------------|----
(Intercept) | 2.34 (1.89-2.91) | 1.98 (1.65-2.38) | ...
age         | 1.05 (1.02-1.08) | 1.03 (1.01-1.06) | ...
sex         | 0.87 (0.75-1.01) | 0.92 (0.81-1.05) | ...
}
Each cell contains: "Estimate (Lower CI - Upper CI)"
}

\subsection{Missing Data}{

GLM uses complete cases by default. Observations with missing values in any
variable are excluded from that specific model.
}

\subsection{Convergence Issues}{

If a model fails to converge or encounters errors:
\itemize{
\item A warning is issued with the outcome name and error message
\item That outcome column is skipped in the output
\item Other outcomes continue processing
}
}
}
\examples{
# Create example data
set.seed(456)
n <- 500
df <- data.frame(
  age = rnorm(n, 50, 10),
  sex = factor(sample(c("M", "F"), n, replace = TRUE)),
  treatment = factor(sample(c("A", "B"), n, replace = TRUE)),
  outcome1 = rpois(n, lambda = 5),
  outcome2 = rpois(n, lambda = 8),
  outcome3 = rpois(n, lambda = 3)
)

# Basic usage: Poisson regression for multiple outcomes
regtab(df,
       outcomes = c("outcome1", "outcome2", "outcome3"),
       predictors = ~ age + sex + treatment,
       family = poisson(link = "log"))

# With custom labels and no robust SEs
regtab(df,
       outcomes = c("outcome1", "outcome2"),
       predictors = "age + sex",
       labels = c(outcome1 = "Primary Endpoint", outcome2 = "Secondary Endpoint"),
       robust = FALSE)

# Logistic regression with p-values
df$binary_outcome <- rbinom(n, 1, 0.4)
regtab(df,
       outcomes = "binary_outcome",
       predictors = ~ age + sex,
       family = binomial(),
       p_values = TRUE)

}
